package simulation.geometry;

import java.io.IOException;
import java.io.Writer;

import simulation.DiffusionSimulation;
import tools.CL_Initializer;

import numerics.Point3D;
import numerics.RealMatrix;
import numerics.Rotations;
import numerics.Vector3D;

/** implements a cylindrical geometry. Cylinder is defined 
 *  like a ray: a vector V defines its axis, another P its position
 *  in space and a scalar radius r provides its thickness.
 *  
 *  This class also provides methods for getting intersections
 *  and normals for reflecting steps.
 * 
 * @author matt
 *
 */
public class BasicCylinder implements Cylinder{

	/** numerical tolerence */
	private static final double TINYNUM=CellularLattice.TINYNUM;
	
	/** position vector */
	public final Point3D P;
	
	/** position vector in rotated cylinder coords */
	private final Point3D Prot;
	
	/** pos as point */
	public final double[] Parr;
	
	/** orientation vector */
	private final Vector3D V;
	
	/** radius */
	private final double r;

	/** rotation matrix that rotates the cylinders 
	 * orientation onto the coordinate z-axis
	 */
	private final RealMatrix rotMat;
	
	/** rotation matrix that inverts the above rotation */
	private final RealMatrix invRotMat;
	
	/** permeability of membrane -- moved here from substrate */
	private final double p;
	
	/** 
	 * diffusivity inside.
	 * 
	 * TODO: tie into constructor
	 */
	private final double d=CL_Initializer.DIFF_CONST;
	
	/** dimensionality of space */
	private final int D=DiffusionSimulation.D;

	/** index of initial positon of walker before reflection loop */
	private double[] initialPos= new double[D];
	
	/** records whether the initial walker position is inside or outside the cylinder */
	private boolean initiallyIn;

	/** number of steps when outputting cross section to file */
	private int N=20;
	
	/** bounding box for cylinder */
	private final BoundingBox bbox;

	/** 
	 * constructor without specifying direction. constructs a 
	 * cylinder parallel to the z-axis at the position specified
	 * 
	 * @param P position of cylinder
	 * @param r radius of cylinder
	 * @param p permeability of cylinder
	 */
	public BasicCylinder(double[] P,  double r, double p){
		
		// set cylinder position and point... 
		this.P= new Point3D(P);

		// ... and as array...
		this.Parr= new double[D];
		for(int i=0; i<D; i++){
			Parr[i]=P[i];
		}
		
		// here no rotation of position is neccessary
		Prot=this.P;
		
		// set radius
		this.r=r;
		
		// set permeability
		this.p=p;

		// set default orientation parallel to z-axis
		double[] Varr= new double[]{0.0, 0.0, 1.0};
		
		// set orientation
		this.V= new Vector3D(Varr);

		
		
		// assemble bounding box
		double[][] v= new double[][]{{1.0, 0.0, 0.0},
									 {0.0, 1.0, 0.0},
									 {0.0, 0.0, 1.0}};
		double[] vmin= new double[]{P[0]-r, P[1]-r, 0.0};
		double[] vmax= new double[]{P[0]+r, P[1]+r, Double.MAX_VALUE}; 
		
		
		// construct bounding box
		this.bbox = new BoundingBox(v, vmin, vmax);
				
		// rotation matrices not used in this case
		rotMat=null;
		invRotMat=null;
		
	}

	
	
	/** constructor that specifies orientation.
	 * 
	 * @param P position of cylinder
	 * @param V orientation of cylinder axis
	 * @param r radius of cylinder
	 * @param p permeability of cylinder
	 */
	protected BasicCylinder(double[] P, double[] V, double r, double p){
		
		// set cylinder position and point... 
		this.P= new Point3D(P);

		// ... and as array...
		this.Parr= new double[D];
		for(int i=0; i<P.length; i++){
			Parr[i]=P[i];
		}
		
		
		// set radius
		this.r=r;
		
		// set permeability
		this.p=p;
		
		// normalise V vector
		double modV=0.0;
		for(int i=0; i<D; i++){
			modV+=V[i]*V[i];
		}
		
		modV=Math.sqrt(modV);
		
		for(int i=0; i<D; i++){
			V[i]=V[i]/modV;
		}
		
		// set orientation
		this.V= new Vector3D(V);

		double[][] v= new double[][]{{1,0,0},{0,1,0}};
		double[] vmin= new double[]{P[0]-r, P[1]-r};
		double[] vmax= new double[]{P[0]+r, P[1]+r}; 
		
		
		// construct bounding box
		//this.bbox = new BoundingBox(v, vmin, vmax);
		this.bbox=null;
		
		// angle between z-axis and cylinder axis
		double theta= Math.acos(V[2]);
		
		
		// set rotations for cylinders that are not parallel to z-axis
		Vector3D zAxis= new Vector3D(0.0, 0.0, 1.0);
		Vector3D axel= this.V.cross(zAxis);
		
		rotMat=Rotations.getRotMat(axel, -theta);
		invRotMat=Rotations.getRotMat(axel, theta);
		
		// finally, set the rotated cylinder position
		// ... and rotated into cylinder coords
		Prot=new Point3D(Rotations.transformPoint(rotMat, Parr));
		//ProtArr= new double[]{Prot.x, Prot.y, Prot.z};
	}
	
	public boolean crosses(double[] rawPos, double[] rawStep,
			double[] normal, double[] d, boolean skipCurrent, double origLength, 
			double[] intDist, boolean[] in, double[] p){
		
		// position and radius of the cylinder
		double[] P= getPosition();
		double[] Praw= P;
 
		double r= getRadius();
		
		double[] walkerPos=rawPos;
		double[] step=rawStep;
		
		if(rotMat!=null){
			// if cylinder is not parallel to z-axis,
			// rotate cylinder & walker position and 
			// step into cylinder frame
			
			walkerPos=Rotations.transformPoint(rotMat, rawPos);
			step=Rotations.transformPoint(rotMat, step);
			P=Rotations.transformPoint(rotMat, P);
		}
		
		
		// start and end points of current step projected into plane in axis coords
		double[] w1= new double[]{walkerPos[0]-P[0], walkerPos[1]-P[1], 0.0};
		double[] w2= new double[]{walkerPos[0]-P[0]+step[0], 
								  walkerPos[1]-P[1]+step[1], 
								  0.0};
		
		// start and end points in substrate coords
		double[] ws1= new double[]{rawPos[0]-Praw[0], rawPos[1]-Praw[1], rawPos[2]-Praw[2]};
		double[] ws2= new double[]{rawPos[0]-Praw[0]+rawStep[0], 
				  				   rawPos[1]-Praw[1]+rawStep[1], 
				  				   rawPos[2]-Praw[2]+rawStep[2]};

		
		double[] newPos= new double[D];
		
		// distance of start and end points from cylinder axis
		double d1=Math.sqrt(w1[0]*w1[0] + w1[1]*w1[1]);
		double d2=Math.sqrt(w2[0]*w2[0] + w2[1]*w2[1]);
		
		double ds1=Math.sqrt(ws1[0]*ws1[0] + ws1[1]*ws1[1]+ ws1[2]*ws1[2]);
		double ds2=Math.sqrt(ws2[0]*ws2[0] + ws2[1]*ws2[1]+ ws2[2]*ws2[2]);
		
		
		
		// if skipCurrent is false, this is an inital call
		// before the reflection loop, so we need to update 
		// the initial position of walker and the initially
		// in flag for future skip checks
		if(!skipCurrent){
			//initiallyIn=inside(ws1);
			initiallyIn=inside(rawPos);
			in[0]=initiallyIn;
		}

		
		
		// if start and end points and both inside the cylinder, 
		// we're definitely not crossing the circumference
		if((d1<=r)&&(d2<=r)){
			// we're indside the circle, just check the chords and go

			return false;
		}
		
		
		// otherwise there potentially is an intersection
		// get the intersection parameters
		//double[] root = get2dIntersectionRoots(walkerPos, step);
		double[] root=getIntersectionRoots(walkerPos, step);
		
		// no real roots = no intersections
		if(root==null){
			return false;
		}

		// both roots less than zero = no intersections
		if((root[0]<0.0) && (root[1]<0.0)){
			return false;
		}

		// if first root is out of range but second is in, swap them
		if(!((root[0]>=0.0)&&(root[0]<=1.0))){
			if((root[1]>=0.0)&&(root[1]<=1.0)){
				double temp=root[0];
				
				root[0]=root[1];
				root[1]=temp;				
			}
		}
		
		// if both roots in range, make sure lowest one
		// is first so that correct intersection is used
		if((root[0]>=0.0)&&(root[0]<=1.0)){
			if((root[1]>=0.0)&&(root[1]<=1.0)){
				if(root[0]>root[1]){
					double temp=root[0];
					
					root[0]=root[1];
					root[1]=temp;
				}
			}
		}
				

		// calculate step length
		double stepLen=0.0;
		for(int i=0; i<2; i++){
			stepLen+=step[i]*step[i];
		}
		stepLen=Math.sqrt(stepLen);
		
		for(int i=0; i<D; i++){
			newPos[i]=rawPos[i]+rawStep[i];
		}
				

		// if we're not in a chord interval, check intersection
		// with circumference.
		for(int i=0; i<root.length; i++){
			if((root[i]>0.0)&&(root[i]<=1.0)){
				// set normal and distance from origin in normal direction
				double[] intPoint= new double[]{walkerPos[0]+root[i]*step[0]-P[0], 
												walkerPos[1]+root[i]*step[1]-P[1], 
												walkerPos[2]+root[i]*step[2]-P[2]}; 
				
				double theta=Math.atan2(intPoint[1], intPoint[0]);

				double newNormal[] = new double[normal.length];
				
				double newD;
				
				// set normal using radial unit vector
				newNormal[0]=Math.cos(theta);
				newNormal[1]=Math.sin(theta);
				newNormal[2]=0.0;
				
				newD=0.0;
				for(int j=0; j<2; j++){
					newD+=(intPoint[j]+P[j])*newNormal[j];
				}
				
				
				// check if we need to skip this intersection
				if(skipCurrent){
	
					boolean skipIt=checkSkipping(newPos);
					
					// if we're skipping, jump to the next root in the i loop
					if(skipIt){
						continue;
					}
				}
			
				// if we've got here then we need to return 
				// the distance and normal to the amending
				// routines
				d[0]=newD;
				intDist[0]=0.0;
				
				// in case of arbitrary orientation, need to
				// back-rotate the normal into the voxel frame
				// before returning it
				if(invRotMat!=null){
					double[] tempNormal= Rotations.transformPoint(invRotMat, newNormal);
					newNormal=tempNormal;
				}
				
				for(int j=0; j<normal.length; j++){
					normal[j]=newNormal[j];
					//intDist[0]+=(root[i]*step[i])*(root[i]*step[i]);
				}
				intDist[0]=root[i];
				
				
				// if skipCurrent is false, this is an inital call
				// before the reflection loop, so we need to update 
				// the initial position of walker and the initially
				// in flag for future skip checks
				if(!skipCurrent){
					for(int j=0; j<D; j++){
						initialPos[j]=w1[j];
					}
					//if(inside(walkerPos)){
					if(inside(rawPos)){
						initiallyIn=true;
					}
					else{
						initiallyIn=false;
					}
				}
				p[0]=this.p;
				return true;
			}
		}
		
		return false;
	}

	
	/**
	 * checks if the current intersection should be ignored.
	 * The safest way to do this is to use the fact that the
	 * cylinder is circular and to check the initially inside
	 * flag, which is true if the walker starts inside the 
	 * cylinder, and compare it to whether the end of the step 
	 * is inside or outside.
	 * 
	 * Given that skipping is always true when this method is
	 * called, and the fact that the cylinder is circular it means
	 * that the current intersection should be ignored if the
	 * flags are equal (start and finish inside or start and 
	 * finish outside) because in that case we're sitting on the
	 * circumference when this method is called.
	 * 
	 * simple routine, very complex reasoning!
	 * 
	 *  @param newPos the new position after the step is made
	 */
	private final boolean checkSkipping(double[] newPos){
		
		/*double[] pos;
		
		if(invRotMat==null){
			pos=newPos;
		}
		else{
			// if necessary, back-rotate the position into substrate coords
			pos=Rotations.transformPoint(invRotMat, newPos);
		}*/
		
		if(initiallyIn==inside(newPos)){
			return true;
		}
		else{
			return false;
		}
	}

	
	
	/** calculates the distance d of a given point from the central
	 * axis of the cylinder using the formula
	 * 
	 * d = sqrt( (Q-P)^2 - [(Q-P).V]^2)
	 * 
	 * @param Q position vector or point
	 * @return the shortest distance from the point Q to the central 
	 *         axis of the cylinder.
	 */
	public double getDistanceFrom(double[] Q){
		
		// vector Q-P
		Vector3D QminusP= new Vector3D(new Point3D(Q), P);
		
		// length of vector Q-P squared
		double QminusPsquared=QminusP.modSquared();
		
		// projection of Q-P onto orientation vector V
		double QminusPdotV=QminusP.dot(V);
		
		// distance is square root of difference between the two above
		double dist=Math.sqrt(QminusPsquared-QminusPdotV*QminusPdotV);
		
		return dist;
	}
	
	/**
	 * decides whether a point Q is inside the cylinder or not by comparing
	 * the shortest disance of the point to the central axis to the radius of 
	 * the cylinder
	 * 
	 * @param Q the point to test
	 * @return true if dist from axis <= radius, otherwise false
	 */
	public boolean inside(double[] Q){
		
		double dist =getDistanceFrom(Q);
		
		if(dist<=r){
			return true;
		}
		else{
			return false;
		}
	}
	
	/**
	 * solves the quadratic ray-cylinder intersection equation to give
	 * parametric lengths along the step. both roots are returned unless 
	 * they are complex, in which case returns null.
	 * 
	 * TODO: assumes cylinder is parallel to z-axis
	 * 
	 * @param pos walker pos in cell coords
	 * @param s normalised step vector
	 * 
	 * @return both roots, possibly the same. null if complex
	 */
	protected double[] getIntersectionRoots(double[] pos, double[] s){
		
		// space for roots
		double[] t= new double[2];
		
		// position coords in cylinder's coord frame
		double[] Q=new double[D];
		
		double[] Praw= {Prot.x, Prot.y, Prot.z};
		
		// transform cell coords into cylinder's coord frame
		// transate into coord frame
		for(int i=0; i<D; i++){  
			Q[i]=pos[i]-Praw[i];
		}
		// rotate so the z-axes coincide
		
		
		// construct coefficients
		double A=(s[0]*s[0] + s[1]*s[1]);
		double B=2.0*(Q[0]*s[0] + Q[1]*s[1]);
		double C=Q[0]*Q[0] + Q[1]*Q[1] - r*r;
		
		// catch complex case 
		if(B*B<4.0*A*C){
			return null;
		}
		else{
			// otherwise construct solutions
			t[0]= (-B + Math.sqrt(B*B - 4*A*C))/(2.0*A);
			t[1]= (-B - Math.sqrt(B*B - 4*A*C))/(2.0*A);

			for(int i=0; i<2; i++){
				if(Math.abs(t[i])<TINYNUM){
					t[i]=0.0;
				}
			}
			
			
			return t;
		}
		
	}
	
	/**
	 * calculates the surface normal at the intersection of
	 * a step and the cylinder.
	 * 
	 * TODO: assumes cylinder is parallel to the z-axis
	 * 
	 * @param pos the walker pos in cell coords
	 * @param t length param along step
	 * @param s normalised step vector
	 * @param normal space for surface normal
	 */
	protected void getIntersectionNormal(double[] pos, double t, double[] s, double[] normal){
		
		double[] Q= new double[D];
		double[] Praw= {P.x, P.y, P.z};
		
		Vector3D sRot=new Vector3D(s);
		
		// rotate vector into new coord from
		sRot=Rotations.rotateVector(sRot, rotMat);
		
		for(int i=0; i<D; i++){
			Q[i] = (pos[i] - Praw[i]) + t*s[i];
		}
				
		for(int i=0; i<D-1; i++){
			normal[i] = Q[i]/r;
		}
		
		normal[D-1]=0.0;
	}
	
	/**
	 * @return diffusivity inside the cylinder
	 */
	public double getDiffusivityAt(double[] subsCoords){
		if(inside(subsCoords)){
			return d;
		}
		
		return CL_Initializer.DIFF_CONST;
	}

	/**
	 * @return oriented bounding box for cylinder
	 */
	public BoundingBox getBoundingBox(){
		return bbox;
	}

	/**
	 * checks if bounding box is intersected by a step
	 * 
	 * @param subsCoords position mapped onto substrate
	 * @param step step vector
	 * 
	 * @return true of false
	 */
	public boolean boundingBoxIntersects(double[] subsCoords, double[] step) {
		
		return bbox.intersectedBy(subsCoords, step);
	}

	/**
	 * checks if cylinder intersects a specified axis-aligned cubic region.
	 * 
	 * this check works by comparing the distances of 9 points (the cube's
	 * corners plus its centre) to the cylinder axis. If the distance from
	 * the centre to the cylinder axis is less than any of the corners to the
	 * axis then the cylinder intersects, unless the distances of ALL the corners 
	 * to the axis is less than the radius, in which case the cylinder is bigger
	 * than the box and there's no intersection. 
	 * 
	 * @param bottomLeft
	 * @param topRight
	 * 
	 * @return true of false
	 */
	public final boolean intersectsCubicRegion(double[] bottomLeft, double[] topRight){
		
		/** centre -- mean of corners */
		double[] centre= new double[D];
		
		for(int i=0; i<D; i++){
			centre[i]= (bottomLeft[i]+topRight[i])/2.0;
		}
		
		/** corners. this only works in 3d! */
		double[][] corner= new double[][]{{bottomLeft[0], bottomLeft[1], bottomLeft[2]},
				                    	  {bottomLeft[0], bottomLeft[1], topRight[2]},
				                    	  {bottomLeft[0], topRight[1], bottomLeft[2]},
				                    	  {bottomLeft[0], topRight[1], topRight[2]},
				                    	  {topRight[0], bottomLeft[1], bottomLeft[2]},
				                    	  {topRight[0], bottomLeft[1], topRight[2]},
				                    	  {topRight[0], topRight[1], bottomLeft[2]},
				                    	  {topRight[0], topRight[1], topRight[2]}};	
		
		// get cylinder distance
		double cylCentreDist= getDistanceFrom(centre);
		
		// check centre distance against corner distances
		double cylCornerDist;
		boolean smaller=false;
		boolean cornerOutside=false;
		for(int i=0; i<corner.length; i++){
			cylCornerDist= getDistanceFrom(corner[i]);
			
			// if centre distance small than corner distance, flag it up
			if(cylCentreDist<=cylCornerDist){
				smaller=true;
			}
			
			// if centre distance less than any previous corner
			if(cylCornerDist>r){
				cornerOutside=true;
			}
			
			// if both flags are true there's an intersection
			if(smaller && cornerOutside){
				return true;
			}
		}
		
		/** 
		 * if we've got this far, either smaller is false
		 * or all of the corners are less than the radius
		 * from the axis and the cylinder is too big for
		 * the box.
		 */
		return false;
	}
	
	/**
	 * returns the position vector for the cylinder.
	 * If the cylinder is not aligned with the z-axis
	 * the position will be in the cylinder's rotated
	 * coordinate frame.
	 * 
	 * @return P vector as array of doubles
	 */
	public double[] getPosition(){
		return Parr;
	}
	
	/** returns the axis vector of the cylinder
	 * 
	 * @return V vector (new array)
	 */
	public double[] getAxis(){
		return new double[]{V.x, V.y, V.z};
	}
	
	/**
	 * returns the radius of the cylinder
	 * 
	 * @return cylinder radius in meters
	 */
	public double getRadius(){
		return r;
	}
	
	
	public void toFile(Writer writer) throws IOException {
		
		for(int i=0; i<=N; i++){
			double theta= 2.0*Math.PI*((double)i)/N;
			
			double x= P.x + r*Math.cos(theta);
			double y= P.y + r*Math.sin(theta);
			
			writer.write(x+","+y+"\n");
		}
		
		
	}
	
	
	/**
	 * test rotation int arbitrarily orientated
	 * cylinder.
	 * 
	 */
	private static void testArbOrientation(){
		
		double[] P= new double[]{0.0, 4.0E-6, 0.0};
		double[] V= new double[]{1.0, 0.0, 0.0};
		double r= 01.9E-6;
		double p= 0.0;
		
		// cylinder parallel to x-axis
		BasicCylinder xCyl= new BasicCylinder(P, V, r, p);
		
		// test position and step. intersect with step in z-dir
		double[] rawPos= new double[]{5.579550874118729E-6,
									  4.932731024543929E-6, 
									  1.6552955529658E-6};
		double[] rawStep= new double[]{-2.312586608428411E-7,
									   -7.941170655425175E-7,
									    3.7819754468400464E-7};
		
		double[] rawFinalPos= new double[rawPos.length];
		for(int i=0; i<rawFinalPos.length; i++){
			rawFinalPos[i]=rawPos[i]+rawStep[i];
		}
		
		//check the start and end points to see if they are inside the cylinder
		System.err.println("start point inside is "+xCyl.inside(rawPos));
		System.err.println("end point inside is "+xCyl.inside(rawFinalPos));
		
		
		// return spaces
		double[] normal= new double[rawPos.length];
		double[] d= new double[1];
		double[] pSpace= new double[1];
		double[] intDist= new double[1];
		boolean[] in= new boolean[1];
		
		// call the crossing code
		boolean crosses= xCyl.crosses(rawPos, rawStep, normal, d, false, 0.5, intDist, in, pSpace);
		crosses= xCyl.crosses(rawPos, rawStep, normal, d, true, 0.5, intDist, in, pSpace);

		
		// check results
		System.err.println("crosses is "+crosses+" (expected true)");
		System.err.println("ind dist is "+intDist[0]+" (expected 0.1)");
		System.err.println("normal is ("+normal[0]+","+normal[1]+","+normal[2]+") (expected (0.0,0.0,1.0)");
	}
	
	
	/** 
	 * entrypoint. calls test code.
	 * 
	 * @param args
	 */
	public static void main(String[] args){
		
		testArbOrientation();
		
	}
	
}
