package simulation.geometry;

import java.util.logging.Logger;

import numerics.RealMatrix;
import numerics.Rotations;
import numerics.Vector3D;

import simulation.DiffusionSimulation;
import simulation.SimulationParams;


/**
 * substrate will two principle directions.
 * cylinders are square-packed and packed into
 * alternating sheets:
 * 
 * o o o
 * =====
 * o o o
 * 
 * default angle between populations 90 degrees.
 * 
 * The implementation here differs slightly from 
 * the parallel cylinders in that the unit cell 
 * needs to be larger. Here the cell is 2L*2L*2L
 * (cylinder sep L) so that we can cover two rows
 * of cylinders.
 * 
 * First set of cylinders will be parallel to the
 * z-axis. The second population is rotated with 
 * respect to the first about the y axis by the
 * angle specified. Default case, therefore, has
 * one population parallel to z-axis and the other
 * parallel to the x-axis.
 * 
 * @author matt (m.hall@cs.ucl.ac.uk)
 *
 */
public class CrossingCylinderSubstrate extends CylinderSubstrate {

	/** logging object */
	private final Logger logger= Logger.getLogger(this.getClass().getName());
	
	/** dimensionality of space */
	private final double D= DiffusionSimulation.D;
	
	/** crossing angle */
	private final double cAngle;
	
	/** cylinder separation */
	private final double R;
	
	/** 
	 * constructor for default crossing angle.
	 * 
	 * @param R cylinder separation
	 * @param r cylinder radius
	 */
	public CrossingCylinderSubstrate(double R, double r, SimulationParams simParams){
		super(new double[]{2.0*R, 2.0*R, 2.0*R}, simParams);
		
		if(R<2.0*r){
			logger.warning("Cylinder separation on crossing cylinder substrate is less than twice the radius.");
			logger.warning("overlapping cylinders on this substrate will behave unpredictably");
		}
		
		Cylinder[] cylinder= new BasicCylinder[11];
		
		//double[] xAxis= new double[]{1.0, 0.0, 0.0};
		
		double p= simParams.getP();
		
		// set crossing angle
		this.cAngle= SimulationParams.sim_cAngle;

		// construct crossing population cylinder axis
		RealMatrix rotMat= Rotations.getRotMat(new double[]{0.0, 1.0, 0.0}, cAngle);
		
		Vector3D zAxis= new Vector3D(0.0, 0.0, 1.0);
		Vector3D newAxis= Rotations.rotateVector(zAxis, rotMat);
		
		double[] axis= new double[]{newAxis.x, newAxis.y, newAxis.z};
				
		// set cylinder separation
		this.R= R;
		
		// bottom row (z-axis)
		cylinder[0]= new BasicCylinder(new double[]{0.0, 0.0, 0.0}, r, p);
		cylinder[1]= new BasicCylinder(new double[]{R, 0.0, 0.0}, r, p);
		cylinder[2]= new BasicCylinder(new double[]{2.0*R, 0.0, 0.0}, r, p);
	
		// middle row (x-axis)
		cylinder[3]= new BasicCylinder(new double[]{0.0, R, 0.0}, axis, r, p);
		cylinder[4]= new BasicCylinder(new double[]{0.0, R, R}, axis, r, p);
		cylinder[5]= new BasicCylinder(new double[]{0.0, R, 2.0*R}, axis, r, p);

		// bottom row (z-axis)
		cylinder[6]= new BasicCylinder(new double[]{0.0, 2.0*R, 0.0}, r, p);
		cylinder[7]= new BasicCylinder(new double[]{R, 2.0*R, 0.0}, r, p);
		cylinder[8]= new BasicCylinder(new double[]{2.0*R, 2.0*R, 0.0}, r, p);

		// extras on the middle row
		cylinder[9]= new BasicCylinder(new double[]{0.0, R, -R}, axis, r, p);
		cylinder[10]= new BasicCylinder(new double[]{0.0, R, 3.0*R}, axis, r, p);
        
		
		super.setCylinders(cylinder);
		
	}
	
	/**
	 * returns the peak coordinate (the place where the
	 * spike sits on each axis for delta-peaked initial
	 * conditions)
	 * 
	 * @return cylinder sep (half the size of the unit cell)
	 * 
	 */
	public double getPeakCoord() {
		
		return R;
	}

	/**
	 * return the size of the unit cell
	 * 
	 * @return 2 x cylinder sep
	 */
	public double[] getSubstrateSize() {
		
		return new double[] {2.0*R, 2.0*R, 2.0*R};
	}

}
