package simulation.geometry;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.logging.Logger;

import simulation.SimulationParams;


import numerics.MTRandom;


/**
 * This type of substrate is used to model the changes in ADC
 * in the immediate aftermath of ischemic insult (i love that phrase...)
 * It models a substrate of cylinders of constant radius in disordered
 * positions across a substrate of specified size.
 * 
 * The substrate enlarges the radius of all cylinders
 * 
 * @author matt
 *
 */
public class StiffInflammationSubstrate extends CylinderSubstrate {

	/** logging object */
	private final Logger logger 
			= Logger.getLogger(this.getClass().getName());
	
	/** random number generator */
	private final MTRandom rng;
	
	/** number of inflammation steps */
	private final int steps;
	
	/** size of substrate */
	private final double[] L;
	
	/** number of cylinders */
	private final int N;
	
	/** the radius increment */
	private final double rinc;
	
	/**	axis of cylinders */
	private final double[] axis= new double[]{0.0, 0.0, 1.0};

	/** flags that indicate if each cylinder is still expanding
	 * or if it has stopped
	 */
	private final boolean[] expanding;
	
	/** set of nearest neighbour cylinders. entry n is the index of
	 * the closest other cylinder to cylinder n
	 */
	private final int[] nearest;
	
	/**
	 * set of distances to nearest neighbours (axis to axis distances)
	 */
	private final double[] nearestDist;
	
	/** 
	 * don't want to the first time...
	 */
	private boolean firstTime=true;

	/**
	 * membrane permiability
	 */
	private final double p;
	
	/**
	 * numerical tolerence
	 */
	private static final double TOL=1E-14;
	
	/** 
	 * construct an inflammation substrate of size L containing N
	 * cylinders initially with radius rmin, expanding to radius
	 * rmax after generating voxels voxels
	 * 
	 * @param L size of substrate (meters)
	 * @param N number of cylinders
	 * @param rmin initial cylinder radius
	 * @param rmax final cylinder radius
	 * @param voxels number of voxels we're generating (steps in inflammation)
	 * 
	 * TODO: assumes we can always fit N clyinders in
	 * TODO: takes no account of periodic boundaries
	 */
	public StiffInflammationSubstrate(double[] L, int N, double rmin, double rmax, int steps, SimulationParams simParams){
		
		super(L, simParams);
		
		// initialise the random number generator
		rng=new MTRandom(7121);

		this.p=simParams.getP();

		
		// initialise the substrate fields
		this.L=L;                       // size
		this.N=N;                       // number of cylinders
		this.rinc=(rmax-rmin)/steps;    // radius increment
		this.steps=steps;
		
		
		// initialise the cylinder array
		cylinder= new BasicCylinder[N];
		
		// initialise position and orientation vectors
		double[] pos= new double[D];
		
		pos[2]=0.0;
		
		for(int i=0; i<N; i++){
			
			boolean ok=false;
			int count =0;
			
			while(!ok){
				ok=true;

				// pick some new coordinates for the 
				double x= rng.nextDouble()*L[0];
				double y= rng.nextDouble()*L[1];
			
				pos[0]=x;
				pos[1]=y;
			
				for(int j=0; j<i; j++){
					if(cylinder[j].getDistanceFrom(pos)<=2.0*rmin){
						ok=false;
						break;
					}
				}

				count++;
			}
	
			
			// create the cylinder
			cylinder[i]= new BasicCylinder(pos, rmin, p);
		}
		
		// initialise expanding flags array
		expanding= new boolean[N];
		for(int i=0; i<expanding.length; i++){
			expanding[i]=true;
		}
		
		// initialise array of nearest neighbours
		int[] tempNearest= new int[N];
		double[] tempNearestDist= new double[N];
		
		for(int i=0; i<N; i++){
			double min_dist;
			int neighbour=-1;
			
			double[] cylinderPos= cylinder[i].getPosition();
			
			min_dist=Double.MAX_VALUE;
			for(int j=0; j<N; j++){
				if(j==i){
					continue;
				}
				
				double dist=cylinder[j].getDistanceFrom(cylinderPos);
				if(dist<min_dist){
					min_dist=dist;
					neighbour=j;
				}
			}
			
			tempNearest[i]=neighbour;
			tempNearestDist[i]=min_dist;
		}
		
		// finally, sort the cylinders into order of nearest-neighnbour distance order
		Cylinder[] reorderedCyl= new BasicCylinder[N];
		int[] reorderedNeighbour= new int[N];
		double[] reorderedDist= new double[N];
		boolean[] done= new boolean[N];

		// set all done flags to false
		for(int i=0; i<N; i++){
			done[i]=false;
		}
		
		// sort (somewhat inefficiently) into order
		for(int i=0; i<N; i++){
			double smallestDist=Double.MAX_VALUE;
			int smallestInd=-1;
			for(int j=0; j<N; j++){
				if(done[j]){
					continue;
				}
				if(tempNearestDist[j]<smallestDist){
					smallestDist=tempNearestDist[j];
					smallestInd=j;
				}
			}
			reorderedCyl[i]=cylinder[smallestInd];
			reorderedDist[i]=tempNearestDist[smallestInd];
			done[smallestInd]=true;
		}
		
		// re-evaluate the nearest neighbours (reordering make originals invalid)
		for(int i=0; i<N; i++){
			double min_dist;
			int neighbour=-1;
			
			double[] cylinderPos= reorderedCyl[i].getPosition();
			
			min_dist=Double.MAX_VALUE;
			for(int j=0; j<N; j++){
				if(j==i){
					continue;
				}
				
				double dist=reorderedCyl[j].getDistanceFrom(cylinderPos);
				if(dist<min_dist){
					min_dist=dist;
					neighbour=j;
				}
			}
			
			reorderedNeighbour[i]=neighbour;
		}
				
		// copy over the reordered arrays
		cylinder=reorderedCyl;
		nearestDist=reorderedDist;
		nearest=reorderedNeighbour;
	}
	
	
	/**
	 * overrides the initialiser function. this activates the 
	 * swelling function at the beginning of each run, except for
	 * the first time, when we want to ignore the call 
	 */
	public final void init(){
		
		if(firstTime){
			firstTime=false;
		}
		else{
			swell();
		}
	}
	
	
	/**
	 * enlarges the radii of all cylinders that are still swelling,
	 * updates the array of expanding flags 
	 *
	 */
	private final void swell(){

		int changes=1;
		
		double[] desiredRad = new double[N];
		double[] newRad = new double[N];
		
		//System.err.println("rinc = "+rinc);
		
		// assign new radii
		for(int i=0; i<N; i++){
			if(expanding[i]){
				desiredRad[i]=cylinder[i].getRadius()+rinc;
			}
			else{
				desiredRad[i]=cylinder[i].getRadius();
			}
			newRad[i]=desiredRad[i];
		}
		
		while(changes!=0){
			changes=0;

			int a=37;
			int b=33;
			
			for(int i=0; i<N; i++){
				int neighbour= nearest[i];
				double radiiSum=newRad[i]+newRad[neighbour];

//				if(i==a|i==b){
//					System.err.println("cylinder "+i+": radius= "+cylinder[i].getRadius()+" nearest= "+neighbour+", neighbour radius = "+ cylinder[neighbour].getRadius()+" dist= "+nearestDist[i]+" radiusSum= "+radiiSum+" rinc = "+rinc);
//				}
				
				// if there's not enough room between the 
				// cylinders for the new radii amend the distances
				// and halt the expansion of both cylinders
				if(radiiSum>nearestDist[i]){

					//double currentSum = cylinder[i].getRadius() + cylinder[neighbour].getRadius();
					
//					if(i==a|i==b){
//						System.err.println("\tradiiSum too big. expanding["+i+"] = "+expanding[i]+", expanding["+neighbour+"]= "+expanding[neighbour]);	
//					}
					
					if(expanding[i]){
						changes++;
						
						if(expanding[neighbour]){
							// both are expanding
							double currentSum;
							
							if(i>neighbour){
								currentSum=cylinder[i].getRadius()+newRad[neighbour];
							}
							else{
								currentSum=cylinder[i].getRadius()+cylinder[neighbour].getRadius();
							}
							
							double newInc=(nearestDist[i]-currentSum)/2.0;
							
//							if(i==a|i==b){		
//								System.err.println("\t\tboth expanding. newInc= ("+nearestDist[i]+" - "+currentSum+")/2 = "+newInc);
//							}
							
							newRad[i]=cylinder[i].getRadius()+newInc;
							newRad[neighbour]=cylinder[neighbour].getRadius()+newInc;
//							if(i==a|i==b){
//								System.err.println("\t\t\tnewRad["+i+"] = "+ cylinder[i].getRadius()+" + "+newInc+" = "+newRad[i]);
//								System.err.println("\t\t\tnewRad["+neighbour+"] = "+ cylinder[neighbour].getRadius()+" + "+newInc+" = "+newRad[neighbour]);
//							}
							
							expanding[i]=false;
							expanding[neighbour]=false;
						}
						else{
							// only current is expanding
							
							double newInc=(nearestDist[i]-radiiSum);
//							if(i==a|i==b){
//								System.err.println("\t\tonly "+i+" expanding. newInc= "+nearestDist[i]+" - "+radiiSum+" = "+newInc);
//							}
							
							newRad[i]=cylinder[i].getRadius() + newInc;
//							if(i==a|i==b){
//								System.err.println("\t\t\tnewRad["+i+"] = "+ cylinder[i].getRadius()+" + "+newInc+" = "+newRad[i]);
//							}
							expanding[i]=false;
						}
					}
				}
			}			
		}

		
		// sanity check to see if any overlapping cylinders remain
		for(int i=0; i<cylinder.length;i++){
			int neighbour=nearest[i];
			
			if(cylinder[i].getRadius()+cylinder[neighbour].getRadius()>=nearestDist[i]+TOL){
				String errMess= new String("cylinders "+i+" and "+neighbour+
					                   " are overlapping ("+cylinder[i].getRadius()+" + "+
					                   cylinder[neighbour].getRadius()+"("+
					                   (cylinder[i].getRadius()+cylinder[neighbour].getRadius())+
					                   ") > sep "+nearestDist[i]);
			
				logger.warning(errMess);
			}
		}
		
		
		// remake cyliners with new radii
		for(int i=0; i<cylinder.length; i++){
			cylinder[i] = new BasicCylinder(cylinder[i].getPosition(), newRad[i], p);
		}
		
	}
	
	
	
	/**
	 * maps a set of global coordinates into the periodic substrate space
	 * 
	 * @param walkerPos the original posiiton
	 * @param newPos the position in substrate space
	 * 
	 */
	/*protected void getSubstrateCoords(double[] walkerPos, double[] offset, double[] newPos) {
		for(int i=0; i<walkerPos.length; i++){
			newPos[i]=walkerPos[i]+offset[i]-Math.floor(walkerPos[i]/L);
		}
	}*/

	/**
	 * calculates the volume fraction contained within cylinders.
	 * this is just the ratio of the total cross sectional area of
	 * cylinders to the substrate cross section.
	 * 
	 * @return volume fraction in cylinders
	 */
	public final double getVolumeFraction(){
		
		double substrateArea= L[0]*L[1];
		
		double cylinderArea=0.0;
		
		for(int i=0; i<cylinder.length; i++){
			double r= cylinder[i].getRadius();
			
			cylinderArea+=Math.PI*r*r;
		}
		
		return cylinderArea/substrateArea;
	}
	
	
	
	/** returns the substrate size in meters
	 * 
	 * @return length of a side of the substrate
	 */
	public double[] getSubstrateSize() {
		return L;
	}

	/**
	 * tells the initialiser where to place the delta-peak of walkers
	 * if that form of initial conditions is specified
	 * 
	 * @return middle of the substrate
	 */
	public double getPeakCoord() {
		return L[0]/2.0;
	}

	
	public final void extractVolFracBehaviour(){
		
		Writer fracWriter= null;
		
		try{
			fracWriter = new BufferedWriter(new FileWriter("volFrac.dat"));
			
			boolean stillgoing=true;
			int i=0;
			
			while(stillgoing){
				init();
				
				fracWriter.write(i+" "+getVolumeFraction()+"\n");
				System.err.print(i+" "+getVolumeFraction()+"\n");
				
				i++;
				
				stillgoing=false;
				for(int j=0; j<expanding.length; j++){
					if(expanding[j]){
						stillgoing=true;
						break;
					}
				}
			}
			
			fracWriter.flush();
			fracWriter.close();
		}
		catch(IOException e){
			
			logger.severe(e.getMessage());
			logger.severe(e.getStackTrace().toString());
			
			throw new RuntimeException(e);
		}
		
		
	}
	
	
	
	/**
	 * @param args command line arguments
	 */
	public static void main(String[] args) {
		// TODO Auto-generated method stub
		StiffInflammationSubstrate ifSubst = new StiffInflammationSubstrate(new double[]{10.0, 10.0, 10.0}, 1000, 0.005, 5.005, 1000, null);
		
		ifSubst.extractVolFracBehaviour();
	}

}
