/* Substrate.java created on 28-Nov-2005
 * (simulation)
 * 
 * author: Matt Hall (m.hall@cs.ucl.ac.uk)
 * 
 */
package simulation.geometry;

import imaging.SchemeV0;

import java.util.logging.Logger;

import numerics.MTRandom;

import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import simulation.dynamics.StepAmender;
import simulation.dynamics.StepAmenderFactory;
import simulation.dynamics.StepGeneratorFactory;
import simulation.dynamics.Walker;
import tools.CL_Initializer;


/**
 *  Camino fibre reconstruction and tracking toolkit
 * 
 * Substrate (simulation)
 * 
 * 
 * 
 *
 * @author Matt Hall (m.hall@cs.ucl.ac.uk)
 *
 */
public abstract class Substrate{

    /**
     * logging object
     */
    Logger logger = Logger.getLogger(this.getClass().getName());
    
    /**
     * the dimensionality of the substrate
     */
    private final int D=DiffusionSimulation.D;
    
    /**
     * membrane transition probability
     */
    private final double[] p= new double[1];
    
    /**
     * membrane transition probability for recheck
     */
    private final double[] recheckp= new double[1];

    
    /**
     * substrate dimensions
     */
    protected final double[] L;
    
    /**
     * space to store normal
     */
    private final double[] normal;
    
    /**
     * space to store normal in recheck
     */
    private final double[] recheckNormal;

    
    /**
     * space to store amended step
     */
    private final double[] amended;

    /**
     * space to store amended step in recheck
     */
    private final double[] recheckAmended;

    
    
    /**
     * space to store unamended step
     */
    private final double[] unamended;

    /**
     * space to store unamended step in reckeck
     */
    private final double[] recheckUnamended;

    /**
     * space to store step to barrier
     */
    private final double[] toBarrier;

    /**
     * space to store step to barrier
     */
    private final double[] recheckToBarrier;

    
    /**
     * point of intersection
     */
    private final double[] intPoint;
    
    /**
     * space for amended step
     */
    private final double[] newStep;

    /**
     * space to store membrane distance
     */
    private final double[] d;

    /**
     * space to store membrane distance in recheck
     */
    private final double[] recheckd;

    /**
     * space to store intracellular flag
     */
    private final boolean[] in;
    
    /**
     * space to store intracellular flag in recheck
     */
    private final boolean[] recheckIn;

    /**
     * position of walker on substrate
     */
    protected final double[] subsCoords= new double[D];
    
    /**
     * rechecking offset
     */
    private final double[] offset= new double[D];
    
    /**
     * random number generator
     */
    private final MTRandom twister;
    
    /**
     * step amender
     */
    private final StepAmender amender;
        
    /**
     * cheat factor for curved surfaces
     */
    public static double factor= 1.0;
    
    /** 
     * cheat factor for detected interaction
     */
    protected double[] intFactor;

    /** 
     * fraction of substrate size to distance of test boundary from actual 
     */
    public final double a= 100000;
    
    /** 
     * boundary bottom-left 
     */
    private final double[] xmin= new double[D];
    
    /** 
     * boundary top-right 
     */
    private final double[] xmax= new double[D];
    
    /**
     * boundary widths in each direction
     */
    public double[] border= new double[D];
    
    /**
     * flag to read-out cylinder info
     */
    protected boolean substrateInfo= SimulationParams.substrateInfo;
    
    
    /** 
     * constructor with p specified
     * 
     * @param simParams simulation parameters
     * @param substrateDims substrate dimensions
     */
    public Substrate(SimulationParams simParams, double[] substrateDims){

    	//this.p=simParams.getP();
        this.L=substrateDims;
        
        this.normal=new double[D];
        this.recheckNormal= new double[D];
        this.amended= new double[D];
        this.recheckAmended= new double[D];
        this.unamended= new double[D];
        this.recheckUnamended= new double[D];
        this.d= new double[1];
        this.recheckd= new double[1];
        this.in= new boolean[1];
        this.recheckIn= new boolean[1];
        this.toBarrier= new double[D];
        this.recheckToBarrier= new double[D];
        this.intPoint= new double[D];
        this.newStep= new double[D];
        
        this.twister=new MTRandom(CL_Initializer.seed);
        
        this.amender=StepAmenderFactory.getStepAmender(SimulationParams.sim_amender_type);
        
        
        for(int i=0; i<D; i++){
            xmin[i]=-L[i]/a;
            xmax[i]= L[i]*(1+1/a);
            
            border[i]=L[i]/a;
        }
    }
        
    
    
    public final void amend(Walker walker, double[] step, double t, int n){
   	
        boolean crosses=false;
    	
        boolean isAmended= false;
        
        boolean stepMade=false;
        
        double origLength=0.0;
    	
        // reset the new step
        for(int i=0; i<D; i++){
        	newStep[i]=0.0;
        }
        
        // check if step will cross a barrier
    	crosses=crossesMembrane(walker, newStep, step, normal, d, stepMade, origLength, in, p);
        
        int count=0;
    	
        // if it crosses, we have to amend it and check for more crossings
        while(crosses){                    
        	
        	getSubstrateCoords(walker.r, newStep, subsCoords);
        	
        	count++;
            // amend the step -- this gives the step the barrier 
            // and the amended and unamended steps away from it
            amender.amendStep(walker, subsCoords, newStep, step, normal, d, origLength, toBarrier, amended, unamended, L, t, n);
                        
            // set the amended flag
            isAmended= true;
            
            // always step to the barrier
            //walker.makeStep(toBarrier);                    
            for(int j=0; j<D; j++){
            	newStep[j]+=toBarrier[j];
            }
            
        	// pick a number
            double prob=twister.nextDouble();
            
            
            if(prob>=p[0]){
            	// we're reflecting and new step is the 
            	// amended step reflected away from the barrier
            	for(int j=0; j<D; j++){
            		step[j]=amended[j];
            	}
            }
            else{
            	// we're not reflecting, and the new step 
            	// is the unamended step through the barrier
            	for(int j=0; j<D; j++){
            		step[j]=unamended[j];
            	}
            }
            
            stepMade= true;
            
            // in either case, must check if the new step also crosses a barrier 
            crosses=crossesMembrane(walker, newStep, step, normal, d, stepMade, origLength, in, p);
            
        }
        
        // if step is amended
        if(isAmended){
        	for(int i=0; i<D; i++){
        		// add the final part of the step to the newStep
        		newStep[i]+=step[i];
        		
        		// replace the step value for returning
        		step[i]=newStep[i];
        	}
        }
    }
    
    
    /**
     * checks intersection with the boundaries of the substrate. The
     * actual barriers that are checked against are a very tiny amount 
     * wider than those of the actual substrate. This is to ensure that 
     * there's no issue with floating point accuracy when the walker is 
     * mapped back into substrate coords the next time we go round the 
     * barrier check loop.
     * 
     * 
     * @param walkerPos the position of the walker in substrate coords
     * @param offset current offset from starting point
     * @param stepVector proposed step vector
     * @param normal space to store boundary normal
     * @param d distance to boundary
     * @param skipCurrent should we skip the current boundary
     * @param origLength original step length
     * @param in initially intracellular?
     * @param p permeability of boundary (always one here)
     * 
     * @return true is boundary crossed
     */
    protected boolean checkBoundaryIntersection(double[] walkerPos, double[] offset, double[] stepVector,
            double[] normal, double[] d, boolean skipCurrent, double origLength, double[] intDist,  
            boolean[] in, double[] p){
        
       
        //getSubstrateCoords(walkerPos, offset, subsCoords);
        
        double[] dist= new double[D];
        int[] crossInd= new int[D];
        boolean[] minMax= new boolean[D]; // true for min, false for max
        
        int crossCount=0;
        
        
        for(int i=0; i<D; i++){
            double endPoint=walkerPos[i]+stepVector[i];
            if(endPoint<xmin[i]){
                // we're crossing on the lower end
                
                //dist[crossCount]=Math.abs(xmin[i]-walkerPos[i]);
                dist[crossCount]=(xmin[i]-walkerPos[i])/stepVector[i];
                crossInd[crossCount]=i;
                minMax[crossCount]=true;
                
                crossCount++;
            }
            else if(endPoint>xmax[i]){
                // we're crossing at the upper end
                
                dist[crossCount]= (xmax[i]-walkerPos[i])/stepVector[i];
                crossInd[crossCount]= i;
                minMax[crossCount]=false;
                
                crossCount++;
            }
        }
        
        // if no intersection return false
        if(crossCount==0){
            return false;
        }
        
        // otherwise find closest intersection
        int closest=0;
        double minDist=dist[0];
        int minInd=crossInd[0];
        
        for(int i=1; i<crossCount; i++){
            if(dist[i]<minDist){
                minDist=dist[i];
                minInd=crossInd[i];
                closest=i;
            }
        }
        
        // assemble geometric data 
        double stepLen=0.0;
        for(int i=0; i<D; i++){
            stepLen+=stepVector[i]*stepVector[i];
        }
        stepLen=Math.sqrt(stepLen);
        
        if(minMax[closest]){
            d[0]= -L[minInd]/a;
        }
        else{
            d[0]=L[minInd]*(1+1/a);
        }

        //intDist[0]=(minDist/Math.abs(stepVector[minInd]));
        intDist[0]=minDist;
        
        // always totally permeable
        p[0]=1.0;
        
        // normal parallel to appropriate axis, sign not important
        for(int i=0; i<D; i++){
            if(i==minInd){
                normal[i]=1.0;
            }
            else{
                normal[i]=0.0;
            }
        }
        
        return true;
    }
    
    /** 
     * checks if a step will cross a barrier or not
     * 
     * @param walker the walker
     * @param step the step vector
     * @param normal container for the normal
     * @param skipCurrent flag saying whether 
     *                     to ignore the current 
     *                     normal and distance 
     *                     or not
     * 
     * @return the geometry's answer
     */
    public abstract boolean crossesMembrane(Walker walker, double[] offset, double[] step, double[] normal, double[] d, boolean skipCurrent, double origLength, boolean[] in, double[] p);
    
    /**
     * @return the size of the substrate
     */
    public abstract double[] getSubstrateSize();
    
    
    /** 
     * @return substrate peak coord
     */
    public abstract double getPeakCoord();

        
    /**
     *  initialiser for substrate. usually does nothing, but it's
     *  there if we need it for whatever reason.
     *  
     */
    public abstract void init();
    
    
    /**
     * checks if a walker is in intracellular (true) or
     * extracellular (false) space
     * 
     * @param walker the walker to check
     * 
     * @return true if intracellular, false otherwise
     */
    public abstract boolean intracellular(Walker walker);
    
    /**
     * returns the diffusivity at a given location on the substrate
     * 
     * @param location in world coords
     * @return diffusivity at that location
     */
    public double getDiffusivityAt(double[] walkerPos){
    	
    	return CL_Initializer.DIFF_CONST;
    }
    
    /** 
     * maps physical walker location into substrate cell.
     * this basic version assumes the cell is square.
     * 
     * @param walker the walker whose position we are mapping
     * 
     */
    public void getSubstrateCoords(double[] walkerPos, double[] offset, double[] newPos){
    	
    	for(int i=0; i<D; i++){
    		newPos[i]=walkerPos[i]+offset[i];
    		double windingNum=Math.floor(newPos[i]/L[i]);
    		newPos[i]=newPos[i]-windingNum*L[i];
    	}

    }
    
    /**
     * test access to the amender. this is used in various test cases
     * across the various substrate classes
     * 
     * @param walker walker making the step
     * @param step step being made
     * @param normal barrier normal
     * @param d distance to barrier
     * @param origLength original length opf step
     * @param toBarrier step to the barrier
     * @param amended amended portion of step
     * @param unamended unamended portion of step
     */
    public void testAmendment(Walker walker, double[] step, 
    		double[] normal, double[] d, double origLength, 
    		double[] toBarrier, double[] amended, double[] unamended){
    	
    	double[] offset= new double[]{0.0, 0.0, 0.0};
    	
    	getSubstrateCoords(walker.r, offset, subsCoords);
    	
    	// pass everything directly to the amender
    	amender.amendStep(walker, subsCoords, offset, step, normal, d, origLength, toBarrier, 
    					amended, unamended, L, 0.0, 1);
    	
    }
    
    
    /** 
     * test of step amending code.
     * 
     * creates a walker and a step with known geometry and tests what comes out
     * to see that the geometry is amending steps in the correct way.
     * 
     * walker at coords (1.5, 1.5), plane membrane at p=1.0 from origin with
     * normal (0.0, 1.0) and a step of length 2.0, with vector (1.0, -1.0).
     * 
     * should mean that the step is amended to (1.0, 0.0) and the final coords
     * of the walker are (2.5, 1.5).
     */
    public static void testStepAmendment() {

        // the next four lines are dummies that allow us to 
        // instantiate a substrate. The details of the Geometry
        // object and the simulation parameters are all ignored
        // in this test code.
        Object[] geomParams = new Object[] {new Double(1.0), new Integer(3)};
        double[] stepParams = new double[] {0.1};

        SchemeV0 imParams=SchemeV0.getSchemeV0(7);       
        SimulationParams simParams= new SimulationParams(1, 1000, 0.0, 
                SimulationParams.SPIKE, 0, geomParams, StepGeneratorFactory.FIXEDLENGTH, 
                1.5, imParams);
 
        Substrate substrate = SubstrateFactory.getSubstrate(SubstrateFactory.CYL_1_INFLAM, geomParams, simParams);
        
        
        double[] r0= new double[] {1.5, 1.5, 0.0};      // (initial) walker position
        double[] step= new double[] {1.0, -1.0, 0.0};   // step vector
        
        double[] normal= new double[] {0.0, 1.0, 0.0};  // plane normal
        double[] d=new double[] {1.0};             // distance of plane from origin
        
        Walker walker = new Walker(r0);
        
        // test the new step size
/*        assertTrue(Math.abs(step[0]-1.0)<=1E-12);
        assertTrue(Math.abs(step[1])<=1E-12);
        assertTrue(Math.abs(step[2])<=1E-12);
        
        // test the walker's ability to make the step
        assertTrue(Math.abs(walker.r[0]-2.5)<=1E-12);
        assertTrue(Math.abs(walker.r[1]-1.5)<=1E-12);
        assertTrue(Math.abs(walker.r[2])<1E-12);
*/        
            
    }
    
    
    protected static final void testBoundaryIntersection(){
        
        Object[] geomParams = new Object[4];
        
        SimulationParams simParams= new SimulationParams(1, 1000, 0.0, SimulationParams.UNIFORM, SubstrateFactory.CYL_1_FIXED, geomParams, StepGeneratorFactory.FIXEDLENGTH, 1E-5, 0.1);
        
        CylinderSubstrate cylSubs = new ParallelCylinderSubstrate(4E-6, 1E-6, ParallelCylinderSubstrate.SQUARE, simParams);
        
        
        // construct walker and step for lower x boundary
        double[] r0=new double[]{2E-6, 2E-6, 2E-6};
        Walker walker = new Walker(r0);
        
        double[] step= new double[]{-3E-6, 0.0, 0.0};
        
        cylSubs.amend(walker, step, 0.0, 0);
        
        // construct walker and step for upper x boundary
        r0= new double[]{6E-6, 6E-6, 6E-6};
        walker= new Walker(r0);
        
        step = new double[]{3E-6, 0.0, 0.0};
        
        cylSubs.amend(walker, step, 0.0, 0);
        
    }
    
    
    private static final void testIntersectionOrdering(){
        
        Object[] geomParams = new Object[4];
        
        SimulationParams simParams= new SimulationParams(1, 1000, 0.0, SimulationParams.UNIFORM, SubstrateFactory.CYL_1_FIXED, geomParams, StepGeneratorFactory.FIXEDLENGTH, 1E-5, 0.1);
        
        CylinderSubstrate cylSubs = new ParallelCylinderSubstrate(4E-6, 1E-6, ParallelCylinderSubstrate.SQUARE, simParams);
        
        // test multiple intersections. cylinder then boundary
        double[] r0= new double[]{4E-6, 6E-6, 4E-6};
        Walker walker= new Walker(r0);
        
        double[] step= new double[]{0.0, 2.5E-6, 0.0};
        
        cylSubs.amend(walker, step, 0.0, 0);
        
        
        // boundary then cylinder
        r0= new double[]{0.5E-6, 4E-6, 4E-6};
        walker = new Walker(r0);
        
        step= new double[]{-3E-6, 0.0, 0.0};
        
        cylSubs.amend(walker, step, 0.0, 0);
        
    }
    
    
    
    private static final void testVeryLongStep(){
        
        Object[] geomParams = new Object[4];
        
        SimulationParams simParams= new SimulationParams(1, 1000, 0.0, SimulationParams.UNIFORM, SubstrateFactory.CYL_1_FIXED, geomParams, StepGeneratorFactory.FIXEDLENGTH, 1E-5, 0.1);
        
        CylinderSubstrate cylSubs = new ParallelCylinderSubstrate(4E-6, 1E-6, ParallelCylinderSubstrate.SQUARE, simParams);
        
        // step that wraps the substrate more than once (downwards)
        double[] r0= new double[]{6E-6, 2E-6, 2E-6};
        Walker walker= new Walker(r0);
        
        double[] step= new double[]{0.0, -20E-6, 0.0};
        
        cylSubs.amend(walker, step, 0.0, 0);
        
        
        // now do same upwards
        step= new double[]{0.0, 20E-6, 0.0};
        
        cylSubs.amend(walker, step, 0.0, 0);
        
        
        // and finally, lond steps with cylinders
        
    }
    
    
    public static void main(String[] args){
        
        //testBoundaryIntersection();
        
        //testIntersectionOrdering();
        
        testVeryLongStep();
    }
}
