/* SubstrateFactory.java created on 28-Nov-2005
 * (simulation)
 * 
 * author: Matt Hall (m.hall@cs.ucl.ac.uk)
 * 
 */
package simulation.geometry;

import java.util.logging.Logger;

import simulation.SimulationParams;


import misc.LoggedException;

/**
 *  Camino fibre reconstruction and tracking toolkit
 * 
 * SubstrateFactory (simulation)
 * 
 * 
 * 
 *
 * @author Matt Hall (m.hall@cs.ucl.ac.uk)
 *
 */
public class SubstrateFactory {

    private static Logger logger=Logger.getLogger("simulation.GeometryFactory");
    
    
    // cell-based geometries
    public static final int CELL_ISO=0;             /** all cells have barriers */
        
    public static final int CELL_STRIPED=1;         /** striped lattice */

    public static final int CELL_MODULAR=2;         /** cubic block separated by space all round */
    
    public static final int CELL_FIXEDFRAC=3;       /** first n planes in cubes filled */
    
    public static final int CELL_PERC=4;            /** percolation cluster */
    
    // parallel cylinder geometries 
    public static final int CYL_1_FIXED=5;          /** parallel fixed radius cylinders in 1 dir */
    
    public static final int CYL_1_DISTRIB=6;        /** parallel distributed radius cylinders in 1 dir */
    
    public static final int CYL_1_INFLAM=7;         /** parallel cylinders on inflaming substrate 
    												 * (randomly positioned, incrementing constant radii) */
    			
    public static final int CYL_1_FACET=8;			/** facetted cylinders, rather than smooth surfaces */
    
    public static final int TRI_PLY_MESH=9;			/** triangular mesh from PLY file */
    
    public static final int CYL_1_PERC=10;			/** randomly placed, parallel, non-overlapping cylinders */

    public static final int CYL_1_MYELIN=11;		/** sen and basser's myelinated cylinder substrates */
    
    
    // crossing cylinder geometries
    public static final int CYL_2_FIXED=12;			/** basic crossing fibres */
    
    
    
    // misc substrates
    public static final int EMPTY=13;               /** substrate containing nothing at all */
    
    /** factory method for making geometries.
     * 
     * @param geometryType geometry type code
     * @param geometryParams additional parameters
     * 
     * @return new Geometry instance
     */
    public static Substrate getSubstrate(int substrateType, Object[] geometryParams, SimulationParams simParams){
    
        if(substrateType==CELL_ISO){
            if(geometryParams.length!=2){
                String errMess = new String("isocellular lattice needs 2 parameters, but was passed "+geometryParams.length);
                logger.severe(errMess);
                throw new RuntimeException(errMess);
            }
            
            return new IsoCellularLattice(((Double)geometryParams[0]).doubleValue(), ((Integer)geometryParams[1]).intValue(), simParams);
        }
        else if(substrateType==CELL_STRIPED){
            if(geometryParams.length!=3){
                String errMess = new String("striped cellular lattice needs 3 parameters, but was passed "+geometryParams.length);
                logger.severe(errMess);
                throw new RuntimeException(errMess);
            }

            return new StripedCellularLattice(((Double)geometryParams[0]).doubleValue(), 
                    	((Integer)geometryParams[1]).intValue(), ((Integer)geometryParams[2]).intValue(), simParams);
        }
        else if(substrateType==CELL_MODULAR){
            if(geometryParams.length!=4){
                String errMess = new String("modular lattice needs 4 parameters, but was passed "+geometryParams.length);
                logger.severe(errMess);
                throw new RuntimeException(errMess);
            }

            return new ModCellularLattice(((Double)geometryParams[0]).doubleValue(), ((Integer)geometryParams[1]).intValue(),
                    	((Double)geometryParams[2]).intValue(), ((Integer)geometryParams[3]).intValue(), simParams); 
            
        }
        else if(substrateType==CELL_FIXEDFRAC){
            if(geometryParams.length!=3){
                String errMess = new String("fixed fraction cellular lattice needs 3 parameters, but was passed "+geometryParams.length);
                logger.severe(errMess);
                throw new RuntimeException(errMess);
            }

            return new FixedFracCellularLattice(((Double)geometryParams[0]).doubleValue(), ((Integer)geometryParams[1]).intValue(), 
                    	((Double)geometryParams[2]).doubleValue(), simParams);
        }
        else if(substrateType==CELL_PERC){
            if(geometryParams.length!=3){
                String errMess = new String("percolation lattice needs 3 parameters, but was passed "+geometryParams.length);
                logger.severe(errMess);
                throw new RuntimeException(errMess);    
            }
            
            return new PercCellularLattice(((Double)geometryParams[0]).doubleValue(), ((Integer)geometryParams[1]).intValue(),
            		((Double)geometryParams[2]).doubleValue(), simParams);
        }
        else if(substrateType==CYL_1_FIXED){
        	if(geometryParams.length!=3){
        		String errMess = new String("parallel cylinder substrate requires 3 params, but was passed "+geometryParams.length);
        		logger.severe(errMess);
        		throw new RuntimeException(errMess);
        	}
        	
        	return new ParallelCylinderSubstrate(((Double)geometryParams[0]).doubleValue(), 
        			((Double)geometryParams[1]).doubleValue(), ((Integer)geometryParams[2]).intValue(), simParams);
        }
        else if(substrateType==CYL_1_DISTRIB){
        	if(geometryParams.length!=4){
        		String errMess= new String("distributed cylinder substrate requires 4 parameters but was passed "+geometryParams.length);
        		logger.severe(errMess);
        		throw new RuntimeException(errMess);
        	}

        	return new DistributedRadiusCylinderSubstrate(((Double)geometryParams[0]).doubleValue(), 
        			((Integer)geometryParams[1]).intValue(), ((Double)geometryParams[2]).doubleValue(), 
        			((Double)geometryParams[3]).doubleValue(), simParams);
        }
        else if(substrateType==CYL_1_INFLAM){
        	if(geometryParams.length!=6){
        		String errMess = new String ("Inflammatory cylinder substrate needs 6 params but was passed "+geometryParams.length);
        		logger.severe(errMess);
        		throw new RuntimeException(errMess);
        	}
        	
        	return new SquashyInflammationSubstrate(((Double)geometryParams[0]).doubleValue(), 
        			((Double)geometryParams[1]).doubleValue(), ((Integer)geometryParams[2]).intValue(),
        			((Integer)geometryParams[3]).intValue(), ((Integer)geometryParams[4]).intValue(), 
        			((Double)geometryParams[5]).doubleValue(), simParams);
        	
        }
        else if(substrateType==CYL_1_FACET){
        	if(geometryParams.length!=3){
        		String errMess = new String("Facetted cylinder substrate requires geometry params array of length three but was passed "+geometryParams.length);
        		throw new LoggedException(errMess);
        	}
        	
        	return new FacetCylinderSubstrate(((Double)geometryParams[0]).doubleValue(), 
        			((Double)geometryParams[1]).doubleValue(), ((Integer)geometryParams[2]).intValue(), simParams);

        }
        else if(substrateType==TRI_PLY_MESH){
        	if(geometryParams.length!=3){
        		String errMess = new String("triangular mesh substrate requires 3 params, but passed "+geometryParams.length);
        		throw new LoggedException(errMess);
        	}
        	
        	return new MeshSubstrate((String)geometryParams[0], 
        			((Double)geometryParams[1]).doubleValue(), ((Double)geometryParams[2]).doubleValue(),simParams);
        }
        else if(substrateType==CYL_1_PERC){
        	if(geometryParams.length!=7){
        		String errMess = new String ("Percolation cylinder substrate needs 6 params but was passed "+geometryParams.length);
        		logger.severe(errMess);
        		throw new RuntimeException(errMess);
        	}
        	
        	return new PercolationSubstrate(((Double)geometryParams[0]).doubleValue(), 
        			((Double)geometryParams[1]).doubleValue(), ((Integer)geometryParams[2]).intValue(),
        			((Integer)geometryParams[3]).intValue(), ((Integer)geometryParams[4]).intValue(), 
        			((Double)geometryParams[5]).doubleValue(), ((Integer)geometryParams[6]).intValue(), simParams);
        	
        }
        else if(substrateType==CYL_1_MYELIN){
        	if(geometryParams.length!=7){
        		throw new RuntimeException("Parallel myelinated cylinder substrate requires 7 parameters but was passed "+geometryParams.length);
        	}
        	
        	return new ParallelCylinderSubstrate(((Double)geometryParams[0]).doubleValue(), ((Double)geometryParams[1]).doubleValue(), 
        			((Double)geometryParams[2]).doubleValue(), ((Double)geometryParams[3]).doubleValue(), 
        			((Double)geometryParams[4]).doubleValue(), ((Double)geometryParams[5]).doubleValue(),
        			((Integer)geometryParams[6]).intValue(), simParams);

        }
        else if(substrateType==CYL_2_FIXED){
        	if(geometryParams.length!=2){
        		throw new LoggedException("Fixed radius crossing cylinder substrate requires either 2 or 3 parameters, but was passed "+geometryParams.length);
        	}
        	
        	return new CrossingCylinderSubstrate(((Double)geometryParams[0]).doubleValue(), 
        									((Double)geometryParams[1]).doubleValue(), simParams);
        }
        else if(substrateType==EMPTY){
            return new EmptySubstrate(simParams);
        }
        else{
            String errMess=new String("unrecognised substrate type "+substrateType);
            logger.severe(errMess);
            throw new RuntimeException(errMess);
        }    
        
    }

 
    
    
    
    public static void main(String[] args) {
    }
}
