package simulation.measurement;

import imaging.Scheme;
import imaging.SchemeV0;

import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.util.logging.Logger;

import misc.LoggedException;
import numerics.MTRandom;
import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import simulation.dynamics.Walker;
import simulation.geometry.Substrate;
import tools.CL_Initializer;
import data.DataSource;
import data.OutputManager;
import data.StandardTestFunctions;
import data.VoxelOrderDataSource;

public class PGSEscan implements SyntheticScan {

    /** logging object */
    private final Logger logger= Logger.getLogger(this.getClass().getName());
    
    /** dimensionality of the system. */
    private static final int D=DiffusionSimulation.D;
    
    /** number of walkers */
    private final int N_walkers;
    
    /** the array of walkers from the simulation */
    private Walker[] walker;
    
    /** the gradient directions array */
    private final double[][] gradient;
    
    /** the SNR for the readings */
    private final double snr;
    
    /** the std dev  corresponding to the snr */
    private double noiseDev;
    
    /** random number generator */
    private final MTRandom twister=new MTRandom((1736401757<<32)|(1625498295));
    
    /** number of measurments */
    private final int numMeas;
    
    /** gyromagnetic constant */
    private static final double GAMMA= SchemeV0.GAMMA;
    
    /** gradient strength in each measurement */
    private final double[] G;
    
    /** duration of gradient block in each measurement */
    private final double[] delta;
    
    /** diffusion time in each measurement */
    private final double[] DELTA;

    /** time increment associated with each timestep */
    private final double dt;
    
    /** start of first gradient block (seconds) */
    private final double[] firstBlockStart;
    
    /** end of first gradient block (seconds) */
    private final double[] firstBlockEnd;
    
    /** start of second gradient block (seconds) */
    private final double[] secondBlockStart;
    
    /** end of second gradient block (seconds) */
    private final double[] secondBlockEnd;
    
    /** the substrate being scanned */
    private final Substrate substrate;
    
    /** PGSE scan type */
    private final int scanType=ScanFactory.PGSE_SCAN;
    

    
    /** constructor. 
     * 
     * @param simParams the simualtion parameters
     * @param imPars the imaging parameters
     * @param walker the set of walkers from the simulation
     */
    protected PGSEscan(SimulationParams simParams, Scheme scheme, Substrate substrate, Walker[] walker){
        
    	this.numMeas= scheme.numMeasurements();
    	
    	this.G= new double[numMeas];
    	this.delta= new double[numMeas];
    	this.DELTA= new double[numMeas];
    	this.gradient= new double[numMeas][];
    	
    	this.firstBlockStart= new double[numMeas];
    	this.firstBlockEnd= new double[numMeas];
    	this.secondBlockStart= new double[numMeas];
    	this.secondBlockEnd= new double[numMeas];
    	
    	for(int i=0; i<numMeas; i++){
    		G[i]=scheme.getModG(i);
    		delta[i]=scheme.getDelta(i);
    		DELTA[i]=scheme.getDELTA(i);
    		gradient[i]= scheme.getQ(i);
    		
    		// make sure directions are normalised
    		double gradLen=0.0;
    		for(int j=0; j<gradient[i].length; j++){
    			gradLen+=gradient[i][j]*gradient[i][j];
    		}
    		gradLen=Math.sqrt(gradLen);
    		if(gradLen>0.0){
    			for(int j=0; j<gradient[i].length; j++){
    				gradient[i][j]/=gradLen;
    			}
    		}
    		
            if(delta[i]>=DELTA[i]){
                logger.warning("WARNING! gradient block duration in measurement "+i+" (delta="
                        +delta[i]+"s) in synthetic scan is longer than interval between blocks ("
                        +DELTA[i]+"s)");
            }
    	}

        this.snr= CL_Initializer.SNR;
    	
        this.substrate= substrate;
        
        this.N_walkers=simParams.getN_walkers();
        this.walker=walker;        
        
        this.dt= simParams.getDt();
        
        for(int i=0; i<numMeas; i++){
        	this.firstBlockStart[i] = scheme.getHalfP90(i);
	        this.firstBlockEnd[i] = this.firstBlockStart[i] + delta[i];
	        
	        this.secondBlockStart[i] = this.firstBlockStart[i] + DELTA[i];
	        this.secondBlockEnd[i] = this.secondBlockStart[i] + delta[i];
        }        
    }
    
    /** checks if a given walker is in the central voxel or not
     * 
     * @param walker the walker to check
     * @return true or false
     */
    /*private boolean inVoxel(Walker walker){
        
        boolean inVoxel=true;
        
        for(int i=0; i<D; i++){
            if((walker.r[i]<voxelMin)||(walker.r[i]>voxelMax)){
                inVoxel=false;
                break;
            }
        }
        
        /*return inVoxel;
    }*/
    
    
    
    /**
     * returns the phase shift due a particular gradient at a given position
     * takes account of gradient blocks that do not coincide with timesteps.
     * this should fix the bias due to poor b-value approximation with very 
     * long timesteps.
     * 
     * @param r position
     * @param t current time
     * @param dir index of gradient
     * @param tLast the last time the scan was queried by the walker
     * 
     * @return GAMMA * del * G[dir].r
     */
    public double getPhaseShift(Walker walker, double t, int dir, double tLast){
    	
    	double sgn=1.0;

    	// if before first block
    	if(t<firstBlockStart[dir]){
    		return 0.0;
    	}
    	
    	// if between blocks
    	if((t>=firstBlockEnd[dir])&&(t<secondBlockStart[dir])){
    		if((t>=firstBlockEnd[dir])&&(tLast<firstBlockEnd[dir])){
    			// the black ended between this call and the last one
    			// so need to calculate the partial contribution
    			double dt= firstBlockEnd[dir]-tLast;
    	    	double Gdotr=0.0;
    	    	for(int i=0; i<D; i++){
    	    		Gdotr+=gradient[dir][i]*(walker.r[i]-walker.r0[i]);
    	    	}
    	    	double rawPhase= GAMMA*(G[dir])*(Gdotr)*dt;
    	    	
    	    	return  sgn*(rawPhase);
    			
    		}
    		
    		return 0.0;
    	}
    	
    	// if after second block
    	if(t>=secondBlockEnd[dir]){
    		if((t>=secondBlockEnd[dir])&&(tLast<secondBlockEnd[dir])){
    			// the black ended between this call and the last one
    			// so need to calculate the partial contribution
    			double dt= secondBlockEnd[dir]-tLast;
    	    	double Gdotr=0.0;
    	    	for(int i=0; i<D; i++){
    	    		Gdotr+=gradient[dir][i]*(walker.r[i]-walker.r0[i]);
    	    	}
    	    	double rawPhase= GAMMA*(G[dir])*(Gdotr)*dt;
    	    	
    	    	return sgn*(rawPhase);
    			
    		}

    		
    		return 0.0;
    	}

    	// flip the sign of the phase change if in the second block
    	if(t>=secondBlockStart[dir]){
    		sgn=-1.0;
    	}
        
		if((t>=secondBlockStart[dir])&&(tLast<secondBlockStart[dir])){
			// the black ended between this call and the last one
			// so need to calculate the partial contribution
			double dt= t-secondBlockStart[dir];
	    	double Gdotr=0.0;
	    	for(int i=0; i<D; i++){
	    		Gdotr+=gradient[dir][i]*(walker.r[i]-walker.r0[i]);
	    	}
	    	double rawPhase= GAMMA*(G[dir])*(Gdotr)*dt;
	    	
	    	return sgn*(rawPhase);
			
		}

		
    	// return phase shift for walker
    	double Gdotr=0.0;
    	for(int i=0; i<D; i++){
    		Gdotr+=gradient[dir][i]*(walker.r[i]-walker.r0[i]);
    	}
    	
    	double retVal= GAMMA*(G[dir])*(Gdotr)*dt;
    	return  sgn*(retVal);
    	
    }

    /**
     * returns the phase shift due a particular gradient at a given position
     * takes account of gradient blocks that do not coincide with timesteps.
     * this should fix the bias due to poor b-value approximation with very 
     * long timesteps.
     * 
     * @param r position
     * @param t current time
     * @param tLast the last time the scan was queried by the walker
     * 
     * @return GAMMA * del * G.r
     */
    public static double getPhaseShift(double[] r, double t, double tLast, 
    		double firstStart, double firstEnd, double secondStart, double secondEnd,
    		double[] gradient, double G){
    	
    	double sgn=1.0;
    	
    	// if before first block
    	if(t<firstStart){
    		return 0.0;
    	}
    	
    	// if between blocks
    	if((t>=firstEnd)&&(t<secondStart)){
    		if((t>=firstEnd)&&(tLast<firstEnd)){
    			// the black ended between this call and the last one
    			// so need to calculate the partial contribution
    			double dt= firstEnd-tLast;
    	    	double Gdotr=0.0;
    	    	for(int i=0; i<D; i++){
    	    		Gdotr+=gradient[i]*r[i];
    	    	}
    	    	return sgn*GAMMA*(G)*(Gdotr)*dt; 
    			
    		}
    		
    		return 0.0;
    	}
    	
    	// if after second block
    	if(t>=secondEnd){
    		if((t>=secondEnd)&&(tLast<secondEnd)){
    			// the black ended between this call and the last one
    			// so need to calculate the partial contribution
    			double dt= secondEnd-tLast;
    	    	double Gdotr=0.0;
    	    	for(int i=0; i<D; i++){
    	    		Gdotr+=gradient[i]*r[i];
    	    	}
    	    	return sgn*GAMMA*(G)*(Gdotr)*dt; 
    			
    		}

    		
    		return 0.0;
    	}

    	// flip the sign of the phase change if in the second block
    	if(t>=secondStart){
    		sgn=-1.0;
    	}
        
		if((t>=secondStart)&&(tLast<secondStart)){
			// the black ended between this call and the last one
			// so need to calculate the partial contribution
			double dt= t-secondStart;
	    	double Gdotr=0.0;
	    	for(int i=0; i<D; i++){
	    		Gdotr+=gradient[i]*r[i];
	    	}
	    	return sgn*GAMMA*(G)*(Gdotr)*dt; 
			
		}

		
    	// return phase shift for walker
    	double Gdotr=0.0;
    	for(int i=0; i<D; i++){
    		Gdotr+=gradient[i]*r[i];
    	}
    	
    	double dt=t-tLast;
    	
    	double retVal=sgn*GAMMA*(G)*(Gdotr)*dt;
    	return  retVal;
    	
    }

    
    /**
     * returns the change in magnetisation for the given location
     * and time.
     * 
     * @param r location
     * @param t time
     * 
     * @return TODO: returns zero
     * 
     */
    public final double getMagnetisationChange(double[] r, double t){
    	return 0.0;
    }
    
    
    /**
     * returns the PGSE scan type
     * 
     * @return scanType
     */
    public final int getScanType(){
    	return scanType;
    }
    
    /**
     * gets the signals from the summations of the displacements in
     * the walkers.
     * 
     * @return array of signals, zeroth is unweighted, the rest 
     *         correspond to each gradient direction.
     * 
     */
    public double[] getSignals(){
        
        double[] signal=new double[numMeas];
        
        double Sreal;        
        double SintReal=0.0;
        double SextReal=0.0;
        
        int numIn=0;
        int numExt=0;
        
        double noiseTerm;
        

        // apply test function rotation matrix to gradients
        // (by default this is the identity)
        double[][] trans=StandardTestFunctions.getRotationMatrix().entries;
        for(int i=0; i<gradient.length; i++){
            double[] grad = new double[D];
            double modGrad=0.0;
            

            // calculate new gradient
            grad[0]= gradient[i][0]*trans[0][0] + gradient[i][1]*trans[0][1] + gradient[i][2]*trans[0][2];
            grad[1]= gradient[i][0]*trans[1][0] + gradient[i][1]*trans[1][1] + gradient[i][2]*trans[1][2];
            grad[2]= gradient[i][0]*trans[2][0] + gradient[i][1]*trans[2][1] + gradient[i][2]*trans[2][2];
                        
            modGrad = Math.sqrt(grad[0]*grad[0] + grad[1]*grad[1] + grad[2]*grad[2]);
            
            if(modGrad>0.0){
                grad[0]/=modGrad;
                grad[1]/=modGrad;
                grad[2]/=modGrad;
            }


            // replace old gradient
            gradient[i]=grad;
        }
        
        if(snr>0.0){
            this.noiseDev=((double)N_walkers)/snr;
        }
        
                
        // now calculate the remainder of the signals
        logger.info("generating "+gradient.length+" signals");
        for(int i=0; i<gradient.length; i++){
            
            // reset the signal sums
            Sreal=0.0;

            SintReal=0.0;
            
            SextReal=0.0;
            
            numIn=0;
            numExt=0;
            
            for(int j=0; j<walker.length; j++){
                
                double phi=walker[j].getPhaseShift(i);
                
                // add to sum of signals 
                Sreal+=Math.cos(phi);
               
                // check if the signals come from intra
                // or extra cellular compartments and
                // update the appropriate signal
                if(substrate.intracellular(walker[j])){
                    SintReal+=Math.cos(phi);
                    
                    numIn++;
                }
                else{
                    SextReal+=Math.cos(phi);
                    
                    numExt++;
                }
            }

            
            noiseTerm=0.0;
            
            // add noise to net signal
            if(snr>0.0){
                noiseTerm=twister.nextGaussian()*noiseDev;
            }
            Sreal+=noiseTerm;

            if(snr>0.0){
                noiseTerm=twister.nextGaussian()*noiseDev;              
            }
            
            // take modulus of real and imaginary parts to provide measured signal
            signal[i]=Math.sqrt(Sreal*Sreal + noiseTerm*noiseTerm);
            
            double signalInt= SintReal;
            double signalExt= SextReal;

            double b= (GAMMA*delta[i]*G[i])*(GAMMA*delta[i]*G[i])*(DELTA[i]-delta[i]/3);
            double Sfree= Math.exp(-b*CL_Initializer.DIFF_CONST);
            
            logger.info("signal = "+    signal[i]+"  intra= "+signalInt+
                    " extra= "+signalExt+" ("+numIn+" in, "+numExt+" out)"+" free = "+Sfree);

            //System.err.println("dir "+i+" del= "+delta[i]+" DEL= "+DELTA[i]+" |G|= "+G[i]+" Ghat= ("+gradient[i][0]+","+gradient[i][1]+","+gradient[i][2]+") sig= "+signal[i]);
            
        }

        FileWriter angDistWriter;
        int bins=50;
        int[] hist= new int[bins];
        
        double[] R= new double[D];
        for(int i=0; i<walker.length; i++){
            for(int j=0; j<D; j++){
                R[j]=walker[i].r[j]-walker[i].r0[j];
            }
            
            double theta= Math.atan2(R[1], R[0])+Math.PI;
            int bin= (int)((theta/(2.0*Math.PI))*bins);
            if(bin>=hist.length){
                bin=bin%hist.length;
            }
            //System.err.println("walker "+i+" theta= "+theta+" bin= "+bin);
            
            hist[bin]++;
        }
        
        
        try{
            angDistWriter= new FileWriter("angDist3.csv");
        }
        catch(IOException ioe){
            throw new LoggedException(ioe);
        }
        
        double binWidth= 2.0*Math.PI/bins;
        for(int i=0; i<hist.length; i++){
            double freq= (hist[i]/(walker.length*binWidth));
            double ang=i*(2.0*Math.PI/bins);
            
            try{
                angDistWriter.write(ang+","+freq+"\n");
            }
            catch(IOException ioe){
                throw new LoggedException(ioe);
            }
        }
        
        try{
            angDistWriter.flush();
            angDistWriter.close();
        }
        catch(IOException ioe){
            throw new LoggedException(ioe);
        }
        
        return signal;
    }
    
    
    /**
     * gets the signals from the summations of the displacements in
     * the walkers in a specified comnpartment. 
     * 
     * @param intra if true, return contributions from the intracellular
     *              compartment, otherwise return contributions from the 
     *              extracellular compartment
     * 
     * @return array of signals, zeroth is unweighted, the rest 
     *         correspond to each gradient direction from spins in 
     *         the specified compartment.
     * 
     */
    public double[] getCompartmentalSignals(boolean intra){
        
        double[] signal=new double[numMeas];
        
        double Sreal;        
        //double SintReal=0.0;
        //double SextReal=0.0;
        
        int numIn=0;
        int numExt=0;
        
        double noiseTerm;
        

        // apply test function rotation matrix to gradients
        // (by default this is the identity)
        double[][] trans=StandardTestFunctions.getRotationMatrix().entries;
        for(int i=0; i<gradient.length; i++){
            double[] grad = new double[D];
            double modGrad=0.0;
            

            // calculate new gradient
            grad[0]= gradient[i][0]*trans[0][0] + gradient[i][1]*trans[0][1] + gradient[i][2]*trans[0][2];
            grad[1]= gradient[i][0]*trans[1][0] + gradient[i][1]*trans[1][1] + gradient[i][2]*trans[1][2];
            grad[2]= gradient[i][0]*trans[2][0] + gradient[i][1]*trans[2][1] + gradient[i][2]*trans[2][2];
                        
            modGrad = Math.sqrt(grad[0]*grad[0] + grad[1]*grad[1] + grad[2]*grad[2]);
            
            if(modGrad>0.0){
                grad[0]/=modGrad;
                grad[1]/=modGrad;
                grad[2]/=modGrad;
            }


            // replace old gradient
            gradient[i]=grad;
        }
        
        if(snr>0.0){
            this.noiseDev=((double)N_walkers)/snr;
        }
        
                
        // now calculate the remainder of the signals
        logger.info("generating "+gradient.length+" signals");
        for(int i=0; i<gradient.length; i++){
            
            // reset the signal sums
            Sreal=0.0;

            //SintReal=0.0;
            
            //SextReal=0.0;
            
            numIn=0;
            numExt=0;
            
            FileWriter phaseWriter;
            int bins=100;
            
            double min=Double.MAX_VALUE;
            double max=-Double.MAX_VALUE;
                        
            for(int j=0; j<walker.length; j++){
                
                double phi=walker[j].getPhaseShift(i);
                
                if(phi<min){
                    min=phi;
                }
                
                if(phi>max){
                    max=phi;
                }
                
                // add to sum of signals 
                Sreal+=Math.cos(phi);
               
                // check if the signals come from intra
                // or extra cellular compartments and
                // update the appropriate signal
                if((substrate.intracellular(walker[j]))==intra){
                    Sreal+=Math.cos(phi);
                    
                    numIn++;
                }
            }

            
            noiseTerm=0.0;
            
            // add noise to net signal
            if(snr>0.0){
                noiseTerm=twister.nextGaussian()*noiseDev;
            }
            Sreal+=noiseTerm;

            if(snr>0.0){
                noiseTerm=twister.nextGaussian()*noiseDev;              
            }
            
            // take modulus of real and imaginary parts to provide measured signal
            signal[i]=Math.sqrt(Sreal*Sreal + noiseTerm*noiseTerm);
            
        }
        
        return signal;
    }
    
    
    
    

    /**
     * updates the scan variables in each timestep. (does 
     * nothing in PGSE sequence)
     * 
     * @param t the time in seconds
     */
    public final void update(int t){
    	    	
    }
    
    public int getNumMeasurements(){
    	return numMeas;
    }
    
    /**
     * main entrypoint. read trajectory file and synthesise
     * measurements using parameter ranges given on cammonadline.
     * 
     * TODO: current implementation is very restrictive, limited to
     * stadard PGSE sequence and (unconventional) parameter ranges 
     * that are parsed locally.
     * 
     * @param args
     */
    public static void main(String[] args) {

    	int D= DiffusionSimulation.D;
    	
    	double[] Glims= new double[2];
    	int Gincs=-1;
    	
    	double[] delLims= new double[2];
    	int delIncs=-1;
    	
    	double[] DELlims= new double[2];
    	int DELincs=-1;
    	
    	//boolean alchemy=false;
    	
    	
    	CL_Initializer.CL_init(args);
    	
    	for(int i=0; i<args.length; i++){
    		if(args[i].equals("-Grange")){
    			Glims[0]=Double.parseDouble(args[i+1]);
    			Glims[1]=Double.parseDouble(args[i+2]);
    			Gincs= Integer.parseInt(args[i+3]);
    			
    			CL_Initializer.markAsParsed(i, 4);
    		}
    		else if(args[i].equals("-delrange")){
    			delLims[0]=Double.parseDouble(args[i+1]);
    			delLims[1]=Double.parseDouble(args[i+2]);
    			delIncs= Integer.parseInt(args[i+3]);
    			
    			CL_Initializer.markAsParsed(i, 4);
    		}
    		else if(args[i].equals("-DELrange")){
    			DELlims[0]=Double.parseDouble(args[i+1]);
    			DELlims[1]=Double.parseDouble(args[i+2]);
    			DELincs= Integer.parseInt(args[i+3]);
    			
    			CL_Initializer.markAsParsed(i, 4);
    		}
    		//else if(args[i].equals("-makegoldstandard")){
    		//	alchemy=true;
    		//	
    		//	CL_Initializer.markAsParsed(i);
    		//}
    	}
    	
    	CL_Initializer.checkParsing(args);
    
    
    	CL_Initializer.initImagingScheme();
    
    	// imaging scheme
    	Scheme scheme=CL_Initializer.imPars;
    	
    	double[][] dir= new double[scheme.numMeasurements()][];
    	for(int i=0; i<scheme.numMeasurements(); i++){
    		dir[i]=scheme.getQ(i);
    		double modQ=scheme.getModQ(i);
    		
    		if(modQ>0.0){
    			for(int j=0; j<dir[i].length; j++){
    				dir[i][j]/=modQ;
    			}
    		}
    	}
    	
    	OutputManager om= new OutputManager();
    	
    	double[] G= new double[Gincs+1];
    	double[] DEL= new double[DELincs+1];
    	double[] del= new double[delIncs+1];
    	
    	// initialise the parameters arrays
    	G[0]=Glims[0];
    	G[Gincs]=Glims[1];
    	if(Gincs>0){
    		for(int i=1; i<Gincs; i++){
    			double inc = (Glims[1]-Glims[0])/Gincs;
    		
    			G[i]=Glims[0]+i*inc;
    		}
    	}
    	
    	del[0]=delLims[0];
    	del[delIncs]=delLims[1];
    	if(delIncs>0){
    		for(int i=1; i<delIncs; i++){
    			double inc = (delLims[1]-delLims[0])/delIncs;
    		
    			del[i]=delLims[0]+i*inc;
    	
    		}
    	}
    	
    	DEL[0]=DELlims[0];
    	DEL[DELincs]=DELlims[1];
    	if(DELincs>0){
    		for(int i=1; i<DELincs; i++){
    			double inc = (DELlims[1]-DELlims[0])/DELincs;
    		
    			DEL[i]=DELlims[0]+i*inc;
    		}
    	}

    	
    	// initialise input file
    	String trajFile = SimulationParams.trajFile;
    	
    	if(trajFile==null){
    		throw new LoggedException("trajectories filename has not been set.");
    	}

    	// input stream for trajectories file
    	DataInputStream trajectories;
    	
    	// data source for inputting gold standard
    	DataSource data=null;
    	
    	// stats on squared differences
    	double meanDiff=0.0;			// mean of sqaured difference
    	double stdDevDiff=0.0;;			// stdDev of squared difference
    	double meanSq=0.0;				// mean square squared-diff (used to calc stddev)
    	
    	// read trajfile header
    	double duration;         // duration of simulation 
    	double N;                // number of walkers
    	double T;                // number of time steps
    	double dt;               // duration of timestep
    	
    	int count=0;          // counts number fo terms in mean/stddev
    	
    	double[] Sreal= new double[scheme.numMeasurements()];         // single element array to store synthetic signal in
    	  	
    	//if(!alchemy){
    	//	data= new VoxelOrderDataSource(CL_Initializer.inputFile, 1, CL_Initializer.inputDataType);
    	//}
    	
    	for(int i=0; i<DEL.length; i++){           // loop over diffusion times
    		for(int j=0; j<del.length; j++){           // loop over block durations

    			System.err.println("i= "+i+" j= "+j);
    			
    			if(del[j]>DEL[i]){
    				continue;
    			}
    			
    			try{
    				trajectories= new DataInputStream(new FileInputStream(trajFile));
    				
    				// read header information
    				duration= trajectories.readDouble();
		    		if(duration<del[j]+DEL[i]){
		    			//throw new LoggedException("simulation duration of "+duration+" is too short for del= "
		    			//						+del[j]+" DEL= "+DEL[i]+" ("+(del[j]+DEL[i])+")");
		    			continue;
		    		}
		    		
		    		N= trajectories.readDouble();
		    		T= trajectories.readDouble();
		    		
		    		dt=duration/T;
		    	}
		    	catch(IOException ioe){
		    		throw new LoggedException(ioe);
		    	}
		    	
		    	double[][][] sum= new double[(int)N][D][2];
		    	
		    	double t1=0.0;
		    	double t2=del[j];
		    	double t3=DEL[i];
		    	double t4=DEL[i]+del[j];
		    	
		    	double[] r= new double[D];
		    	
		    	// read trajectories and assemble sums for each walker
		    	try{
		    		// loop over each timestep
		    		for(int k=0; k<(int)T; k++){
		    			// loop over each walker
		    			for(int n=0; n<(int)N; n++){
		    				// read time and walker number
		    				double t= trajectories.readDouble();
		    				int walkerNum= trajectories.readInt();
		    				
		    				// read walker position
		    				for(int m=0; m<D; m++){
		    					r[m]=trajectories.readDouble();

		    					// if in first block, add to 1st sum
			    				if((t>=t1)&&(t<t2)){
			    					sum[walkerNum][m][0]+=r[m];
			    				}
			    				// if in second block, add to 2nd sum
			    				else if((t>=t3)&&(t<t4)){
			    					sum[walkerNum][m][1]+=r[m];
			    				}
		    				}
		    				
		    				if(t>t4){
		    					break;
		    				}
		    			}
		    		}
		    		
		    		// now we've constructed the sums, we can generate the 
		    		// signals for each G and direction and read out
		    		for(int k=0; k<G.length; k++){
		    			
		    			for(int l=0; l<dir.length; l++){

		    				Sreal[l]=0.0;
		    				
		    				for(int n=0; n<sum.length; n++){
				    			double phi=0.0;
				    			
		    					for(int m=0; m<D; m++){
		    						phi+=dir[l][m]*(sum[n][m][1]-sum[n][m][0]);
		    					}
		    				
		    			
			    				// scale with constants and parameters
			    				phi *= SchemeV0.GAMMA*G[k]*dt;
		                    
			    				// add to sum of signals 
			    				Sreal[l]+=Math.cos(phi);
			    				//Simag+=Math.sin(phi);
		    				}
		    			
				    		
		                    count++;
		    			}
                        om.output(Sreal);
		    		}


		    		
		    		trajectories.close();
		    		
		    	}
		    	catch(IOException ioe){
		    		throw new LoggedException(ioe);
		    	}
    		}
    	}
    	
    	//if(alchemy){
			om.close();
    	//}
    	
    	meanDiff/=count;
    	meanSq/=count;
    	
    	stdDevDiff=Math.sqrt(meanSq-meanDiff*meanDiff);
    	
    	System.err.println("mean = "+meanSq+"\tstdDev = "+stdDevDiff);
    	
    }

}
