package tractography;

import data.*;
import imaging.*;
import misc.*;
import numerics.*;


/**
 * Defines image for tractography from tensor data. 
 * 
 *
 *
 * @version $Id: BallStickTractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class BallStickTractographyImage extends PD_TractographyImage {

    
    /**
     * Constructs an image directly from an array of vectors.
     * This allows direct access from matlab.
     *
     * @param vectors The array of vectors (stick orientations).
     *
     *
     * @param npds The array of numbers of directions
     *
     * @param dataDims array of data dimensions {xDataDim,
     * yDataDim, zDataDim}.
     *
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim,
     * yVoxelDim, zVoxelDim}.
     */
    public BallStickTractographyImage(Vector3D[][][][] vecs, int[][][] npds, double[] voxelDims) {
	
        super(vecs, voxelDims, 1);

        for(int i=0; i<xDataDim; i++) {
            for(int j=0; j<yDataDim; j++) {
                for(int k=0; k<zDataDim; k++) {
                    numPDs[i][j][k] = npds[i][j][k];
                }
            }
        }

        computeIsotropicMask();
    }


    /**
     * Constructs an image from the data sources. 
     *
     * @param dataSource sources for the model parameters.
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     *
     */
    public BallStickTractographyImage(VoxelOrderDataSource dataSource, int[] dataDims, 
				       double[] voxelDims) {
	

	super(dataDims, voxelDims, 1);

        for (int k = 0; k < zDataDim; k++) {
            for (int j = 0; j < yDataDim; j++) {
                for (int i = 0; i < xDataDim; i++) {
                    
                    double[] voxel = dataSource.nextVoxel();

		    numPDs[i][j][k] = (voxel[0] < 0.0) ? 0 : 1;

		    vectors[i][j][k] = new Vector3D[] {new Vector3D(voxel[4], voxel[5], voxel[6])};
                }
            }
        }
        
		
	computeIsotropicMask();

    }


    /**
     * Copy constructor. Does not copy cached or publically mutable data.
     *
     *
     */
    protected BallStickTractographyImage(BallStickTractographyImage im) {
	super(im);
    }    
    
 
   
   
    /**
     * Gets an image from the data file. 
     * If <code>anisMapFile</code> is not <code>null</code>, it is read and used 
     * for isotropic masking.
     * 
     *
     * @param inputFile the data file.
     * @param dataType the data type of the data file and <code>anisMapFile</code>.
     * @param maxPDs the maximum number of PDs in a voxel.
     * @param anisMap the anisotropy map, which is used to create the tract mask.
     * May be <code>null</code> if not required.
     * @param anisThresh threshold for the anisotropy in the computation of the tract mask.
     * 
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * 
     */
    public static final BallStickTractographyImage getBallStickTractographyImage(String inputFile, 
                                                                     String dataType,  
                                                                     double[][][] anisMap, 
                                                                     double anisThresh, 
                                                                     int[] dataDims, 
                                                                     double[] voxelDims) {


        int numComponents = 7;

	VoxelOrderDataSource dataSource = new VoxelOrderDataSource(inputFile, numComponents, dataType);
     
        BallStickTractographyImage image = new BallStickTractographyImage(dataSource, dataDims, voxelDims);

        if (anisMap != null) {
            image.computeIsotropicMask(anisMap, anisThresh);
        }

        return image;	    

    }

    
}
