package tractography;

import data.*;
import imaging.*;
import misc.*;
import numerics.*;

import java.util.Random;


/**
 * Bayesian tractography image using Dirac priors for "nuisance parameters", 
 * see Friman et al, TMI 25:965-978 (2006). 
 *
 * @version $Id: BayesDiracTractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class BayesDiracTractographyImage extends TractographyImage {

   
    protected final float[][][][] data;
    protected final Scheme ip;

    protected final BayesDiracRandomizer randomizer;


    // need an enum for randomization type

    /**
     * Matlab constructor.
     *
     * @param 
     * @param 
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * 
     */
    public BayesDiracTractographyImage(float[][][][] data, double[] voxelDims, int[][][] numPDs, 
					  Scheme scheme, DataModel model, int pointSetInd, Random ran ) { 
	
	super(new int[] {data.length, data[0].length, data[0][0].length}, voxelDims, 1);


	ip = scheme;
	this.data = data;

	switch (model) {
	    
	case CYL_SYMM_DT: 
	    randomizer = new BayesDiracCylSymmTensorRandomizer(data, ip, pointSetInd, ran);
	    break;

	case BALL_STICK: 
	    randomizer = new BayesDiracBallStickRandomizer(data, ip, pointSetInd, ran);
	    break;

	default:
	    throw new LoggedException("Unsupported data model " + model);
	}

	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    this.numPDs[i][j][k] = numPDs[i][j][k];
		}
	    }
	}


	
	computeIsotropicMask();
    }



    /**
     * @return the princpal direction or directions in the voxel.
     */
    public Vector3D[] getPDs(int i, int j, int k) {

	vectors[i][j][k] = randomizer.getRandomizedPDs(i,j,k);

	return super.getPDs(i,j,k);
 
    }


    /**
     * @return the princpal direction or directions in the voxel, given an estimated local fiber orientation.
     * If the PDs in the voxel are not dependent on the fiber orientation, then this method returns the
     * same thing as getPDs(int, int, int).
     *
     */
    public Vector3D[] getPDs(int i, int j, int k, Vector3D fibreOrientation) {
	vectors[i][j][k] = randomizer.getRandomizedPDs(i,j,k, fibreOrientation);

	return super.getPDs(i,j,k);

    }


    public void setExternalPriors(PICoTractographyImage priorImage) {

	// check that image dimensions are the same

	int[] dataDims = priorImage.getDataDims();
	double[] voxelDims = priorImage.getVoxelDims();

	if (dataDims[0] != xDataDim || dataDims[1] != yDataDim || dataDims[2] != zDataDim) {
	    throw new LoggedException("External prior image has different data dimensions");
	}

	if (voxelDims[0] != xVoxelDim || voxelDims[1] != yVoxelDim || voxelDims[2] != zVoxelDim) {
	    throw new LoggedException("External prior image has different voxel dimensions");
	}

	randomizer.setExternalPriors(priorImage.randomizer);
    }



    /**
     * Gets an image from the data file. 
     * If <code>anisMapFile</code> is not <code>null</code>, it is read and used 
     * for isotropic masking.
     * 
     *
     * @param inputFile the data file.
     * @param dataType the data type of the data file.
     * @param numPDs array containing number of PDs in each voxel.
     * @param anisMap the anisotropy map, which is used to create the tract mask.
     * May be <code>null</code> if not required.
     * @param anisThresh threshold for the anisotropy in the computation of the tract mask.
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * 
     */
    public static final BayesDiracTractographyImage getBayesDiracTractographyImage(String inputFile, 
										   String dataType,
										   Scheme scheme,
										   DataModel dataModel,
										   int[][][] numPDs,
										   double[][][] anisMap,
										   double anisThresh,
										   int[] dataDims, 
										   double[] voxelDims,
										   int pointSetInd,
										   Random ran) {
	

	int numMeas = scheme.numMeasurements();
	
        VoxelOrderDataSource dataSource = new VoxelOrderDataSource(inputFile, numMeas, dataType);

	float[][][][] data = new float[dataDims[0]][dataDims[1]][dataDims[2]][numMeas];

	
	for (int k = 0; k < dataDims[2]; k++) {
	    for (int j = 0; j < dataDims[1]; j++) {
		for (int i = 0; i < dataDims[0]; i++) {

		    double[] voxel = dataSource.nextVoxel();
		    
		    for (int n = 0; n < numMeas; n++) {
			data[i][j][k][n] = (float)voxel[n];
		    }
		}
	    }
	}
				       
	BayesDiracTractographyImage image = new BayesDiracTractographyImage(data, voxelDims, numPDs, 
									    scheme, dataModel, pointSetInd, ran);
	
        if (anisMap != null) {
            image.computeIsotropicMask(anisMap, anisThresh);
        }
	
	return image;
    }

    
    /**
     * Sets the curvature prior concentration using a Watson distribution.
     *
     */
    public void setCurvePriorKappa(double k) {
	randomizer.setCurvatureK(k);
    }

    /**
     * Sets the curvature prior concentration to |v_i \dot v_{i-1}|^\gamma given the current
     * orientation v_i and the previous one v_{i-1}.
     *
     */
    public void setCurvePriorGamma(double g) {
	randomizer.setCurvatureG(g);
    }

}
