package tractography;

import misc.*;
import numerics.*;

import java.util.*;
import java.util.logging.*;

/**
 * Labels a seed image with the target that has the maximum connection probability to each seed.
 *
 * @version $Id: ConnectivitySegmentedImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class ConnectivitySegmentedImage {

    /**
     *
     * Logging object
     *
     */
    private static Logger logger = Logger.getLogger("camino.tractography.ConnectivitySegmentedImage");

    // voxel dims of seed space
    private final double xVoxelDim;
    private final double yVoxelDim;
    private final double zVoxelDim;


    // dimensions of seed space
    private final int xDataDim;
    private final int yDataDim;
    private final int zDataDim;

    private FreeFormROI roi = null;

    private Voxel[] seedList = null;

    private int numSeeds = 0;

    private short[][][] targets;

    private int minTargetIndex = 0;
    private int maxTargetIndex = 0;

    private int numTargets = 0;

    private boolean directional = false;

    // if directional, vector tells us forwards / backwards (or left / right, or whatever)
    private Vector3D forwards = null;

    private boolean countFirstEntry = true;

    // lazy initialized
    private short[][][][] labelledSeedImage = null;

    // streamline counts to associated label
    private int[][][][] targetSCImage; 

    private double[][][][] targetCPImage;

    TargetCP_Image[] seedTargetCP;

    /**
     * Initializes the image with the dimensions of the seed space.
     *
     *
     */
    public ConnectivitySegmentedImage(FreeFormROI seeds, short[][][] targets,
				      double xVoxelDim, double yVoxelDim, double zVoxelDim) {

	roi = seeds;

	seedList = roi.getVoxels();

	numSeeds = seedList.length;

	seedTargetCP = new TargetCP_Image[numSeeds];

	for (int s = 0; s < numSeeds; s++) {
	    seedTargetCP[s] = new TargetCP_Image(targets, xVoxelDim, yVoxelDim, zVoxelDim);
	}

	this.targets = targets;


	xDataDim = targets.length;
	yDataDim = targets[0].length;
	zDataDim = targets[0][0].length;

	this.xVoxelDim = xVoxelDim;
	this.yVoxelDim = yVoxelDim;
	this.zVoxelDim = zVoxelDim;


	for (int i = 0; i < xDataDim; i++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int k = 0; k < zDataDim; k++) {
		    if (targets[i][j][k] > 0 && targets[i][j][k] < minTargetIndex) {
			minTargetIndex = targets[i][j][k];
		    }
		    if (targets[i][j][k] > 0 && targets[i][j][k] > maxTargetIndex) {
			maxTargetIndex = targets[i][j][k];
		    }
		}
	    }
	}
	
	numTargets = maxTargetIndex - minTargetIndex + 1;


    }




    /**
     * apply some tracts to a specified seed point. The index corresponds to the ordering of
     *  seed points from this object's <code>FreeFormROI</code> roi, so seed point n is 
     * <code>roi.getVoxels()[n]</code>.
     *
     */
    public void processTracts(int seedIndex, TractCollection tracts) {

	// reset these
	labelledSeedImage = null;
	targetSCImage = null;
	targetCPImage = null;

	seedTargetCP[seedIndex].processTracts(tracts);


    }

    


    /**
     * @return a 4D image <code>im</code> where the seed voxels are labelled with the target 
     * ROI to which they have maximum connectivity. If the connectivity is directional, then
     * <code>im[0]</code> contains seed points labelled according to their connectivity to 
     * targets in the forward direction.
     *
     *
     */
    public short[][][][] getSegmentedSeeds() {

	int numVolumes = directional ? 2 : 1;
	
	if (labelledSeedImage == null) {

	    short[][][][] image = new short[numVolumes][xDataDim][yDataDim][zDataDim];
	    int[][][][] sc =  new int[numVolumes][xDataDim][yDataDim][zDataDim];
	    double[][][][] cp =  new double[numVolumes][xDataDim][yDataDim][zDataDim];

	    for (int s = 0; s < numSeeds; s++) {
		
		// target streamline counts for each seed point
		int[][] targetStreamlineCounts = seedTargetCP[s].getStreamlineCountsArray();
		
		double norm = (double)seedTargetCP[s].totalStreamlines();
		
		int x = seedList[s].x;
		int y = seedList[s].y;
		int z = seedList[s].z;
		
		for (int i = 0; i < numVolumes; i++) {
		    
		    int maxTargetSC = 0;

		    short maxTargetLabel = 0;
		    
		    for (int t = minTargetIndex; t < numTargets; t++) {
			if (targetStreamlineCounts[i][t] > maxTargetSC) {
			    maxTargetSC = targetStreamlineCounts[i][t];
			    maxTargetLabel = (short)t;
			}
		    }
		    
		    image[i][x][y][z] = maxTargetLabel;
		    sc[i][x][y][z] = maxTargetSC;
		    cp[i][x][y][z] = norm > 0.0 ? maxTargetSC / norm : 0.0;
		    
		}
	    }
	    
	    targetCPImage = cp;
	    targetSCImage = sc;
	    labelledSeedImage = image;
	    
	}
	
	short[][][][] defCopy = new short[numVolumes][xDataDim][yDataDim][zDataDim];
	    
	for (int n = 0; n < numVolumes; n++) {
	    for (int i = 0; i < xDataDim; i++) {
		for (int j = 0; j < yDataDim; j++) {
		    System.arraycopy(labelledSeedImage[n][i][j], 0, defCopy[n][i][j], 0, zDataDim);
		}
	    }
	}

	return defCopy;

    }


    /**
     * @return the seeds labelled with the connection probability to the target with the highest
     * connectivity to the seed. If a seed voxel s is labelled with target N, then the value of 
     * s returned from this image is the connection probability to target N.
     */
    public double[][][][] getMaxTargetCP() {

	if (targetCPImage == null) {
	    getSegmentedSeeds();
	}

	int numVolumes = targetCPImage.length;

	double[][][][] defCopy = new double[numVolumes][xDataDim][yDataDim][zDataDim];
	    
	for (int n = 0; n < numVolumes; n++) {
	    for (int i = 0; i < xDataDim; i++) {
		for (int j = 0; j < yDataDim; j++) {
		    System.arraycopy(targetCPImage[n][i][j], 0, defCopy[n][i][j], 0, zDataDim);
		}
	    }
	}

	return defCopy;

	
    }

    /**
     * @return the seeds labelled with the streamline count to the target with the highest
     * connectivity to the seed. 
     *
     */
    public int[][][][] getMaxTargetSC() {

	if (targetSCImage == null) {
	    getSegmentedSeeds();
	}

	int numVolumes = targetSCImage.length;

	int[][][][] defCopy = new int[numVolumes][xDataDim][yDataDim][zDataDim];
	    
	for (int n = 0; n < numVolumes; n++) {
	    for (int i = 0; i < xDataDim; i++) {
		for (int j = 0; j < yDataDim; j++) {
		    System.arraycopy(targetSCImage[n][i][j], 0, defCopy[n][i][j], 0, zDataDim);
		}
	    }
	}

	return defCopy;

	
    }



    /**
     * Call this method to split the streamlines in two at the seed point and produce 
     * separate connection probability maps from each. The vector <code>v</code> defines a "forward" 
     * direction and should be approximately tangential to the streamline path at the seed point.
     * <p> 
     * Two volumes will be returned from <code>getStreamlineCounts</code> 
     * and <code>getConnectionProbabilities</code>. The first volume is connection probabilities 
     * for streamline segments where the dot product of the streamline direction and v is greater 
     * than zero. The second volume is connection probabilities in the other direction. 
     * <p>
     * As an example, consider mapping the connectivity from the corpus callosum. We have streamlines
     * seeded along the mid-sagittal line, which proceed to the left and right hemispheres. We would
     * therefore use v = [-1, 0, 0]. Streamlines are then split at the seed point and the segments  
     * proceeding left (or in any direction with a negative x component) form the first connection 
     * probability volume. Those proceeding to the right (direction with a positive x component)
     * form the second connection probability volume. We can therefore get connection probability
     * maps to the left and right hemispheres from the same seed point. This example shows
     * the importance of choosing the direction v carefully. It is important that v is not 
     * perpendicular to the streamline direction at any seed point.
     * 
     *
     * @param v a "forward" direction. Should be approximately tangential to the streamline 
     * trajectory at the seed point.
     */
    public void setDirectional(Vector3D v) {
	forwards = v.normalized();
	directional = true;

	for (int s = 0; s < numSeeds; s++) {
	    seedTargetCP[s].setDirectional(forwards);
	}
    }



    /**
     * The default behaviour is to count the first entry of a streamline to a target zone. 
     * If this method is called with a <code>false</code> parameter, then streamlines are
     * allowed to connect to any number of target zones.
     *
     * @see #setDirectional(numerics.Vector3D)
     */
    public void setCountFirstEntry(boolean b) {
	countFirstEntry = b;
	
	for (int s = 0; s < numSeeds; s++) {
	    seedTargetCP[s].setCountFirstEntry(countFirstEntry);
	}
    }


}
