package tractography;

import numerics.*;

/**
 * Superclass for trackers that use Euler's method to track through the data. 
 * The interpolated data is not necessarily linearly interpolated. Concrete subclasses 
 * should provide an interpolator.
 *
 * @version $Id: EulerFibreTracker.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public abstract class EulerFibreTracker extends FibreTracker {

    private final double stepSize;
   
    /**
     * Concrete subclasses should assign an object to this.
     *
     */
    protected ImageInterpolator interpolator;


    /** Construct an EulerFibreTracker with additional options set.
     * @param data the dataset within which the tracking will take place.
     * @param stepLength the distance (in mm) to move before performing the next interpolation. 
     * @param ipThresh the minimum absolute value of the cosine of the
     * angle between the tracking directions after traversing one slice thickness. 
     * So if slice thickness is 2.5 mm, and step size is 0.25mm, then the curvature will be assessed 
     * every 10 steps. 
     *
     */
    public EulerFibreTracker(TractographyImage image, double stepLength, double ipThresh) {
	super(image, ipThresh);
	stepSize = stepLength;

	if (stepSize <= 0.0) {
	    throw new IllegalArgumentException("Can't track with step size " + stepSize);
	}
    }


    /**
     * @return the distance in mm between points of Tracts from this tracker.
     */ 
    public double stepSize() {
	return stepSize;
    }

    protected Vector3D[] getPDs(int i, int j, int k) {
	throw new UnsupportedOperationException("Interpolated trackers should take PDs from interpolator");
    }


    protected final Tract trackFromSeed(Point3D seedPos, int pdIndex, boolean direction) {

	Vector3D trackingDirection, previousBearing;
	
	Tract path = new Tract(200, 100.0);
	
	Point3D currentPos = seedPos;

	path.addPoint(seedPos, 0.0);

	if (!inBounds(seedPos)) {
	    return path;
	}

	int i = (int)(currentPos.x / xVoxelDim);
	int j = (int)(currentPos.y / yVoxelDim);
	int k = (int)(currentPos.z / zVoxelDim);

	if (isotropic[i][j][k]) {
	    return path;
	}

	trackingDirection = interpolator.getTrackingDirection(seedPos, pdIndex, direction);

	// Now track the rest of the way, always moving in direction that has the largest dot product
	// with the vector currentPos - previousPos

	// sometimes the tracking can get stuck in an infinite loop
	// As a safeguard, terminate tracts that exceed 1m 
	short pointsAdded = 1;
	final int maxPoints = (int)(1000.0 / stepSize);

	visitedVoxel[i][j][k] = 1;
	
	// check that we don't curve more than IP threshold in the course of the last voxel
	final int checkCurve = (int) (zVoxelDim / stepSize) > 0 ? (int) (zVoxelDim / stepSize) : 1;
	
	Vector3D checkCurveBearing = trackingDirection;
	
	while (true) {

	    currentPos = currentPos.displaced( trackingDirection.scaled(stepSize) );    

	    // Quit if we are out of bounds
	    if (!inBounds(currentPos)) {
		return path;
	    }

	    i = (int)(currentPos.x / xVoxelDim);
	    j = (int)(currentPos.y / yVoxelDim);
	    k = (int)(currentPos.z / zVoxelDim);
	    
	    if (isotropic[i][j][k]) {
		return path;
	    }

	    if (visitedVoxel[i][j][k] - pointsAdded > checkCurve) {
		return path;
	    }
	    else {
		visitedVoxel[i][j][k] = pointsAdded;
	    }

	    previousBearing = trackingDirection;
	    
	    trackingDirection = interpolator.getTrackingDirection(currentPos, previousBearing);
 
	    if (pointsAdded % checkCurve == 0) {
		// no Math.abs here, since we want to know if we're going backwards
		if ( checkCurveBearing.dot(trackingDirection) < ipThreshold) {
		    path.addPoint(currentPos, stepSize);
		    return path;
		}
		checkCurveBearing = trackingDirection;
	    }

	    // record position
	    path.addPoint(currentPos, stepSize);

	    pointsAdded++;

	    if (pointsAdded == maxPoints) {
		return path;
	    }

	}

	
    }




    
}
