package tractography;

import data.*;
import misc.LoggedException;
import numerics.*;

import java.util.Random;

/**
 * Tractography image with all three eigenvectors of spherical PDF(s) specified in each voxel. 
 * These can be from the diffusion tensor or from other models. 
 *
 *
 * @version $Id: PICoTractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class PICoTractographyImage extends TractographyImage {

  
    // vectors[i][j][k] contains {e1, e2, e3...} for each tensor / spherical function.

    protected final double[][][][] pdfParams;

    protected final PICoRandomizer randomizer;

    protected final PICoPDF pdf;


    /**
     * Constructs an image from the data sources. The data source should be the output of PICoPDFs.
     *
     * @param dataSource sourcs for the data, in the format of the picopdfs program.
     * @param maxPDs the maximum number of PDs in a voxel.
     * @param pdf PICo PDF type.
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param ran a source of random numbers.
     *
     */
    public PICoTractographyImage(VoxelOrderDataSource dataSource, int maxPDs, PICoPDF pdf,
                                 int[] dataDims, double[] voxelDims, Random ran) {
	
        super(dataDims, voxelDims, 3);

	this.pdf = pdf;
      
        pdfParams = new double[xDataDim][yDataDim][zDataDim][];

	int paramsPerPD = pdf.numParams;
        
	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
                    double[] voxel = dataSource.nextVoxel();

                    numPDs[i][j][k] = (int)voxel[0];

                    if (numPDs[i][j][k] < 0 || numPDs[i][j][k] > maxPDs) {
                        throw new LoggedException("Invalid number of components, " + numPDs[i][j][k] + ", in input data. " + 
                                                  "Check -inputmodel, PICo PDF and -numpds options.");
                    }

                    vectors[i][j][k] = new Vector3D[3 * numPDs[i][j][k]];

                    pdfParams[i][j][k] = new double[numPDs[i][j][k]*paramsPerPD];

                    for (int p = 0; p < numPDs[i][j][k]; p++) {
                        int start = 2 + p * (10 + paramsPerPD);

                        vectors[i][j][k][3*p] = 
                            new Vector3D(voxel[start],voxel[start+1],voxel[start+2]);

                        vectors[i][j][k][3*p+1] = 
                            new Vector3D(voxel[start+3],voxel[start+4],voxel[start+5]);

                        vectors[i][j][k][3*p+2] = 
                            new Vector3D(voxel[start+6],voxel[start+7],voxel[start+8]);
                        
                        for (int a = 0; a < paramsPerPD; a++) {
                            pdfParams[i][j][k][paramsPerPD*p + a] = voxel[start + 9 + a];
                        }
                    }
                    
                    
                }
            }
        }

        // default mask just segments background
        computeIsotropicMask();

	randomizer = pdf.getRandomizer(this, ran);

    }


    /**
     *  
     * @param vectors vector data in the order {e1, e2, e3}.
     *
     * @param pdfParams the array of PICo parameters.
     *
     * @param pdf the PICoPDF corresponding to the pdfParams.
     *
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * 
     */
    protected PICoTractographyImage(Vector3D[][][][] vectors, double[][][][] pdfParams, PICoPDF pdf,
				    int[] dataDims, double[] voxelDims, Random ran) {

	super(vectors, voxelDims, 3);

        this.pdfParams = pdfParams;

	this.pdf = pdf;
	
	randomizer = pdf.getRandomizer(this, ran);
    }



    /**
     * Copy constructor. 
     *
     * @param ran the random number generator for the newly constructed image.
     */
    protected PICoTractographyImage(PICoTractographyImage im, Random ran) {
	super(new int[] {im.xDataDim, im.yDataDim, im.zDataDim}, 
	      new double[] {im.xVoxelDim, im.yVoxelDim, im.zVoxelDim}, 3);

        pdfParams = new double[xDataDim][yDataDim][zDataDim][];

	pdf = im.pdf;

	randomizer = pdf.getRandomizer(this, ran);

	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
	
                    numPDs[i][j][k] = im.numPDs[i][j][k];

		    
		    vectors[i][j][k] = new Vector3D[3*numPDs[i][j][k]];

		    for (int v = 0; v < 3*numPDs[i][j][k]; v++) {
			vectors[i][j][k][v] = im.vectors[i][j][k][v];
		    }
		  
                    
                    int numPICoParams = im.pdfParams[i][j][k].length;
                    pdfParams[i][j][k] = new double[numPICoParams];
			
                    for (int p = 0; p < numPICoParams; p++) {
                        pdfParams[i][j][k][p] = im.pdfParams[i][j][k][p];
                        
                    }
                
		}
	    }
            
	}
	
	// default mask just segments background
	computeIsotropicMask();
	
    }

 
    /**
     * Gets the eigenvectors of the PDF for the voxel in the order  <code>{e1, e2, e3, e1...}</code>. 
     *
     */
    protected Vector3D[] getEigenvectors(int i, int j, int k) {
	return vectors[i][j][k];
    }


    /**
     * @return the PICo PDF params for this voxel.
     */
    protected double[] getPICoPDFParams(int i, int j, int k) {
        return pdfParams[i][j][k];
    }


    /**
     * Gets a new sample from each PICo PDF in this voxel.
     */
    public Vector3D[] getPDs(int i, int j, int k) {
	return randomizer.getRandomizedPDs(i,j,k);
    }


    /**
     * Gets an image from the data file. 
     * If <code>anisMapFile</code> is not <code>null</code>, it is read and used 
     * for isotropic masking.
     * 
     *
     * @param inputFile the data file.
     * @param dataType the data type of the data file and <code>anisMapFile</code>.
     * @param maxPDs the maximum number of PDs in a voxel.
     * @param pdf PICo PDF type.
     * @param anisMap the anisotropy map, which is used to create the tract mask.
     * May be <code>null</code> if not required.
     * @param anisThresh threshold for the anisotropy in the computation of the tract mask.
     * 
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * 
     */
    public static final PICoTractographyImage getPICoTractographyImage(String inputFile, 
								       String dataType, int maxPDs, 
								       PICoPDF pdf,
								       double[][][] anisMap, 
								       double anisThresh, 
								       int[] dataDims, 
								       double[] voxelDims,
								       Random ran) {

	int paramsPerPD = pdf.numParams;
        
        // components per voxel in the data
        int numComponents = 1 + maxPDs * (10 + paramsPerPD);

        VoxelOrderDataSource dataSource = new VoxelOrderDataSource(inputFile, numComponents, dataType);
	       
        PICoTractographyImage image = new PICoTractographyImage(dataSource, maxPDs, pdf, 
                                                                dataDims, voxelDims, ran);

        if (anisMap != null) {
            image.computeIsotropicMask(anisMap, anisThresh);
        }

        return image;	    

    }





    
}
