package tractography;

import numerics.Point3D;

/**
 * Interface for a tractography region of interest. The ROI should be defined in voxel space,
 * so that you may ask is (voxel v) within the ROI? It should also define seed points for 
 * tracking. Ordinarily these seed points would be the centre of each voxel in the ROI, but
 * that is not required. A voxel is defined as being in the ROI if there is at least one seed
 * point somewhere within the voxel.
 * <p>
 * An ROI is a compound entity, for example an Analyze image may contain several regions with 
 * distinct integer intensities.
 *
 * @author Philip Cook
 * @version $Id: RegionOfInterest.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 */
public interface RegionOfInterest {



    /**
     * Tests to see if a specific voxel is in this region.
     *
     * @param x the x index of the voxel.
     * @param y the y index of the voxel.
     * @param z the z index of the voxel.
     * @return true if the voxel is within the region bounds; false otherwise.
     */
    public boolean containsVoxel(int x, int y, int z);

    
    
    /**
     * @return All the voxels in the ROI. The order must agree with <code>getSeedPoints()</code>,
     * so that the same array index corresponds to the voxel containing that seed. This list will
     * be degenerate if there are multiple seed points within a voxel.
     */
    public Voxel[] getVoxels();


    /**
     * Tests if a point is within a voxel that contains one or more seed points.
     *
     * @param point the point to test.
     * @return true if the point is within a voxel containing a seed ; false otherwise.
     */
    public boolean containsMMPoint(Point3D point);


    /**
     * @return All the seed points for this region.
     */
    public Point3D[] getSeedPoints();


    /**
     * @return the number of distinct regions in this ROI.
     * 
     */
    public int numberOfRegions();


    /**
     * Get a specific ROI.
     *
     * @param index the index of the required region, 
     * where <code>-1 < index < numberOfRegions() </code>.
     */
    public RegionOfInterest getRegion(int index);


}
