package tractography;

import data.*;
import imaging.*;
import inverters.*;
import misc.DT;
import numerics.*;

import java.util.Random;

/**
 * Each call to #getPDs(int, int, int) returns a new bootstrap estimate of the PDs. 
 * The bootstrap data is inverted using the specified one and two tensor index. 
 *
 *
 * @version $Id: RepBS_DT_TractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class RepBS_DT_TractographyImage extends BS_DT_TractographyImage {
 
    private final float[][][][][] data;

    private final int numRepeats; // number of repeats, ie number of times each measurement is made
 
  


    /**
     *
     * @param data the raw bootstrap data, order [x][y][z][repeat][measurement].
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param vc the voxel classification.
     * @param indices the inversion indices for the two-tensor and tensor reconstruction, in that order.
     * @param imPars the imaging scheme for the data.
     * @param r a source of random numbers. The java.util.Random class is not recommended, use
     * tools.MTRandom instead.
     *
     * @see apps.ModelFit
     */
    public RepBS_DT_TractographyImage(float[][][][][] data, double[] voxelDims, int[][][] vc, 
					 ModelIndex[] indices, Scheme imPars, Random r) {

	super(new int[] {data.length, data[0].length, data[0][0].length}, voxelDims, vc, 
	      indices, imPars, r);

	this.data = data;

	numRepeats = data[0][0][0].length;
        
        computeIsotropicMask();
    }


     
       
    /**
     * @return the next bootstrap sample of data.
     *
     */
    protected double[] nextBootstrapSample(int i, int j, int k) {

	double[] sample = new double[numMeasurements];


	for (int m = 0; m < numMeasurements; m++) {
            
            int r = ran.nextInt(numRepeats);

	    sample[m] = data[i][j][k][r][m];
	}

	return sample;
    }


    /**
     * Gets an image from the data file. 
     * If <code>anisMapFile</code> is not <code>null</code>, it is read and used 
     * for isotropic masking.
     * 
     *
     * @param inputFiles the data files.
     * @param dataType the data type of the data file and <code>anisMapFile</code>.
     * @param imPars the imaging scheme for the data.
     * @param inversionIndex the numerical index of the tensor inversion.
     * @param vc the voxel classification for this image. If <code>null</code>, each voxel is assumed to contain
     * one PD. 
     * @param anisMap the anisotropy map, which is used to create the isotropic mask.
     * May be <code>null</code> if not required.
     * @param anisThresh threshold for the anisotropy in the computation of the tract mask.
     * 
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param ran a source of random numbers.
     */
    public static final RepBS_DT_TractographyImage getBS_TractographyImage(String[] inputFiles, 
									    String dataType, 
									    Scheme imPars,
									    ModelIndex[] indices,
									    int[][][] vc,
									    double[][][] anisMap, 
									    double anisThresh, 
									    int[] dataDims, 
									    double[] voxelDims, 
									    Random ran) {
        
	int xDataDim = dataDims[0];
	int yDataDim = dataDims[1];
	int zDataDim = dataDims[2];
	
	int numRepeats = inputFiles.length;
	
	int numMeasurements = imPars.numMeasurements();

	float[][][][][] data = new float[xDataDim][yDataDim][zDataDim][numRepeats][numMeasurements];
	
	for (int r = 0; r < numRepeats; r++) {

	    VoxelOrderDataSource component = new VoxelOrderDataSource(inputFiles[r], numMeasurements, dataType);

	    for (int k = 0; k < zDataDim; k++) {
		for (int j = 0; j < yDataDim; j++) {
		    for (int i = 0; i < xDataDim; i++) {

			double[] d = component.nextVoxel();

			for (int c = 0; c < numMeasurements; c++) {
			    data[i][j][k][r][c] = (float)d[c];
			}
		    }
		}
	    }
	}

	int[][][] voxClass = vc;

	if (voxClass == null) {
	    voxClass = new int[xDataDim][yDataDim][zDataDim];
	}


	RepBS_DT_TractographyImage image = new RepBS_DT_TractographyImage(data, voxelDims, voxClass, 
									  indices, imPars, ran);

	
        if (anisMap != null) {
            image.computeIsotropicMask(anisMap, anisThresh);
        }

        return image;	    

    }
   


}
