package tractography;

import data.*;
import imaging.*;
import inverters.*;
import misc.DT;
import numerics.*;

import java.util.Random;

/**
 * Each call to #getPDs(int, int, int) returns a new bootstrap estimate of the PDs. 
 * The bootstrap data is inverted using the specified one and two tensor index. 
 *
 *
 * @version $Id: WildBS_DT_TractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class WildBS_DT_TractographyImage extends BS_DT_TractographyImage {
 
    private final float[][][][] data;

    private final WildBS_DT_DataSynth dataSynth;


    /**
     *
     * @param data raw diffuion-weighted data.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param vc the voxel classification.
     * @param imPars the imaging scheme for the data.
     * @param r a source of random numbers. The java.util.Random class is not recommended, use
     * tools.MTRandom instead.
     *
     *
     */
    public WildBS_DT_TractographyImage(float[][][][] data, double[] voxelDims, int[][][] vc, 
				       Scheme imPars, Random r) {

	super(new int[] {data.length, data[0].length, data[0][0].length}, 
	      voxelDims, vc, new ModelIndex[] {ModelIndex.LDT}, imPars, r);

	this.data = data;

	dataSynth = new WildBS_DT_DataSynth(null, imPars, -1, r);

        computeIsotropicMask();
    }


  
    
 
    /**
     * @return the next bootstrap sample of data.
     *
     */
    protected double[] nextBootstrapSample(int i, int j, int k) {
	dataSynth.setSourceData(data[i][j][k]);
	
	return dataSynth.nextVoxel();
    }



    /**
     * Gets an image from the data file. 
     * If <code>anisMapFile</code> is not <code>null</code>, it is read and used 
     * for isotropic masking.
     * 
     *
     * @param inputFile the data file.
     * @param dataType the data type of the data file and <code>anisMapFile</code>.
     * @param imPars the imaging scheme for the data.
     * @param vc the voxel classification for this image. This class only supports one PD per voxel. 
     * This volume should be used to distinguish brain from background.
     * @param anisMap the anisotropy map, which is used to create the isotropic mask.
     * May be <code>null</code> if not required.
     * @param anisThresh threshold for the anisotropy in the computation of the tract mask.
     * 
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * 
     */
    public static final WildBS_DT_TractographyImage getBS_TractographyImage(String inputFile, 
									    String dataType, 
									    Scheme imPars,
									    int[][][] vc,
									    double[][][] anisMap, 
									    double anisThresh, 
									    int[] dataDims, 
									    double[] voxelDims,
									    Random ran) {

      
	int xDataDim = dataDims[0];
	int yDataDim = dataDims[1];
	int zDataDim = dataDims[2];
        
        // components per voxel in the data
        int numComponents = imPars.numMeasurements();

        VoxelOrderDataSource dataSource = new VoxelOrderDataSource(inputFile, numComponents, dataType);

	float[][][][] data = new float[dataDims[0]][dataDims[1]][dataDims[2]][numComponents];


	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    data[i][j][k] = new float[numComponents];

		    double[] tmp = dataSource.nextVoxel();

		    for (int c = 0; c < numComponents; c++) {
			data[i][j][k][c] = (float)tmp[c];
		    }
		}
	    }
	}


	int[][][] voxClass = vc;

	if (voxClass == null) {
	    voxClass = new int[xDataDim][yDataDim][zDataDim];
	}


	WildBS_DT_TractographyImage image = new WildBS_DT_TractographyImage(data, voxelDims, voxClass, 
									    imPars, ran);
	
   
	
        if (anisMap != null) {
            image.computeIsotropicMask(anisMap, anisThresh);
        }

        return image;	    

    }




}
