package edu.jhu.ece.iacl.algorithms.manual_label.staple;

/**
 * 
 * @author John Bogovic
 * @date 5/31/2008
 * 
 * Structure for storing and managing performance levels for STAPLE
 */


public class PerformanceLevel {
	
	protected float[][][] pl;
	protected double[][][] dpl;
	/*
	 * First dimension stores information for each rater.
	 * The remaining is an NxN matrix where N is the number of labels
	 * and the i-j-k^th element characterizes the probability that the 
	 * ith rater will decide label k is present when label j is the truth. 
	 */
	
	public PerformanceLevel(){	
	}
	
	public PerformanceLevel(int numlabels, int numraters){
		pl=new float[numraters][numlabels][numlabels];
	}
	public PerformanceLevel(int numlabels,int numraters,boolean isdouble){
		dpl=new double[numraters][numlabels][numlabels];
	}
	
	public float[][][] getPerformance(){
		return pl;
	}
	public double[][][] getPerformanceD(){
		return dpl;
	}
	
	public float get(int i, int j, int k){
		return pl[i][j][k];
	}
	public double getD(int i, int j, int k){
		return dpl[i][j][k];
	}
	
	public void set(int i, int j, int k, float f){
		pl[i][j][k]=f;
	}
	public void set(int i, int j, int k, double f){
		dpl[i][j][k]=f;
	}
	
	/**
	 * Initialize all performance levels the the same value.
	 * @param The value with which to initialize all peformance level values.  A number ~=1 (e.g. 0.99999) is suggested. 
	 */
	public void initialize(float init){
		for(int i=0; i<pl.length; i++){
			for(int j=0; j<pl[0].length; j++){
				for(int k=0; k<pl[0][0].length; k++){
					pl[i][j][k]=init;
				}
			}
		}
	}
	public void initialize(double init){
		for(int i=0; i<dpl.length; i++){
			for(int j=0; j<dpl[0].length; j++){
				for(int k=0; k<dpl[0][0].length; k++){
					dpl[i][j][k]=init;
				}
			}
		}
	}
	public void initialize2(float init){
		float initc = (1-init)/pl[0].length;
		for(int i=0; i<pl.length; i++){
			for(int j=0; j<pl[0].length; j++){
				for(int k=0; k<pl[0][0].length; k++){
					if(j==k){
						pl[i][j][k]=init;
					}else{
						pl[i][j][k]=initc;
					}
				}
			}
		}
	}
	public void initialize2(double init){
		double initc = (1-init)/dpl[0].length;
		for(int i=0; i<dpl.length; i++){
			for(int j=0; j<dpl[0].length; j++){
				for(int k=0; k<dpl[0][0].length; k++){
					if(j==k){
						dpl[i][j][k]=init;
					}else{
						dpl[i][j][k]=initc;
					}
				}
			}
		}
	}
	
	public void clear(){
		if(pl!=null){
			clearF();
		}else if(dpl!=null){
			clearD();
		}
	}
	public void clearF(){
		for(int i=0; i<pl.length; i++){
			for(int j=0; j<pl[0].length; j++){
				for(int k=0; k<pl[0][0].length; k++){
					pl[i][j][k]=0;
				}
			}
		}
	}
	
	public void clearD(){
		for(int i=0; i<dpl.length; i++){
			for(int j=0; j<dpl[0].length; j++){
				for(int k=0; k<dpl[0][0].length; k++){
					dpl[i][j][k]=0;
				}
			}
		}
	}
	
	public void divideByTots(int col, float tot){
		for(int i=0; i<pl.length; i++){
			for(int j=0; j<pl[0][0].length; j++){
				pl[i][j][col]=pl[i][j][col]/tot;
				if(pl[i][j][col]>1){
					reinitColumn(col,0.9999f);
				}
			}
		}
	}
	public void divideByTots(int col, double tot){
		for(int i=0; i<dpl.length; i++){
			for(int j=0; j<dpl[0][0].length; j++){
				
//				if(dpl[i][j][col]/tot > 1){
//					System.out.println("Prob.  Divided: "+
//							dpl[i][j][col] +" by " + tot);
//				}
				
				dpl[i][j][col]=dpl[i][j][col]/tot;
			}
		}
	}
	
	/**
	 * Reinitializes column if something messed up happens
	 */
	private void reinitColumn(int col, double init){
		double initc = (1-init)/dpl[0].length;
		for(int i=0; i<dpl.length; i++){
			for(int j=0; j<dpl[0].length; j++){
				if(j==col){
					dpl[i][j][col]=init;
				}else{
					dpl[i][j][col]=initc;
				}
			}
		}
	}
	
	/**
	 * Reinitializes column if something messed up happens
	 */
	private void reinitColumn(int col, float init){
		float initc = (1-init)/pl[0].length;
		for(int i=0; i<pl.length; i++){
			for(int j=0; j<pl[0].length; j++){
				if(j==col){
					pl[i][j][col]=init;
				}else{
					pl[i][j][col]=initc;
				}
			}
		}
	}
	
	/**
	 * Compute normalized trace- 
	 * The sum of the traces of each rater's confusion matrix normalized
	 * by the number of labels and the number of raters
	 * @return Normalized trace
	 */
	public float normalizedTrace(){
		if(pl!=null){
			float out=0;
			for(int i=0; i<pl.length; i++){
				for(int j=0; j<pl[0].length; j++){
					out+=pl[i][j][j];
				}
			}
			out=out/((pl.length)*(pl[0].length));
			return out;
		}else if(dpl!=null){
			float out=0;
			for(int i=0; i<dpl.length; i++){
				for(int j=0; j<dpl[0].length; j++){
					out+=dpl[i][j][j];
				}
			}
			out=out/((dpl.length)*(dpl[0].length));
			return out;
		}else{
			return -1;
		}
	}
	
	public String toString(){
		if(pl!=null){
			String sep = "********************";
			String out = ""+sep+"\n";
			for(int i=0; i<pl.length; i++){
				out = out+ "Rater " + (i+1) + "\n";	
				for(int j=0; j<pl[0].length; j++){
					for(int k=0; k<pl[0][0].length; k++){
						out = out+ "\t" + pl[i][j][k];
					}
					out = out+"\n";
				}
				out = out+sep+"\n";
			}
			return out;
		}else if(dpl!=null){
			String sep = "********************";
			String out = ""+sep+"\n";
			for(int i=0; i<dpl.length; i++){
				out = out+ "Rater " + (i+1) + "\n";	
				for(int j=0; j<dpl[0].length; j++){
					for(int k=0; k<dpl[0][0].length; k++){
						out = out+ "\t" + dpl[i][j][k];
					}
					out = out+"\n";
				}
				out = out+sep+"\n";
			}
			return out;
		}else{
			return null;
		}
		
		
	}
	

}
