package edu.jhu.ece.iacl.plugins.dti;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import edu.jhmi.rad.medic.utilities.MedicUtil;
import edu.jhu.bme.smile.commons.textfiles.TextFileReader;
import edu.jhu.ece.iacl.algorithms.dti.EstimateTensorLLMSE;
import edu.jhu.ece.iacl.algorithms.dti.EstimateTensorLikelihood;
import edu.jhu.ece.iacl.io.CubicVolumeReaderWriter;
import edu.jhu.ece.iacl.io.FileExtensionFilter;
import edu.jhu.ece.iacl.io.ModelImageReaderWriter;
import edu.jhu.ece.iacl.io.StringReaderWriter;
import edu.jhu.ece.iacl.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.*;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.plugins.dti.DWIThreeTensorEstCaminoFileCollection.TensorEstimationWrapper;
import edu.jhu.ece.iacl.structures.image.ImageData;
import edu.jhu.ece.iacl.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.structures.image.ImageDataUByte;
import edu.jhu.ece.iacl.utility.FileUtil;
import gov.nih.mipav.model.structures.ModelImage;

public class DWITensorEstLikelihood extends ProcessingAlgorithm{ 

	/****************************************************
	 * Input Parameters 
	 ****************************************************/
	private ParamFileCollection DWdata4D; 		// SLAB-enabled Imaging Data
	private ParamFileCollection Mask3D;			// SLAB-enabled Binary mask to indicate computation volume
	private ParamFile bvaluesTable;		// .b file with a list of b-values
	private ParamFile kSpaceAvgTable;		// .txt file with a list of k-space averages per volume relative to the reported NF
	private ParamFile gradsTable;		// .grad or .dpf file with a list of gradient directions
	private ParamFileCollection noiseField; 
	private ParamFloat noiseLevel;

	/****************************************************
	 * Output Parameters
	 ****************************************************/
	private ParamFileCollection tensorVolume;	// SLAB-enabled A 4D volume with one tensor estimated per pixel
	private ParamFileCollection noiseFieldVolume; // refined noise field estimate after tensor estimation
	private ParamFileCollection referenceVolume; // refined reference estimate after tensor estimation 

	private static final String rcsid =
		"$Id: DWITensorEstLikelihood.java,v 1.1 2009/03/27 01:28:44 bennett Exp $";
	private static final String cvsversion =
		"$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");

	protected void createInputParameters(ParamCollection inputParams) {

		/****************************************************
		 * Step 1. Set Plugin Information 
		 ****************************************************/
		inputParams.setName("Tensor Estimation: Likelihood");
		inputParams.setLabel("TensorEst:Likelihood");	
		inputParams.setCategory("Modeling.Diffusion");
		inputParams.setPackage("IACL");
		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("http://sites.google.com/site/jhupami/");
		info.add(new AlgorithmAuthor("Bennett Landman","landman@jhu.edu","http://sites.google.com/site/bennettlandman/"));
		info.setDescription("Log-linear minium mean squared error tensor estimation.");
		info.setAffiliation("Johns Hopkins University, Department of Biomedical Engineering");
		info.add(new Citation("  B. A. Landman, P-L. Bazin, and J. L. Prince. Diffusion Tensor Estimation by Maximizing Rician Likelihood, In Proceedings of the 2007 International Conference on Computer Vision Workshop on Mathematical Methods in Biomedical Image Analysis, Rio de Janeiro, Brazil, October 2007."));
		info.setVersion(revnum);	

		/****************************************************
		 * Step 2. Add input parameters to control system 
		 ****************************************************/
		inputParams.add(DWdata4D=new ParamFileCollection("DWI and Reference Image(s) Data (4D)",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions)));
		inputParams.add(gradsTable=new ParamFile("Table of diffusion weighting directions",new FileExtensionFilter(new String[]{"grad","dpf"})));
		inputParams.add(bvaluesTable=new ParamFile("Table of b-values",new FileExtensionFilter(new String[]{"b"})));
		inputParams.add(kSpaceAvgTable=new ParamFile("Table of k-Space Average",new FileExtensionFilter(new String[]{"txt","lst"})));
		inputParams.add(Mask3D=new ParamFileCollection("Mask Volume to Determine Region of Tensor Estimation (3D)",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions)));
		Mask3D.setMandatory(false); // Not required. A null mask will estimate all voxels.			
		inputParams.add(noiseField = new ParamFileCollection("Noise Field",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions)));
		noiseField.setMandatory(false);
		inputParams.add(noiseLevel = new ParamFloat("Constant Noise Level",0));
	}

	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Step 1. Add output parameters to control system 
		 ****************************************************/
		tensorVolume = new ParamFileCollection("Tensor Estimate",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		tensorVolume.setName("Tensor (xx,xy,xz,yy,yz,zz)");
		outputParams.add(tensorVolume);		
		noiseFieldVolume = new ParamFileCollection("Noise Field Estimate",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		noiseFieldVolume.setName("Noise Field");
		outputParams.add(noiseFieldVolume);		
		referenceVolume = new ParamFileCollection("Reference Estimate",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		referenceVolume.setName("Reference");
		outputParams.add(referenceVolume);	

	}

	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {		
		TensorEstimationWrapper wrapper=new TensorEstimationWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}

	protected class TensorEstimationWrapper extends AbstractCalculation {

		protected void execute() {
			/****************************************************
			 * Step 1. Indicate that the plugin has started.
			 * 		 	Tip: Use limited System.out.println statements
			 * 			to allow end users to monitor the status of
			 * 			your program and report potential problems/bugs
			 * 			along with information that will allow you to 
			 * 			know when the bug happened.  
			 ****************************************************/
			System.out.println("DWITensorEstLLMSE: Start");
			//		MedicUtil.setQuiet(false);

			/****************************************************
			 * Step 2. Loop over input slabs
			 ****************************************************/
			List<File> dwList = DWdata4D.getValue();
			List<File> maskList = Mask3D.getValue();
			List<File> noiseFieldList =noiseField.getValue();
			CubicVolumeReaderWriter rw  = CubicVolumeReaderWriter.getInstance();
			ArrayList<File> outTensorVols = new ArrayList<File>();
			ArrayList<File> outNFVols = new ArrayList<File>();
			ArrayList<File> outRefVols = new ArrayList<File>();
			for(int jSlab=0;jSlab<dwList.size();jSlab++) {
				/****************************************************
				 * Step 2. Parse the input data 
				 ****************************************************/
				ImageData dwd=rw.read(dwList.get(jSlab));

				String sourceName = dwd.getName();
				ImageDataFloat DWFloat=new ImageDataFloat(dwd);
				dwd.dispose();
				dwd=null;


				ImageData maskVol=null;
				if(maskList!=null)
					if(maskList.size()>jSlab)
						if(maskList.get(jSlab)!=null)
							maskVol=rw.read(maskList.get(jSlab));

				byte [][][]mask=null;
				if(maskVol!=null) {
					ImageDataUByte maskByte = new ImageDataUByte (maskVol);
					mask = maskByte.toArray3d();
					maskByte.dispose();
					maskByte=null;
					maskVol.dispose();
					maskVol=null;
				}

				float [][]bs=null;		
				TextFileReader text = new TextFileReader(bvaluesTable.getValue());
				try {
					bs = text.parseFloatFile();
				} catch (IOException e) 
				{
					throw new RuntimeException("LLMSE: Unable to parse b-file");
				}

				float [][]grads=null;
				text = new TextFileReader(gradsTable.getValue());
				try {
					grads  = text.parseFloatFile();
				} catch (IOException e) { 

					throw new RuntimeException("LLMSE: Unable to parse grad-file");
				}

				float [][]Kavg=null;		
				text = new TextFileReader(kSpaceAvgTable.getValue());
				try {
					Kavg = text.parseFloatFile();
				} catch (IOException e) 
				{
					throw new RuntimeException("LLMSE: Unable to parse b-file");
				}

				System.out.println("Kspace-avgs:"+Kavg.length);
				double []kSpaceAvgs = new double[Kavg.length];
				for(int i=0;i<Kavg.length;i++)
					kSpaceAvgs[i]=Kavg[i][0];

				/****************************************************
				 * Step 2b. Load the noise field 
				 ****************************************************/
				float[][][] noiseField = new float[DWFloat.getRows()][DWFloat.getCols()][DWFloat.getSlices()];
				System.out.println("Noise Field size: "+DWFloat.getRows()+"x"+DWFloat.getCols()+"x"+DWFloat.getSlices());
				if(noiseFieldList.size()>jSlab) {
					// A noise field was specified 
					ImageData nf=rw.read(noiseFieldList.get(jSlab));
					for(int i=0;i<noiseField.length;i++)
						for(int j=0;j<noiseField[0].length;j++)
							for(int k=0;k<noiseField[0][0].length;k++)
								noiseField[i][j][k]=nf.getFloat(i,j,k);
					nf.dispose();
				} else {
					// use a constant noise file
					for(int i=0;i<noiseField.length;i++)
						for(int j=0;j<noiseField[0].length;j++)
							for(int k=0;k<noiseField[0][0].length;k++)
								noiseField[i][j][k]=noiseLevel.getFloat();
				}


				/****************************************************
				 * Step 3. Perform limited error checking 
				 ****************************************************/
				// If there are 4 columns in the gradient table, remove the 1st column (indecies)
				if(grads[0].length==4) {
					float [][]g2 = new float[grads.length][3];
					for(int i=0;i<grads.length;i++) 
						for(int j=0;j<3;j++)
							g2[i][j]=grads[i][j+1];
					grads=g2;
				}

				if(grads[0].length!=3)
					throw new RuntimeException("LLMSE: Invalid gradient table. Must have 3 or 4 columns.");
				if(bs[0].length!=1)
					throw new RuntimeException("LLMSE: Invalid b-value table. Must have 1 column.");
				float []bval = new float[bs.length];
				for(int i=0;i<bval.length;i++)
					bval[i]=bs[i][0];

				/****************************************************
				 * Step 4. Run the core algorithm. Note that this program 
				 * 		   has NO knowledge of the MIPAV data structure and 
				 * 		   uses NO MIPAV specific components. This dramatic 
				 * 		   separation is a bit inefficient, but it dramatically 
				 * 		   lower the barriers to code re-use in other applications.  		  
				 ****************************************************/
				float[][][][] tensors = new float[DWFloat.getRows()][DWFloat.getCols()][DWFloat.getSlices()][6];			
				float[][][] reference = new float[DWFloat.getRows()][DWFloat.getCols()][DWFloat.getSlices()];;
				// tensors = EstimateTensorLLMSE.estimate(DWFloat.toArray4d(),bval,grads,mask,estOptions.getValue().compareToIgnoreCase("Yes")==0);
				EstimateTensorLikelihood.estimate(tensors,reference,noiseField,DWFloat.toArray4d(),bval,grads,mask,kSpaceAvgs);
				/****************************************************
				 * Step 5. Retrieve the image data and put it into a new
				 * 			data structure. Be sure to update the file information
				 * 			so that the resulting image has the correct
				 * 		 	field of view, resolution, etc.  
				 ****************************************************/		

				ImageData  out= (new ImageDataFloat(tensors));
				out.setHeader(DWFloat.getHeader());			
				tensors = null; //no longer needed			
				out.setName(sourceName+"_LikelihoodTensor");					
				File outputSlab = rw.write(out, getOutputDirectory());
				outTensorVols.add(outputSlab);
				out.dispose();
				out=null;		

				out= (new ImageDataFloat(noiseField));
				out.setHeader(DWFloat.getHeader());			
				noiseField = null; //no longer needed			
				out.setName(sourceName+"_LikelihoodNF");					
				outputSlab = rw.write(out, getOutputDirectory());
				outNFVols.add(outputSlab);
				out.dispose();
				out=null;

				out= (new ImageDataFloat(reference));
				out.setHeader(DWFloat.getHeader());			
				reference = null; //no longer needed			
				out.setName(sourceName+"_LikelihoodRef");					
				outputSlab = rw.write(out, getOutputDirectory());
				outRefVols.add(outputSlab);
				out.dispose();
				out=null;

				DWFloat.dispose();
				DWFloat=null;

			}
			tensorVolume.setValue(outTensorVols);
			noiseFieldVolume.setValue(outNFVols);
			referenceVolume.setValue(outRefVols);

			/****************************************************
			 * Step 6. Let the user know that your code is finished.  
			 ****************************************************/
			System.out.println("DWITensorEstLLMSE: FINISHED");
		}
	}
}
