package edu.jhu.ece.iacl.plugins.labeling.staple;

import edu.jhu.ece.iacl.algorithms.manual_label.staple.*;
import edu.jhu.ece.iacl.io.ArrayDoubleDxReaderWriter;
import edu.jhu.ece.iacl.io.ArrayDoubleReaderWriter;
import edu.jhu.ece.iacl.io.StringReaderWriter;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;

import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;

public class MedicAlgorithmSTAPLEgeneral extends ProcessingAlgorithm{
	
	private ParamFileCollection raterData;

	private ParamDouble eps;
	private ParamInteger maxiters;
	private ParamOption init;
	private ParamBoolean flaglabelprobs;
	
	//output Parameters
	ParamObject<String> pl;
	private ParamFileCollection probsOut;
	private ParamObject<double[][]> segOut;
	
	private static final String rcsid =
		"$Id: MedicAlgorithmSTAPLEgeneral.java,v 1.1 2008/10/24 14:44:53 bogovic Exp $";
	private static final String cvsversion =
		"$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("$Revision: 1.1 $", "").replace(" $", "");

	
	private ArrayList<Number> labels;

	protected void createInputParameters(ParamCollection inputParams) {

		inputParams.add(raterData=new ParamFileCollection("Rater Data"));
		inputParams.setName("staplegeneral");
		inputParams.setLabel("STAPLE General Array "+revnum);
		
		String[] initops = {"Performance"};
		inputParams.add(init=new ParamOption("Initialization Type", initops));
		
		inputParams.add(eps = new ParamDouble("Max Delta for Convergence"));
		eps.setValue(new Double(0.00001));
		inputParams.setCategory("IACL.Labeling.STAPLE");
		inputParams.add(maxiters=new ParamInteger("Max Iterations"));
		maxiters.setValue(new Integer(50));
		
		inputParams.add(flaglabelprobs = new ParamBoolean("Output Label Probabilities?"));
		flaglabelprobs.setValue(false);
		
		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("");
		info.setDescription("STAPLE - Simultaneous Truth and Performance Level Estimation applied to a general array");
		info.add(new AlgorithmAuthor("John Bogovic", "bogovic@jhu.edu", ""));
		info.setAffiliation("Johns Hopkins University, Department of Electrical Engineering");
		info.add(new Citation("Warfield SK, Zou KH, Wells WM, \"Simultaneous Truth and Performance Level Estimation (STAPLE): An Algorithm for the Validation of Image Segmentation\" IEEE TMI 2004; 23(7):903-21  "));	
		info.setVersion(revnum);	
		info.setLongDescription("Given a number of labelings, STAPLE returns membership functions of the Truth.  This implementation requires only" +
				"that the input objects be a collection of double arrays.  The ith element of all input arrays should be in correspondence.  Note that, because" +
				"there is no implied relationship between points in these arrays, no regularization can be applied");
		
	}
	protected void createOutputParameters(ParamCollection outputParams) {		
		probsOut = new ParamFileCollection("Truth Estimates");
		outputParams.add(probsOut);
		probsOut.setMandatory(false);
		
		outputParams.add(segOut = new ParamObject<double[][]>("Label Array", new ArrayDoubleReaderWriter()));
		
		pl = new ParamObject<String>("PerformanceLevels",new StringReaderWriter());
		outputParams.add(pl);
		
	}
	protected void execute(CalculationMonitor monitor) {
		
		STAPLEgeneral staple = new STAPLEgeneral(getData());
		staple.setmaxIters(maxiters.getInt());
		staple.setEps(eps.getDouble());
		staple.setInit(init.getValue());
		staple.iterate();
		
		if(flaglabelprobs.getValue()){
			probsOut.setValue(writeTruth(staple.getTruth(),labels));
		}
		
		segOut.setObject(staple.getHardSeg());
		segOut.setFileName("EstimatedTrueLabelArray");
		
		pl.setObject(staple.getPeformanceLevel().toString());
		pl.setFileName("Performance");
		
		System.out.println("FINISHED");
	}
	
	private ArrayList<double[][]> getData(){
		ArrayList<double[][]> dataout = new ArrayList<double[][]>(raterData.getValue().size());
		for(int i=0; i<raterData.size(); i++){
			dataout.add(ArrayDoubleReaderWriter.getInstance().read(raterData.getValue(i)));
		}
		return dataout;
	}
	
	private List<File> writeTruth(ArrayList<double[][]> truthdata,ArrayList<Number> labels){
		int i=0;
		ArrayList<File> filelist = new ArrayList<File>();
		ArrayDoubleDxReaderWriter writer = new ArrayDoubleDxReaderWriter();
		writer.setNumberFormat(new DecimalFormat("0.00000"));
		for(double[][] ithtruth : truthdata){
//			ArrayDoubleReaderWriter.getInstance().write(ithtruth, new File("/home/john/Desktop/StapleTest/general_dx/out/Truth"+labels.get(i)+".dx"));
			File f = new File(this.getOutputDirectory()+String.valueOf(File.separatorChar)+"Truth"+labels.get(i)+".dx");
			writer.write(ithtruth, f);
			i++;
		}
		return filelist;
	}
	
}

