package edu.jhu.ece.iacl.plugins.labeling.staple;

// $Version: $

import java.util.ArrayList;

import edu.jhu.ece.iacl.algorithms.manual_label.staple.*;
import edu.jhu.ece.iacl.io.StringReaderWriter;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.pipeline.parameter.ParamSurface;
import edu.jhu.ece.iacl.pipeline.parameter.ParamSurfaceCollection;
import edu.jhu.ece.iacl.structures.geom.EmbeddedSurface;

public class MedicAlgorithmSTAPLEsurface extends ProcessingAlgorithm{
	
	private ParamSurfaceCollection ratersurfs;
	ParamObject<String> pl;
	private ParamFloat eps;
	private ParamFloat beta;
	private ParamFloat sigma;
	private ParamInteger maxiters;
	private ParamInteger offset;
	private ParamOption init;
	private ParamBoolean memsFlag;
	private ParamBoolean maxmemsFlag;
	
	private ParamSurfaceCollection memsOut;
	private ParamSurface maxmems;
	private ParamSurface segout;


	
	private static final String rcsid =
		"$Id: MedicAlgorithmSTAPLEsurface.java,v 1.6 2009/07/19 12:22:55 bogovic Exp $";
	private static final String cvsversion =
		"$Revision: 1.6 $";
	private static final String revnum = cvsversion.replace("$Revision: 1.6 $", "").replace(" $", "").replace(" $", "");

	
	protected void createInputParameters(ParamCollection inputParams) {

		
		inputParams.setName("staplesurf");
		inputParams.setLabel("STAPLE Surface "+revnum);

		inputParams.add(ratersurfs=new ParamSurfaceCollection("Rater Surfaces"));
		String[] initops = {"Performance", "Truth"};
		inputParams.add(init=new ParamOption("Initialization Type", initops));
		inputParams.add(beta = new ParamFloat("Regularization",0.0f));
		inputParams.add(sigma = new ParamFloat("Kernel Size",2.0f));
		inputParams.add(eps = new ParamFloat("Max Delta for Convergence"));
		eps.setValue(new Float(0.00001));
		inputParams.setCategory("IACL.Labeling.STAPLE");
		inputParams.add(maxiters=new ParamInteger("Max Iterations"));
		maxiters.setValue(new Integer(30));
		inputParams.add(offset=new ParamInteger("Data Offset",0));
		inputParams.add(memsFlag = new ParamBoolean("Output Label Probability Surfaces?"));
		memsFlag.setValue(false);
		inputParams.add(maxmemsFlag = new ParamBoolean("Output Max Label Probability Surface?"));
		maxmemsFlag.setValue(false);
		
		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("");
		info.setDescription("STAPLE - Simultaneous Truth and Performance Level Estimation applied to surface labels");
		info.add(new AlgorithmAuthor("John Bogovic", "bogovic@jhu.edu", ""));
		info.setAffiliation("Johns Hopkins University, Department of Electrical Engineering");
		info.add(new Citation("Warfield SK, Zou KH, Wells WM, \"Simultaneous Truth and Performance Level Estimation (STAPLE): An Algorithm for the Validation of Image Segmentation\" IEEE TMI 2004; 23(7):903-21  "));	
		info.setVersion(revnum);	
		info.setLongDescription("Given a number of surface labelings of a particular strucute, STAPLE returns membership functions of the Truth.  This implementation requires that" +
				"each element of the input collection be a full surface object, and must therefore include all vertex/edge/face data.  For a more light-weight implementation, see " +
				"STAPLEgeneral.");
		
		
	}
	
	protected void createOutputParameters(ParamCollection outputParams) {	
		
		outputParams.add(segout = new ParamSurface("Labeled Surface"));
		
		outputParams.add(memsOut = new ParamSurfaceCollection("Label Probability Surfaces"));
		memsOut.setMandatory(false);
		
		outputParams.add(maxmems = new ParamSurface("Label Probability Surfaces"));
		maxmems.setMandatory(false);
		
		pl = new ParamObject<String>("PerformanceLevels",new StringReaderWriter());
		pl.setDescription("Returns a description of the performance levels for each rater in matrix form." +
				"The i-jth element represents the probability that label i was selected when label j was the truth.");
		outputParams.add(pl);
		
	}
	protected void execute(CalculationMonitor monitor) {
		
	

		/* A lighter weight implementation */
		
		//copy one of the input surfaces.
		EmbeddedSurface surfout = ratersurfs.getSurfaceList().get(0);
		
		STAPLEgeneral staple = new STAPLEgeneral(getSurfaceDataList(),offset.getInt());
		
		/*
		 * If there is neighbor interaction, inform staple about the 
		 * surface connectivity and weight of interaction
		 */
		if(beta.getFloat()>0){
			staple.setSurface(surfout);
			staple.setBeta(beta.getFloat());
			staple.setSigma(sigma.getFloat());
		}
		
		staple.setmaxIters(maxiters.getInt());
		staple.setEps(eps.getDouble());
		staple.setInit(init.getValue());
		staple.iterate();

		surfout.setVertexData(staple.getHardSeg());
		segout.setValue(surfout);
		segout.setFileName("HardSegmentation_"+this.toString());


		if(memsFlag.getValue()){
//			memsOut.setValue(staple.getTruth());
			ArrayList<double[][]> memdata = staple.getTruth();
			ArrayList<EmbeddedSurface> memsurfs = new ArrayList<EmbeddedSurface>();
			ArrayList<Number> labellist = staple.getLabels();
			int i =0;
			for(double[][] labmem : memdata){
				EmbeddedSurface s = new EmbeddedSurface(surfout);
				s.setName("LabelProbability_"+labellist.get(i).intValue());
				s.setVertexData(labmem);
				memsurfs.add(s);
				i++;
			}

			memsOut.setValue(memsurfs);
		}

		if(maxmemsFlag.getValue()){
			EmbeddedSurface s = new EmbeddedSurface(surfout);
			s.setName(ratersurfs.getSurfaceList().get(0).getName()+"_MaxLabelProbability");
			s.setVertexData(staple.getProbabilityMap());
			maxmems.setValue(s);
		}
		
		pl.setObject(staple.getPeformanceLevel().toString());
		pl.setFileName("Performance");

		System.out.println("FINISHED");
	}
	
	private ArrayList<double[][]> getSurfaceDataList(){
		System.out.println(ratersurfs);
		ArrayList<double[][]> datlist = new ArrayList<double[][]>();
		for(EmbeddedSurface surf : ratersurfs.getSurfaceList()){
			datlist.add(surf.getVertexData());
		}
		return datlist;
	}
}

