/**
 * 
 */
package edu.jhu.ece.iacl.plugins.utilities.roi;

import java.util.Arrays;

import edu.jhu.bme.smile.commons.math.StatisticsDouble;
import edu.jhu.ece.iacl.io.ArrayDoubleTxtReaderWriter;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.structures.image.ImageData;
import edu.jhu.ece.iacl.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.structures.image.ImageDataMipav;
import gov.nih.mipav.model.scripting.ParserException;
import gov.nih.mipav.model.scripting.parameters.ParameterFloat;
import gov.nih.mipav.model.scripting.parameters.ParameterInt;

/**
 * @author Bennett Landman (landman@jhu.edu)
 *
 */
public class AlgorithmROIStats extends ProcessingAlgorithm{
	ParamVolume volulmeParam;
	ParamVolume maskParam;
	ParamFloat resultMean;
	ParamFloat resultMedian;
	ParamFloat resultStd;
	ParamFloat resultMax;
	ParamFloat resultMin;
	ParamInteger resultVoxels;
	ParamObject<double[][]> resultHistogram;
	ParamInteger maskID;
	ParamInteger histogramBins;

	private static final String rcsid =
		"$Id: AlgorithmROIStats.java,v 1.3 2009/03/23 16:20:58 bogovic Exp $";
	private static final String cvsversion =
		"$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");
	
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(volulmeParam=new ParamVolume("Image Volume"));
		inputParams.add(maskParam=new ParamVolume("ROI Mask"));
		maskParam.setMandatory(false);

		inputParams.add(maskID=new ParamInteger("Mask ID",0,Integer.MAX_VALUE));
		inputParams.add(histogramBins=new ParamInteger("Histogram Bins",0,Integer.MAX_VALUE));
		histogramBins.setValue(256);
		
		/****************************************************
		 * Step 1. Set Plugin Information 
		 ****************************************************/
		inputParams.setName("roi_calc");
		inputParams.setLabel("ROI Calculator");
		inputParams.setCategory("IACL.Utilities.ROI");
		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("http://sites.google.com/a/jhu.edu/neuroweb/");
		info.add(new AlgorithmAuthor("Bennett Landman","landman@jhu.edu","http://sites.google.com/a/jhu.edu/neuroweb/"));
		info.setDescription("Calculate ROI-based statistics.");
		
		info.setAffiliation("Johns Hopkins University, Department of Biomedical Engineering");
		//info.add(new Citation(""));
		info.setVersion(revnum);		
	}
	
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(resultVoxels=new ParamInteger("# Voxels"));
		outputParams.add(resultMean=new ParamFloat("Mean"));
		outputParams.add(resultMedian=new ParamFloat("Median"));
		outputParams.add(resultStd=new ParamFloat("Std"));
		outputParams.add(resultMax=new ParamFloat("Max"));
		outputParams.add(resultMin=new ParamFloat("Min"));
		outputParams.add(resultHistogram=new ParamObject<double[][]>("Histogram", new ArrayDoubleTxtReaderWriter()));
//		ArrayDoubleTextReaderWriter t;
	}
	
	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		ImageDataMipav vol1=volulmeParam.getImageData();
		ImageDataMipav volMask=maskParam.getImageData();
		int IDval = maskID.getInt();
		
		int rows=vol1.getRows();
		int cols=vol1.getCols();
		int slices=vol1.getSlices();
		int components = vol1.getComponents();

		if(vol1.getRows()!=rows ||
				vol1.getCols()!=cols ||
				vol1.getSlices()!=slices ||
				vol1.getComponents()!=components)
			throw new RuntimeException("Volumes of unequal size cannot be combined.");
		double[] data;
		int count;
		ImageDataFloat resultVol=new ImageDataFloat(vol1.getName()+"_calc",rows,cols,slices);
		if(volMask!=null){ 	// if a mask is provided, consider only the portion of the image overlapping with the mask
			count=0;	
			for (int i = 0; i < rows; i++) {
				for (int j = 0; j < cols; j++) {
					for (int k = 0; k < slices; k++) {
						if(components>1) {
							for(int l=0;l<components;k++) {
								if(volMask.get(i, j, k, l).intValue()==IDval)
									count++;
							}
						} else {
							if(volMask.get(i, j, k).intValue()==IDval)
								count++;
						}
					}
				}
			}

			data =new double[count];
			count=0;
			for (int i = 0; i < rows; i++) {
				for (int j = 0; j < cols; j++) {
					for (int k = 0; k < slices; k++) {
						if(components>1) {
							for(int l=0;l<components;k++) {

								if(volMask.get(i, j, k, l).intValue()==IDval) {
									data[count] =vol1.get(i, j, k, l).doubleValue(); 
									count++;
								}
							}
						} else {
							if(volMask.get(i, j, k).intValue()==IDval) {
								data[count] =vol1.get(i, j, k).doubleValue(); 
								count++;
							}
						}
					}
				}
			}
		}else{	// if no mask is provided, consider the entire image
			count = rows*cols*slices*components;
			data =new double[count];
			count=0;
			for (int i = 0; i < rows; i++) {
				for (int j = 0; j < cols; j++) {
					for (int k = 0; k < slices; k++) {
						if(components>1) {
							for(int l=0;l<components;k++) {
								data[count] =vol1.get(i, j, k, l).doubleValue(); 
								count++;
							}
						} else {
							data[count] =vol1.get(i, j, k).doubleValue(); 
							count++;
						}
					}
				}
			}
		}
		
		resultVoxels.setValue(count);
		resultMean.setValue(StatisticsDouble.mean(data));
		resultStd.setValue(StatisticsDouble.std(data));
		resultMedian.setValue(StatisticsDouble.median(data));
		resultMax.setValue(StatisticsDouble.max(data));
		resultMin.setValue(StatisticsDouble.min(data));
		resultHistogram.setObject(StatisticsDouble.histogram(data, histogramBins.getInt()));
		resultHistogram.setFileName(vol1.getName()+"_histogram");
	}

}
