package edu.jhu.ece.iacl.plugins.utilities.volume;

import java.util.ArrayList;

import edu.jhu.ece.iacl.io.FileExtensionFilter;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.structures.image.ImageData;
import edu.jhu.ece.iacl.structures.image.ImageDataMipav;

import gov.nih.mipav.model.file.FileInfoBase;

public class MedicAlgorithmSplitImage extends ProcessingAlgorithm{
	
	//Input parameters
	ParamVolume volumein;
	ParamInteger subcomponents;
	ParamOption dimToSplit;
	
	ArrayList<ImageData> subVols;
	
	//Output parameters
	ParamVolumeCollection subImagesOut;
	
	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String rcsid =
		"$Id: MedicAlgorithmSplitImage.java,v 1.3 2008/12/19 21:10:25 bogovic Exp $";
	private static final String cvsversion =
		"$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");

	protected void createInputParameters(ParamCollection inputParams) {
		
		
		AlgorithmInformation info=getAlgorithmInformation();
		info.add(new AlgorithmAuthor("John Bogovic","bogovic@@jhu.edu",""));
		info.setDescription("Breaks an image slice-wise into N subcomponents (-1 as an input lets N = the number of slices)" +
				"Returns a number of volumes of the same dimensionality of the original, " +
				" each containing only a subset of the slices of the original's," +
				" and together containing all of the original slices.  " +
				"This is useful for parallelizing processes by breaking the image into pieces, which can " +
				"be recombined using 'Combine SubImages'");
		info.setAffiliation("Johns Hopkins University, Department of Electrical and Computer Engineering");				
		info.setVersion(revnum);	
		
		
		inputParams.add(volumein=new ParamVolume("Input Volume",null,-1,-1,-1,-1));
		inputParams.add(subcomponents=new ParamInteger("Number of Subcomponents"));
		String[] validdims = {"X","Y","Z","T"};
		inputParams.add(dimToSplit=new ParamOption("Dimension to split",validdims));
		inputParams.setCategory("IACL.Utilities.Volume");
		inputParams.setName("splitimage");
		inputParams.setLabel("Split Image Slices");
	}
	
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(subImagesOut=new ParamVolumeCollection("Output Volumes"));
	}
	
	protected void execute(CalculationMonitor monitor) {
		
		System.out.println("Splitting volume:" + volumein.getName());
		System.out.println("into: " + subcomponents.getInt() + " components");
		System.out.println("across dimension: " + dimToSplit.getValue());
		
		splitImage();
		subImagesOut.setValue(subVols);
	}
	
	private void splitImage(){
		
		// get the bounds used to break the image...
		ArrayList<Integer> bounds = chooseSubsetBounds();
		subVols = new ArrayList<ImageData>(subcomponents.getInt());
		
		FileInfoBase[] finfo = volumein.getModelImage().getFileInfo();
		
		int ithstart=0;
		int boundind=0;
		
		if(dimToSplit.getIndex()==0){
			
			for(int m=0; m<subcomponents.getInt(); m++){
				
				//Determine number of rows this image will have...
				int ithlength = bounds.get(boundind)-ithstart+1;
				
				//Initialize this Cubic volume to add...
				FileInfoBase[] fcat = new FileInfoBase[ithlength];
				for(int i=0; i<fcat.length; i++){
					fcat[i]=finfo[i];
				}
				ImageDataMipav nextvol = new ImageDataMipav(volumein.getName()+"_Rows_from"+ithstart,volumein.getImageData().getType(),ithlength,
						volumein.getImageData().getCols(), volumein.getImageData().getSlices(), volumein.getImageData().getComponents());

				nextvol.getModelImage().setFileInfo(fcat);
				
				//Set appropriate values for this volume...
				for(int i=ithstart;i<ithstart+ithlength;i++){
					for(int j=0;j<nextvol.getCols();j++){
						for(int k=0;k<nextvol.getSlices();k++){
							for(int l=0;l<nextvol.getComponents();l++){
								nextvol.set(i, j, k,l, volumein.getImageData().getDouble(i, j, k, l));
							}
						}
					}
				}   
				
				//Add it to the list.
				subVols.add(nextvol);
				
				//and move on
				ithstart = bounds.get(boundind)+1;
				boundind++;
			}
        	
		}else if(dimToSplit.getIndex()==1){
			for(int m=0; m<subcomponents.getInt(); m++){

				//Determine number of rows this image will have...
				int ithlength = bounds.get(boundind)-ithstart+1;

				//Initialize this Cubic volume to add...
				FileInfoBase[] fcat = new FileInfoBase[ithlength];
				for(int i=0; i<fcat.length; i++){
					fcat[i]=finfo[i];
				}
				ImageDataMipav nextvol = new ImageDataMipav(volumein.getName()+"_Cols_from"+ithstart,volumein.getImageData().getType(),volumein.getImageData().getRows(),
						ithlength, volumein.getImageData().getSlices(), volumein.getImageData().getComponents());

				nextvol.getModelImage().setFileInfo(fcat);

				//Set appropriate values for this volume...
				for(int i=0;i<nextvol.getRows();i++){
					for(int j=ithstart;j<ithstart+ithlength;j++){
						for(int k=0;k<nextvol.getSlices();k++){
							for(int l=0;l<nextvol.getComponents();l++){
								nextvol.set(i, j, k,l, volumein.getImageData().getDouble(i, j, k, l));
							}
						}
					}
				}   

				//Add it to the list.
				subVols.add(nextvol);

				//and move on
				ithstart = bounds.get(boundind)+1;
				boundind++;
			}
		}else if(dimToSplit.getIndex()==2){
			for(int m=0; m<subcomponents.getInt(); m++){

				//Determine number of rows this image will have...
				int ithlength = bounds.get(boundind)-ithstart+1;

				//Initialize this Cubic volume to add...
				FileInfoBase[] fcat = new FileInfoBase[ithlength];
				for(int i=0; i<fcat.length; i++){
					fcat[i]=finfo[i];
				}
				ImageDataMipav nextvol = new ImageDataMipav(volumein.getName()+"_Slices_from"+ithstart,volumein.getImageData().getType(),volumein.getImageData().getRows(),
						volumein.getImageData().getCols(), ithlength, volumein.getImageData().getComponents());

				nextvol.getModelImage().setFileInfo(fcat);

				//Set appropriate values for this volume...
				for(int i=0;i<nextvol.getRows();i++){
					for(int j=0;j<volumein.getImageData().getCols();j++){
						for(int k=ithstart;k<ithstart+ithlength;k++){
							for(int l=0;l<nextvol.getComponents();l++){
								nextvol.set(i, j, k,l, volumein.getImageData().getDouble(i, j, k, l));
							}
						}
					}
				}   

				//Add it to the list.
				subVols.add(nextvol);

				//and move on
				ithstart = bounds.get(boundind)+1;
				boundind++;
			}
		}else if(dimToSplit.getIndex()==3){
			for(int m=0; m<subcomponents.getInt(); m++){

				//Determine number of rows this image will have...
				int ithlength = bounds.get(boundind)-ithstart+1;

				//Initialize this Cubic volume to add...
				FileInfoBase[] fcat = new FileInfoBase[ithlength];
				for(int i=0; i<fcat.length; i++){
					fcat[i]=finfo[i];
				}
				ImageDataMipav nextvol = new ImageDataMipav(volumein.getName()+"_Comps_from"+ithstart,volumein.getImageData().getType(),volumein.getImageData().getRows(),
						volumein.getImageData().getCols(), volumein.getImageData().getSlices(), ithlength);

				nextvol.getModelImage().setFileInfo(fcat);

				//Set appropriate values for this volume...
				for(int i=0;i<nextvol.getRows();i++){
					for(int j=0;j<volumein.getImageData().getCols();j++){
						for(int k=0;k<nextvol.getSlices();k++){
							for(int l=ithstart;l<ithstart+ithlength;l++){
								nextvol.set(i, j, k,l, volumein.getImageData().getDouble(i, j, k, l));
							}
						}
					}
				}   

				//Add it to the list.
				subVols.add(nextvol);

				//and move on
				ithstart = bounds.get(boundind)+1;
				boundind++;
			}
		}else{
			System.err.println("Invalid dimension");
		}
		
	}
	
	// Breaks M (image size along input dim D) into N pieces
	private ArrayList<Integer> chooseSubsetBounds(){
		ArrayList<Integer> bounds = new ArrayList<Integer>();
		
		int M;
		switch (dimToSplit.getIndex()){

		case 0:M=volumein.getRows();
		case 1:M=volumein.getCols();
		case 2:M=volumein.getSlices();
		case 3:M=volumein.getComponents();
		default: M=-1; System.err.println("Invalid dim!");

		}
		
		int N = subcomponents.getInt();
		
		if(N>M)
			if(M>0){
				if(N==-1){		// N==-1 means we have a subcomponent for each slice
					for(int i=1; i<M; i++){
						bounds.add(i);
					}
				}else if(M%N==0){	//We can do something simple here.
					int bound = (M/N)-1;
					bounds.add(bound);
					bound+=(M/N);
					while(bound<M){
						bounds.add(bound);
						bound+=(M/N);
					}
				}else{	// have to add the 'extra' slices to the last group.
					int bound = (M/N)-1;
					bounds.add(bound);
					bound+=(M/N);
					int added = 1;
					while(bound<M){
						if(bound+(M/N)<M && added<N-1){
							bounds.add(bound);
							bound+=(M/N);
						}else{
							bounds.add(M-1);
							bound=M;
						}
					}
					added++;
				}
			}else{
				System.err.println("M must be > 0.  Check input dimension");
			}
		else{
			System.err.println("Number of subcomponents " + N + " must be <= than number" +
					" of slices" + M + " along input dimension " + dimToSplit.getIndex());
		}
		return bounds;
	}
}

