package edu.jhu.cs.cisst.algorithms.geometry.surface;

/**
 * JIST Extensions for Computer-Integrated Surgery
 *
 * Center for Computer-Integrated Surgical Systems and Technology &
 * Johns Hopkins Applied Physics Laboratory &
 * The Johns Hopkins University
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 * @author Blake Lucas
 */
import java.util.Hashtable;
import java.util.LinkedList;

import javax.vecmath.Point3f;
import javax.vecmath.Point3i;

import edu.jhu.cs.cisst.algorithms.segmentation.gac.TopologyPreservationRule3D;
import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;

/**
 * The IsoSurface Generator generates an iso-surface using either the
 * marching-cubes or marching-tetrahedral algorithm. The patent for
 * marching-cubes has expired as of 2005 and is now available for use in the
 * public domain. This implementation is a reimplementation of a C++ version
 * with enhancements that preserve mesh connectivity.
 * 
 * @author Cory Bloyd
 * 
 */
/*
 * Marching Cubes Example Program by Cory Bloyd (corysama@yahoo.com)
 * 
 * A simple, portable and complete implementation of the Marching Cubes and
 * Marching Tetrahedrons algorithms in a single source file. There are many ways
 * that this code could be made faster, but the intent is for the code to be
 * easy to understand.
 * 
 * For a description of the algorithm go to
 * http://astronomy.swin.edu.au/pbourke/modelling/polygonise/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 */
public class IsoSurfaceGenerator extends AbstractCalculation {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.11 $");
	}

	protected TopologyPreservationRule3D.Rule connectivityRule = null;

	public TopologyPreservationRule3D.Rule getConnectivityRule() {
		return connectivityRule;
	}

	public void setConnectivityRule(
			TopologyPreservationRule3D.Rule connectivityRule) {
		this.connectivityRule = connectivityRule;
	}

	protected boolean silent = false;

	public void setSilent(boolean silent) {
		this.silent = silent;
	}

	/** The nudge level set. */
	protected boolean nudgeLevelSet = true;

	/**
	 * Sets the nudge level set.
	 * 
	 * @param nudgeLevelSet
	 *            the new nudge level set
	 */
	public void setNudgeLevelSet(boolean nudgeLevelSet) {
		this.nudgeLevelSet = nudgeLevelSet;
	}

	/**
	 * The Class EdgeSplit is a container to describe a level set crossing
	 * point.
	 */
	protected class EdgeSplit {
		/** The grid point reference less than the level set value. */
		Point3i pt1;

		/** The grid point reference greater than the level set value. */
		Point3i pt2;

		/** The interpolated point on the target level set. */
		Point3f pt3d;

		/** The vertex id for the interpolated point. */
		int vid;

		/**
		 * Instantiates a new edge split.
		 * 
		 * @param pt1
		 *            the lower grid point
		 * @param pt2
		 *            the upper grid point
		 */
		public EdgeSplit(Point3i pt1, Point3i pt2) {
			this.pt1 = pt1;
			this.pt2 = pt2;
		}

		/**
		 * Compare two edge splits.
		 * 
		 * @param split
		 *            the split
		 * 
		 * @return true, if successful
		 */
		public boolean equals(EdgeSplit split) {
			return ((pt1.equals(split.pt1) && pt2.equals(split.pt2)) || (pt1
					.equals(split.pt2) && pt2.equals(split.pt1)));
		}

		/**
		 * Hash value that uniquely identifies edge split.
		 * 
		 * @return the hash value
		 */
		public long hashValue() {
			long d = rows * cols * slices;
			long h1 = hashValue(pt1);
			long h2 = hashValue(pt2);
			if (h1 < h2) {
				return h1 + d * h2;
			} else {
				return h2 + d * h1;
			}

		}

		/**
		 * Generate hash value for grid point
		 * 
		 * @param pt
		 *            the grid point
		 * 
		 * @return the hash value
		 */
		public long hashValue(Point3i pt) {
			return pt.z * rows * cols + rows * pt.y + pt.x;
		}

	}

	/**
	 * The Iso-surface generation method
	 */
	public enum Method {
		MARCHING_CUBES, MARCHING_TETRAHEDRALS
	};

	/**
	 * The Class Triangle stores vertex ids.
	 */
	protected class Triangle {

		/** The vertex ids. */
		protected int[] vids;

		/**
		 * Instantiates a new triangle.
		 */
		public Triangle() {
			this.vids = new int[3];
		}
	}

	/**
	 * The winding order for triangle vertices.
	 */
	public enum Winding {
		CLOCKWISE, COUNTER_CLOCKWISE
	}

	/** The is-surface generation method. */
	protected Method method = Method.MARCHING_CUBES;

	/** The winding order. */
	protected Winding winding = Winding.COUNTER_CLOCKWISE;

	/** The target iso-level. */
	protected float isoLevel = 0;

	/** The vertex count. */
	protected int vertCount = 0;

	/** The volume matrix */
	protected float[][][] volMat;

	/** The volume matrix */
	protected float[][][][] volsMat;

	/** The slices. */
	protected int rows = 0, cols = 0, slices = 0;

	/** The resolution in X. */
	protected float resX = 1;
	/** The resolution in Y. */
	protected float resY = 1;
	/** The resolution in Z. */
	protected float resZ = 1;

	/** The use image resolutions flag. */
	protected boolean useResolutions = true;

	protected int component;

	/**
	 * Lists the positions, relative to vertex 0, of each of the 8 vertices of a
	 * cube.
	 */
	protected static final int a2fVertexOffset[][] = { { 0, 0, 0 },
			{ 1, 0, 0 }, { 1, 1, 0 }, { 0, 1, 0 }, { 0, 0, 1 }, { 1, 0, 1 },
			{ 1, 1, 1 }, { 0, 1, 1 } };

	/**
	 * Lists the index of the endpoint vertices for each of the 12 edges of the
	 * cube.
	 */
	protected static final int a2iEdgeConnection[][] = { { 0, 1 }, { 1, 2 },
			{ 2, 3 }, { 3, 0 }, { 4, 5 }, { 5, 6 }, { 6, 7 }, { 7, 4 },
			{ 0, 4 }, { 1, 5 }, { 2, 6 }, { 3, 7 } };

	/**
	 * a2fEdgeDirection lists the direction vector (vertex1-vertex0) for each
	 * edge in the cube.
	 */
	protected static final float a2fEdgeDirection[][] = { { 1.0f, 0.0f, 0.0f },
			{ 0.0f, 1.0f, 0.0f }, { -1.0f, 0.0f, 0.0f }, { 0.0f, -1.0f, 0.0f },
			{ 1.0f, 0.0f, 0.0f }, { 0.0f, 1.0f, 0.0f }, { -1.0f, 0.0f, 0.0f },
			{ 0.0f, -1.0f, 0.0f }, { 0.0f, 0.0f, 1.0f }, { 0.0f, 0.0f, 1.0f },
			{ 0.0f, 0.0f, 1.0f }, { 0.0f, 0.0f, 1.0f } };

	/**
	 * Lists the index of the endpoint vertices for each of the 6 edges of the
	 * tetrahedron.
	 */
	protected static final int a2iTetrahedronEdgeConnection[][] = { { 0, 1 },
			{ 1, 2 }, { 2, 0 }, { 0, 3 }, { 1, 3 }, { 2, 3 } };

	/**
	 * Lists the index of verticies from a cube that made up each of the six
	 * tetrahedrons within the cube.
	 */
	protected static final int a2iTetrahedronsInACube[][] = { { 0, 5, 1, 6 },
			{ 0, 1, 2, 6 }, { 0, 2, 3, 6 }, { 0, 3, 7, 6 }, { 0, 7, 4, 6 },
			{ 0, 4, 5, 6 }, };

	/**
	 * For any edge, if one vertex is inside of the surface and the other is
	 * outside of the surface then the edge intersects the surface For each of
	 * the 8 vertices of the cube can be two possible states : either inside or
	 * outside of the surface For any cube the are 2^8=256 possible sets of
	 * vertex states This table lists the edges intersected by the surface for
	 * all 256 possible vertex states There are 12 edges. For each entry in the
	 * table, if edge #n is intersected, then bit #n is set to 1
	 */
	public static final int aiCubeEdgeFlags[] = { 0x000, 0x109, 0x203, 0x30a,
			0x406, 0x50f, 0x605, 0x70c, 0x80c, 0x905, 0xa0f, 0xb06, 0xc0a,
			0xd03, 0xe09, 0xf00, 0x190, 0x099, 0x393, 0x29a, 0x596, 0x49f,
			0x795, 0x69c, 0x99c, 0x895, 0xb9f, 0xa96, 0xd9a, 0xc93, 0xf99,
			0xe90, 0x230, 0x339, 0x033, 0x13a, 0x636, 0x73f, 0x435, 0x53c,
			0xa3c, 0xb35, 0x83f, 0x936, 0xe3a, 0xf33, 0xc39, 0xd30, 0x3a0,
			0x2a9, 0x1a3, 0x0aa, 0x7a6, 0x6af, 0x5a5, 0x4ac, 0xbac, 0xaa5,
			0x9af, 0x8a6, 0xfaa, 0xea3, 0xda9, 0xca0, 0x460, 0x569, 0x663,
			0x76a, 0x066, 0x16f, 0x265, 0x36c, 0xc6c, 0xd65, 0xe6f, 0xf66,
			0x86a, 0x963, 0xa69, 0xb60, 0x5f0, 0x4f9, 0x7f3, 0x6fa, 0x1f6,
			0x0ff, 0x3f5, 0x2fc, 0xdfc, 0xcf5, 0xfff, 0xef6, 0x9fa, 0x8f3,
			0xbf9, 0xaf0, 0x650, 0x759, 0x453, 0x55a, 0x256, 0x35f, 0x055,
			0x15c, 0xe5c, 0xf55, 0xc5f, 0xd56, 0xa5a, 0xb53, 0x859, 0x950,
			0x7c0, 0x6c9, 0x5c3, 0x4ca, 0x3c6, 0x2cf, 0x1c5, 0x0cc, 0xfcc,
			0xec5, 0xdcf, 0xcc6, 0xbca, 0xac3, 0x9c9, 0x8c0, 0x8c0, 0x9c9,
			0xac3, 0xbca, 0xcc6, 0xdcf, 0xec5, 0xfcc, 0x0cc, 0x1c5, 0x2cf,
			0x3c6, 0x4ca, 0x5c3, 0x6c9, 0x7c0, 0x950, 0x859, 0xb53, 0xa5a,
			0xd56, 0xc5f, 0xf55, 0xe5c, 0x15c, 0x055, 0x35f, 0x256, 0x55a,
			0x453, 0x759, 0x650, 0xaf0, 0xbf9, 0x8f3, 0x9fa, 0xef6, 0xfff,
			0xcf5, 0xdfc, 0x2fc, 0x3f5, 0x0ff, 0x1f6, 0x6fa, 0x7f3, 0x4f9,
			0x5f0, 0xb60, 0xa69, 0x963, 0x86a, 0xf66, 0xe6f, 0xd65, 0xc6c,
			0x36c, 0x265, 0x16f, 0x066, 0x76a, 0x663, 0x569, 0x460, 0xca0,
			0xda9, 0xea3, 0xfaa, 0x8a6, 0x9af, 0xaa5, 0xbac, 0x4ac, 0x5a5,
			0x6af, 0x7a6, 0x0aa, 0x1a3, 0x2a9, 0x3a0, 0xd30, 0xc39, 0xf33,
			0xe3a, 0x936, 0x83f, 0xb35, 0xa3c, 0x53c, 0x435, 0x73f, 0x636,
			0x13a, 0x033, 0x339, 0x230, 0xe90, 0xf99, 0xc93, 0xd9a, 0xa96,
			0xb9f, 0x895, 0x99c, 0x69c, 0x795, 0x49f, 0x596, 0x29a, 0x393,
			0x099, 0x190, 0xf00, 0xe09, 0xd03, 0xc0a, 0xb06, 0xa0f, 0x905,
			0x80c, 0x70c, 0x605, 0x50f, 0x406, 0x30a, 0x203, 0x109, 0x000 };
	/**
	 * For each of the possible vertex states listed in aiCubeEdgeFlags there is
	 * a specific triangulation of the edge intersection points.
	 * a2iTriangleConnectionTable lists all of them in the form of 0-5 edge
	 * triples with the list terminated by the invalid value -1. For example:
	 * a2iTriangleConnectionTable[3] list the 2 triangles formed when corner[0]
	 * and corner[1] are inside of the surface, but the rest of the cube is not.
	 * I found this table in an example program someone wrote long ago. It was
	 * probably generated by hand
	 */
	public static final int a2iTriangleConnectionTable[][] = {
			{ -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 1, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 8, 3, 9, 8, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 3, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 2, 10, 0, 2, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 8, 3, 2, 10, 8, 10, 9, 8, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 11, 2, 8, 11, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 9, 0, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 11, 2, 1, 9, 11, 9, 8, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 10, 1, 11, 10, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 10, 1, 0, 8, 10, 8, 11, 10, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 9, 0, 3, 11, 9, 11, 10, 9, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 3, 0, 7, 3, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 1, 9, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 1, 9, 4, 7, 1, 7, 3, 1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 10, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 4, 7, 3, 0, 4, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 2, 10, 9, 0, 2, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4, -1, -1, -1, -1 },
			{ 8, 4, 7, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 11, 4, 7, 11, 2, 4, 2, 0, 4, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 0, 1, 8, 4, 7, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 7, 11, 9, 4, 11, 9, 11, 2, 9, 2, 1, -1, -1, -1, -1 },
			{ 3, 10, 1, 3, 11, 10, 7, 8, 4, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 11, 10, 1, 4, 11, 1, 0, 4, 7, 11, 4, -1, -1, -1, -1 },
			{ 4, 7, 8, 9, 0, 11, 9, 11, 10, 11, 0, 3, -1, -1, -1, -1 },
			{ 4, 7, 11, 4, 11, 9, 9, 11, 10, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 5, 4, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 5, 4, 1, 5, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 5, 4, 8, 3, 5, 3, 1, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 10, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 0, 8, 1, 2, 10, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 2, 10, 5, 4, 2, 4, 0, 2, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8, -1, -1, -1, -1 },
			{ 9, 5, 4, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 11, 2, 0, 8, 11, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 5, 4, 0, 1, 5, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 1, 5, 2, 5, 8, 2, 8, 11, 4, 8, 5, -1, -1, -1, -1 },
			{ 10, 3, 11, 10, 1, 3, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 9, 5, 0, 8, 1, 8, 10, 1, 8, 11, 10, -1, -1, -1, -1 },
			{ 5, 4, 0, 5, 0, 11, 5, 11, 10, 11, 0, 3, -1, -1, -1, -1 },
			{ 5, 4, 8, 5, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 7, 8, 5, 7, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 3, 0, 9, 5, 3, 5, 7, 3, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 7, 8, 0, 1, 7, 1, 5, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 7, 8, 9, 5, 7, 10, 1, 2, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 1, 2, 9, 5, 0, 5, 3, 0, 5, 7, 3, -1, -1, -1, -1 },
			{ 8, 0, 2, 8, 2, 5, 8, 5, 7, 10, 5, 2, -1, -1, -1, -1 },
			{ 2, 10, 5, 2, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 7, 9, 5, 7, 8, 9, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7, 11, -1, -1, -1, -1 },
			{ 2, 3, 11, 0, 1, 8, 1, 7, 8, 1, 5, 7, -1, -1, -1, -1 },
			{ 11, 2, 1, 11, 1, 7, 7, 1, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 5, 8, 8, 5, 7, 10, 1, 3, 10, 3, 11, -1, -1, -1, -1 },
			{ 5, 7, 0, 5, 0, 9, 7, 11, 0, 1, 0, 10, 11, 10, 0, -1 },
			{ 11, 10, 0, 11, 0, 3, 10, 5, 0, 8, 0, 7, 5, 7, 0, -1 },
			{ 11, 10, 5, 7, 11, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 3, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 0, 1, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 8, 3, 1, 9, 8, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 6, 5, 2, 6, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 6, 5, 1, 2, 6, 3, 0, 8, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 6, 5, 9, 0, 6, 0, 2, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8, -1, -1, -1, -1 },
			{ 2, 3, 11, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 11, 0, 8, 11, 2, 0, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 1, 9, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 10, 6, 1, 9, 2, 9, 11, 2, 9, 8, 11, -1, -1, -1, -1 },
			{ 6, 3, 11, 6, 5, 3, 5, 1, 3, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 11, 0, 11, 5, 0, 5, 1, 5, 11, 6, -1, -1, -1, -1 },
			{ 3, 11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9, -1, -1, -1, -1 },
			{ 6, 5, 9, 6, 9, 11, 11, 9, 8, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 10, 6, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 3, 0, 4, 7, 3, 6, 5, 10, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 9, 0, 5, 10, 6, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 6, 5, 1, 9, 7, 1, 7, 3, 7, 9, 4, -1, -1, -1, -1 },
			{ 6, 1, 2, 6, 5, 1, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 5, 5, 2, 6, 3, 0, 4, 3, 4, 7, -1, -1, -1, -1 },
			{ 8, 4, 7, 9, 0, 5, 0, 6, 5, 0, 2, 6, -1, -1, -1, -1 },
			{ 7, 3, 9, 7, 9, 4, 3, 2, 9, 5, 9, 6, 2, 6, 9, -1 },
			{ 3, 11, 2, 7, 8, 4, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 10, 6, 4, 7, 2, 4, 2, 0, 2, 7, 11, -1, -1, -1, -1 },
			{ 0, 1, 9, 4, 7, 8, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1 },
			{ 9, 2, 1, 9, 11, 2, 9, 4, 11, 7, 11, 4, 5, 10, 6, -1 },
			{ 8, 4, 7, 3, 11, 5, 3, 5, 1, 5, 11, 6, -1, -1, -1, -1 },
			{ 5, 1, 11, 5, 11, 6, 1, 0, 11, 7, 11, 4, 0, 4, 11, -1 },
			{ 0, 5, 9, 0, 6, 5, 0, 3, 6, 11, 6, 3, 8, 4, 7, -1 },
			{ 6, 5, 9, 6, 9, 11, 4, 7, 9, 7, 11, 9, -1, -1, -1, -1 },
			{ 10, 4, 9, 6, 4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 10, 6, 4, 9, 10, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 0, 1, 10, 6, 0, 6, 4, 0, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1, 10, -1, -1, -1, -1 },
			{ 1, 4, 9, 1, 2, 4, 2, 6, 4, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 0, 8, 1, 2, 9, 2, 4, 9, 2, 6, 4, -1, -1, -1, -1 },
			{ 0, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 3, 2, 8, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 4, 9, 10, 6, 4, 11, 2, 3, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 2, 2, 8, 11, 4, 9, 10, 4, 10, 6, -1, -1, -1, -1 },
			{ 3, 11, 2, 0, 1, 6, 0, 6, 4, 6, 1, 10, -1, -1, -1, -1 },
			{ 6, 4, 1, 6, 1, 10, 4, 8, 1, 2, 1, 11, 8, 11, 1, -1 },
			{ 9, 6, 4, 9, 3, 6, 9, 1, 3, 11, 6, 3, -1, -1, -1, -1 },
			{ 8, 11, 1, 8, 1, 0, 11, 6, 1, 9, 1, 4, 6, 4, 1, -1 },
			{ 3, 11, 6, 3, 6, 0, 0, 6, 4, -1, -1, -1, -1, -1, -1, -1 },
			{ 6, 4, 8, 11, 6, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 7, 10, 6, 7, 8, 10, 8, 9, 10, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 7, 3, 0, 10, 7, 0, 9, 10, 6, 7, 10, -1, -1, -1, -1 },
			{ 10, 6, 7, 1, 10, 7, 1, 7, 8, 1, 8, 0, -1, -1, -1, -1 },
			{ 10, 6, 7, 10, 7, 1, 1, 7, 3, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7, -1, -1, -1, -1 },
			{ 2, 6, 9, 2, 9, 1, 6, 7, 9, 0, 9, 3, 7, 3, 9, -1 },
			{ 7, 8, 0, 7, 0, 6, 6, 0, 2, -1, -1, -1, -1, -1, -1, -1 },
			{ 7, 3, 2, 6, 7, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 3, 11, 10, 6, 8, 10, 8, 9, 8, 6, 7, -1, -1, -1, -1 },
			{ 2, 0, 7, 2, 7, 11, 0, 9, 7, 6, 7, 10, 9, 10, 7, -1 },
			{ 1, 8, 0, 1, 7, 8, 1, 10, 7, 6, 7, 10, 2, 3, 11, -1 },
			{ 11, 2, 1, 11, 1, 7, 10, 6, 1, 6, 7, 1, -1, -1, -1, -1 },
			{ 8, 9, 6, 8, 6, 7, 9, 1, 6, 11, 6, 3, 1, 3, 6, -1 },
			{ 0, 9, 1, 11, 6, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 7, 8, 0, 7, 0, 6, 3, 11, 0, 11, 6, 0, -1, -1, -1, -1 },
			{ 7, 11, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 0, 8, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 1, 9, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 1, 9, 8, 3, 1, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 1, 2, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 10, 3, 0, 8, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 9, 0, 2, 10, 9, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 6, 11, 7, 2, 10, 3, 10, 8, 3, 10, 9, 8, -1, -1, -1, -1 },
			{ 7, 2, 3, 6, 2, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 7, 0, 8, 7, 6, 0, 6, 2, 0, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 7, 6, 2, 3, 7, 0, 1, 9, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6, -1, -1, -1, -1 },
			{ 10, 7, 6, 10, 1, 7, 1, 3, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 7, 6, 1, 7, 10, 1, 8, 7, 1, 0, 8, -1, -1, -1, -1 },
			{ 0, 3, 7, 0, 7, 10, 0, 10, 9, 6, 10, 7, -1, -1, -1, -1 },
			{ 7, 6, 10, 7, 10, 8, 8, 10, 9, -1, -1, -1, -1, -1, -1, -1 },
			{ 6, 8, 4, 11, 8, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 6, 11, 3, 0, 6, 0, 4, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 6, 11, 8, 4, 6, 9, 0, 1, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 4, 6, 9, 6, 3, 9, 3, 1, 11, 3, 6, -1, -1, -1, -1 },
			{ 6, 8, 4, 6, 11, 8, 2, 10, 1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 10, 3, 0, 11, 0, 6, 11, 0, 4, 6, -1, -1, -1, -1 },
			{ 4, 11, 8, 4, 6, 11, 0, 2, 9, 2, 10, 9, -1, -1, -1, -1 },
			{ 10, 9, 3, 10, 3, 2, 9, 4, 3, 11, 3, 6, 4, 6, 3, -1 },
			{ 8, 2, 3, 8, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 9, 0, 2, 3, 4, 2, 4, 6, 4, 3, 8, -1, -1, -1, -1 },
			{ 1, 9, 4, 1, 4, 2, 2, 4, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 1, 3, 8, 6, 1, 8, 4, 6, 6, 10, 1, -1, -1, -1, -1 },
			{ 10, 1, 0, 10, 0, 6, 6, 0, 4, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 6, 3, 4, 3, 8, 6, 10, 3, 0, 3, 9, 10, 9, 3, -1 },
			{ 10, 9, 4, 6, 10, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 9, 5, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 3, 4, 9, 5, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 0, 1, 5, 4, 0, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 11, 7, 6, 8, 3, 4, 3, 5, 4, 3, 1, 5, -1, -1, -1, -1 },
			{ 9, 5, 4, 10, 1, 2, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 6, 11, 7, 1, 2, 10, 0, 8, 3, 4, 9, 5, -1, -1, -1, -1 },
			{ 7, 6, 11, 5, 4, 10, 4, 2, 10, 4, 0, 2, -1, -1, -1, -1 },
			{ 3, 4, 8, 3, 5, 4, 3, 2, 5, 10, 5, 2, 11, 7, 6, -1 },
			{ 7, 2, 3, 7, 6, 2, 5, 4, 9, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 5, 4, 0, 8, 6, 0, 6, 2, 6, 8, 7, -1, -1, -1, -1 },
			{ 3, 6, 2, 3, 7, 6, 1, 5, 0, 5, 4, 0, -1, -1, -1, -1 },
			{ 6, 2, 8, 6, 8, 7, 2, 1, 8, 4, 8, 5, 1, 5, 8, -1 },
			{ 9, 5, 4, 10, 1, 6, 1, 7, 6, 1, 3, 7, -1, -1, -1, -1 },
			{ 1, 6, 10, 1, 7, 6, 1, 0, 7, 8, 7, 0, 9, 5, 4, -1 },
			{ 4, 0, 10, 4, 10, 5, 0, 3, 10, 6, 10, 7, 3, 7, 10, -1 },
			{ 7, 6, 10, 7, 10, 8, 5, 4, 10, 4, 8, 10, -1, -1, -1, -1 },
			{ 6, 9, 5, 6, 11, 9, 11, 8, 9, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 6, 11, 0, 6, 3, 0, 5, 6, 0, 9, 5, -1, -1, -1, -1 },
			{ 0, 11, 8, 0, 5, 11, 0, 1, 5, 5, 6, 11, -1, -1, -1, -1 },
			{ 6, 11, 3, 6, 3, 5, 5, 3, 1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 10, 9, 5, 11, 9, 11, 8, 11, 5, 6, -1, -1, -1, -1 },
			{ 0, 11, 3, 0, 6, 11, 0, 9, 6, 5, 6, 9, 1, 2, 10, -1 },
			{ 11, 8, 5, 11, 5, 6, 8, 0, 5, 10, 5, 2, 0, 2, 5, -1 },
			{ 6, 11, 3, 6, 3, 5, 2, 10, 3, 10, 5, 3, -1, -1, -1, -1 },
			{ 5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2, -1, -1, -1, -1 },
			{ 9, 5, 6, 9, 6, 0, 0, 6, 2, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 5, 8, 1, 8, 0, 5, 6, 8, 3, 8, 2, 6, 2, 8, -1 },
			{ 1, 5, 6, 2, 1, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 3, 6, 1, 6, 10, 3, 8, 6, 5, 6, 9, 8, 9, 6, -1 },
			{ 10, 1, 0, 10, 0, 6, 9, 5, 0, 5, 6, 0, -1, -1, -1, -1 },
			{ 0, 3, 8, 5, 6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 5, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 11, 5, 10, 7, 5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 11, 5, 10, 11, 7, 5, 8, 3, 0, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 11, 7, 5, 10, 11, 1, 9, 0, -1, -1, -1, -1, -1, -1, -1 },
			{ 10, 7, 5, 10, 11, 7, 9, 8, 1, 8, 3, 1, -1, -1, -1, -1 },
			{ 11, 1, 2, 11, 7, 1, 7, 5, 1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 3, 1, 2, 7, 1, 7, 5, 7, 2, 11, -1, -1, -1, -1 },
			{ 9, 7, 5, 9, 2, 7, 9, 0, 2, 2, 11, 7, -1, -1, -1, -1 },
			{ 7, 5, 2, 7, 2, 11, 5, 9, 2, 3, 2, 8, 9, 8, 2, -1 },
			{ 2, 5, 10, 2, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 2, 0, 8, 5, 2, 8, 7, 5, 10, 2, 5, -1, -1, -1, -1 },
			{ 9, 0, 1, 5, 10, 3, 5, 3, 7, 3, 10, 2, -1, -1, -1, -1 },
			{ 9, 8, 2, 9, 2, 1, 8, 7, 2, 10, 2, 5, 7, 5, 2, -1 },
			{ 1, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 7, 0, 7, 1, 1, 7, 5, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 0, 3, 9, 3, 5, 5, 3, 7, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 8, 7, 5, 9, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 8, 4, 5, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1 },
			{ 5, 0, 4, 5, 11, 0, 5, 10, 11, 11, 3, 0, -1, -1, -1, -1 },
			{ 0, 1, 9, 8, 4, 10, 8, 10, 11, 10, 4, 5, -1, -1, -1, -1 },
			{ 10, 11, 4, 10, 4, 5, 11, 3, 4, 9, 4, 1, 3, 1, 4, -1 },
			{ 2, 5, 1, 2, 8, 5, 2, 11, 8, 4, 5, 8, -1, -1, -1, -1 },
			{ 0, 4, 11, 0, 11, 3, 4, 5, 11, 2, 11, 1, 5, 1, 11, -1 },
			{ 0, 2, 5, 0, 5, 9, 2, 11, 5, 4, 5, 8, 11, 8, 5, -1 },
			{ 9, 4, 5, 2, 11, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 5, 10, 3, 5, 2, 3, 4, 5, 3, 8, 4, -1, -1, -1, -1 },
			{ 5, 10, 2, 5, 2, 4, 4, 2, 0, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 10, 2, 3, 5, 10, 3, 8, 5, 4, 5, 8, 0, 1, 9, -1 },
			{ 5, 10, 2, 5, 2, 4, 1, 9, 2, 9, 4, 2, -1, -1, -1, -1 },
			{ 8, 4, 5, 8, 5, 3, 3, 5, 1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 4, 5, 1, 0, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 8, 4, 5, 8, 5, 3, 9, 0, 5, 0, 3, 5, -1, -1, -1, -1 },
			{ 9, 4, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 11, 7, 4, 9, 11, 9, 10, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 8, 3, 4, 9, 7, 9, 11, 7, 9, 10, 11, -1, -1, -1, -1 },
			{ 1, 10, 11, 1, 11, 4, 1, 4, 0, 7, 4, 11, -1, -1, -1, -1 },
			{ 3, 1, 4, 3, 4, 8, 1, 10, 4, 7, 4, 11, 10, 11, 4, -1 },
			{ 4, 11, 7, 9, 11, 4, 9, 2, 11, 9, 1, 2, -1, -1, -1, -1 },
			{ 9, 7, 4, 9, 11, 7, 9, 1, 11, 2, 11, 1, 0, 8, 3, -1 },
			{ 11, 7, 4, 11, 4, 2, 2, 4, 0, -1, -1, -1, -1, -1, -1, -1 },
			{ 11, 7, 4, 11, 4, 2, 8, 3, 4, 3, 2, 4, -1, -1, -1, -1 },
			{ 2, 9, 10, 2, 7, 9, 2, 3, 7, 7, 4, 9, -1, -1, -1, -1 },
			{ 9, 10, 7, 9, 7, 4, 10, 2, 7, 8, 7, 0, 2, 0, 7, -1 },
			{ 3, 7, 10, 3, 10, 2, 7, 4, 10, 1, 10, 0, 4, 0, 10, -1 },
			{ 1, 10, 2, 8, 7, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 9, 1, 4, 1, 7, 7, 1, 3, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 9, 1, 4, 1, 7, 0, 8, 1, 8, 7, 1, -1, -1, -1, -1 },
			{ 4, 0, 3, 7, 4, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 4, 8, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 0, 9, 3, 9, 11, 11, 9, 10, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 1, 10, 0, 10, 8, 8, 10, 11, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 1, 10, 11, 3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 2, 11, 1, 11, 9, 9, 11, 8, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 0, 9, 3, 9, 11, 1, 2, 9, 2, 11, 9, -1, -1, -1, -1 },
			{ 0, 2, 11, 8, 0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 3, 2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 3, 8, 2, 8, 10, 10, 8, 9, -1, -1, -1, -1, -1, -1, -1 },
			{ 9, 10, 2, 0, 9, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 2, 3, 8, 2, 8, 10, 0, 1, 8, 1, 10, 8, -1, -1, -1, -1 },
			{ 1, 10, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 1, 3, 8, 9, 1, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 9, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ 0, 3, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
			{ -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 } };

	public static final int[] aiCubeEdgeFlagsCC626 = new int[] { 0x0, 0x109,
			0x203, 0x30a, 0x406, 0x50f, 0x605, 0x70c, 0x80c, 0x905, 0xa0f,
			0xb06, 0xc0a, 0xd03, 0xe09, 0xf00, 0x190, 0x99, 0x393, 0x29a,
			0x596, 0x49f, 0x795, 0x69c, 0x99c, 0x895, 0xb9f, 0xa96, 0xd9a,
			0xc93, 0xf99, 0xe90, 0x230, 0x339, 0x33, 0x13a, 0x636, 0x73f,
			0x435, 0x53c, 0xa3c, 0xb35, 0x83f, 0x936, 0xe3a, 0xf33, 0xc39,
			0xd30, 0x3a0, 0x2a9, 0x1a3, 0xaa, 0x7a6, 0x6af, 0x5a5, 0x4ac,
			0xbac, 0xaa5, 0x9af, 0x8a6, 0xfaa, 0xea3, 0xda9, 0xca0, 0x460,
			0x569, 0x663, 0x76a, 0x66, 0x16f, 0x265, 0x36c, 0xc6c, 0xd65,
			0xe6f, 0xf66, 0x86a, 0x963, 0xa69, 0xb60, 0x5f0, 0x4f9, 0x7f3,
			0x6fa, 0x1f6, 0xff, 0x3f5, 0x2fc, 0xdfc, 0xcf5, 0xfff, 0xef6,
			0x9fa, 0x8f3, 0xbf9, 0xaf0, 0x650, 0x759, 0x453, 0x55a, 0x256,
			0x35f, 0x55, 0x15c, 0xe5c, 0xf55, 0xc5f, 0xd56, 0xa5a, 0xb53,
			0x859, 0x950, 0x7c0, 0x6c9, 0x5c3, 0x4ca, 0x3c6, 0x2cf, 0x1c5,
			0xcc, 0xfcc, 0xec5, 0xdcf, 0xcc6, 0xbca, 0xac3, 0x9c9, 0x8c0,
			0x8c0, 0x9c9, 0xac3, 0xbca, 0xcc6, 0xdcf, 0xec5, 0xfcc, 0xcc,
			0x1c5, 0x2cf, 0x3c6, 0x4ca, 0x5c3, 0x6c9, 0x7c0, 0x950, 0x859,
			0xb53, 0xa5a, 0xd56, 0xc5f, 0xf55, 0xe5c, 0x15c, 0x55, 0x35f,
			0x256, 0x55a, 0x453, 0x759, 0x650, 0xaf0, 0xbf9, 0x8f3, 0x9fa,
			0xef6, 0xfff, 0xcf5, 0xdfc, 0x2fc, 0x3f5, 0xff, 0x1f6, 0x6fa,
			0x7f3, 0x4f9, 0x5f0, 0xb60, 0xa69, 0x963, 0x86a, 0xf66, 0xe6f,
			0xd65, 0xc6c, 0x36c, 0x265, 0x16f, 0x66, 0x76a, 0x663, 0x569,
			0x460, 0xca0, 0xda9, 0xea3, 0xfaa, 0x8a6, 0x9af, 0xaa5, 0xbac,
			0x4ac, 0x5a5, 0x6af, 0x7a6, 0xaa, 0x1a3, 0x2a9, 0x3a0, 0xd30,
			0xc39, 0xf33, 0xe3a, 0x936, 0x83f, 0xb35, 0xa3c, 0x53c, 0x435,
			0x73f, 0x636, 0x13a, 0x33, 0x339, 0x230, 0xe90, 0xf99, 0xc93,
			0xd9a, 0xa96, 0xb9f, 0x895, 0x99c, 0x69c, 0x795, 0x49f, 0x596,
			0x29a, 0x393, 0x99, 0x190, 0xf00, 0xe09, 0xd03, 0xc0a, 0xb06,
			0xa0f, 0x905, 0x80c, 0x70c, 0x605, 0x50f, 0x406, 0x30a, 0x203,
			0x109, 0x0 };
	public static final int[][] a2iTriangleConnectionTableCC626 = new int[][] {
			{}, { 8, 0, 3 }, { 9, 1, 0 }, { 3, 8, 9, 9, 1, 3 }, { 10, 2, 1 },
			{ 0, 1, 10, 10, 8, 0, 10, 2, 3, 3, 8, 10 }, { 0, 9, 10, 10, 2, 0 },
			{ 8, 9, 10, 10, 2, 8, 2, 3, 8 }, { 11, 3, 2 },
			{ 2, 11, 8, 8, 0, 2 }, { 3, 0, 9, 9, 11, 3, 9, 1, 2, 2, 11, 9 },
			{ 11, 8, 9, 9, 1, 11, 1, 2, 11 }, { 1, 10, 11, 11, 3, 1 },
			{ 10, 11, 8, 8, 0, 10, 0, 1, 10 },
			{ 9, 10, 11, 11, 3, 9, 3, 0, 9 }, { 11, 8, 9, 9, 10, 11 },
			{ 7, 4, 8 }, { 0, 3, 7, 7, 4, 0 },
			{ 4, 9, 1, 1, 7, 4, 1, 0, 8, 8, 7, 1 },
			{ 1, 3, 7, 7, 4, 1, 4, 9, 1 },
			{ 10, 7, 4, 10, 4, 1, 1, 4, 8, 1, 8, 2, 2, 8, 7, 2, 7, 10 },
			{ 4, 0, 1, 4, 1, 10, 4, 10, 7, 7, 10, 2, 7, 2, 3 },
			{ 2, 0, 8, 2, 8, 7, 2, 7, 10, 10, 7, 4, 10, 4, 9 },
			{ 10, 2, 3, 4, 9, 10, 3, 7, 4, 4, 10, 3 },
			{ 8, 3, 2, 2, 4, 8, 2, 11, 7, 7, 4, 2 },
			{ 4, 0, 2, 2, 11, 4, 11, 7, 4 },
			{ 3, 0, 8, 11, 7, 4, 11, 4, 9, 11, 9, 2, 1, 2, 9 },
			{ 11, 1, 2, 7, 9, 1, 7, 4, 9, 1, 11, 7 },
			{ 10, 11, 7, 10, 7, 4, 10, 4, 1, 1, 4, 8, 1, 8, 3 },
			{ 7, 4, 0, 10, 11, 7, 0, 1, 10, 7, 0, 10 },
			{ 8, 3, 0, 10, 11, 7, 4, 9, 10, 10, 7, 4 },
			{ 9, 10, 11, 11, 4, 9, 11, 7, 4 }, { 9, 4, 5 },
			{ 8, 4, 5, 5, 3, 8, 5, 9, 0, 0, 3, 5 }, { 4, 5, 1, 1, 0, 4 },
			{ 5, 1, 3, 3, 8, 5, 8, 4, 5 },
			{ 1, 9, 4, 4, 2, 1, 4, 5, 10, 10, 2, 4 },
			{ 0, 1, 9, 8, 4, 5, 8, 5, 10, 8, 10, 3, 2, 3, 10 },
			{ 2, 0, 4, 4, 5, 2, 5, 10, 2 },
			{ 5, 8, 4, 10, 3, 8, 10, 2, 3, 8, 5, 10 },
			{ 11, 4, 5, 11, 5, 2, 2, 5, 9, 2, 9, 3, 3, 9, 4, 3, 4, 11 },
			{ 11, 8, 4, 11, 4, 5, 11, 5, 2, 2, 5, 9, 2, 9, 0 },
			{ 5, 1, 2, 5, 2, 11, 5, 11, 4, 4, 11, 3, 4, 3, 0 },
			{ 4, 5, 1, 11, 8, 4, 1, 2, 11, 4, 1, 11 },
			{ 3, 1, 9, 3, 9, 4, 3, 4, 11, 11, 4, 5, 11, 5, 10 },
			{ 9, 0, 1, 11, 8, 4, 5, 10, 11, 11, 4, 5 },
			{ 11, 3, 0, 5, 10, 11, 0, 4, 5, 5, 11, 0 },
			{ 10, 11, 8, 8, 5, 10, 8, 4, 5 }, { 5, 9, 8, 8, 7, 5 },
			{ 3, 7, 5, 5, 9, 3, 9, 0, 3 }, { 7, 5, 1, 1, 0, 7, 0, 8, 7 },
			{ 5, 1, 3, 3, 7, 5 },
			{ 7, 5, 10, 7, 10, 2, 7, 2, 8, 8, 2, 1, 8, 1, 9 },
			{ 1, 9, 0, 7, 5, 10, 2, 3, 7, 7, 10, 2 },
			{ 10, 2, 0, 7, 5, 10, 0, 8, 7, 10, 0, 7 },
			{ 3, 7, 5, 5, 2, 3, 5, 10, 2 },
			{ 9, 8, 3, 9, 3, 2, 9, 2, 5, 5, 2, 11, 5, 11, 7 },
			{ 2, 11, 7, 9, 0, 2, 7, 5, 9, 9, 2, 7 },
			{ 3, 0, 8, 5, 1, 2, 11, 7, 5, 5, 2, 11 },
			{ 7, 5, 1, 1, 11, 7, 1, 2, 11 },
			{ 7, 5, 10, 10, 11, 7, 1, 9, 8, 8, 3, 1 },
			{ 1, 9, 0, 7, 5, 10, 10, 11, 7 }, { 8, 3, 0, 10, 11, 7, 7, 5, 10 },
			{ 10, 11, 7, 7, 5, 10 }, { 6, 10, 5 },
			{ 8, 5, 6, 8, 6, 3, 3, 6, 10, 3, 10, 0, 0, 10, 5, 0, 5, 8 },
			{ 9, 5, 6, 6, 0, 9, 6, 10, 1, 1, 0, 6 },
			{ 8, 9, 5, 8, 5, 6, 8, 6, 3, 3, 6, 10, 3, 10, 1 },
			{ 2, 1, 5, 5, 6, 2 },
			{ 6, 2, 3, 6, 3, 8, 6, 8, 5, 5, 8, 0, 5, 0, 1 },
			{ 6, 2, 0, 0, 9, 6, 9, 5, 6 },
			{ 3, 8, 9, 6, 2, 3, 9, 5, 6, 3, 9, 6 },
			{ 2, 10, 5, 5, 3, 2, 5, 6, 11, 11, 3, 5 },
			{ 0, 2, 10, 0, 10, 5, 0, 5, 8, 8, 5, 6, 8, 6, 11 },
			{ 1, 2, 10, 9, 5, 6, 9, 6, 11, 9, 11, 0, 3, 0, 11 },
			{ 10, 1, 2, 8, 9, 5, 6, 11, 8, 8, 5, 6 },
			{ 3, 1, 5, 5, 6, 3, 6, 11, 3 },
			{ 8, 0, 1, 6, 11, 8, 1, 5, 6, 6, 8, 1 },
			{ 9, 3, 0, 5, 11, 3, 5, 6, 11, 3, 9, 5 },
			{ 11, 8, 9, 9, 6, 11, 9, 5, 6 },
			{ 7, 6, 10, 10, 8, 7, 10, 5, 4, 4, 8, 10 },
			{ 3, 7, 6, 3, 6, 10, 3, 10, 0, 0, 10, 5, 0, 5, 4 },
			{ 9, 5, 4, 0, 8, 7, 0, 7, 6, 0, 6, 1, 10, 1, 6 },
			{ 5, 4, 9, 3, 7, 6, 10, 1, 3, 3, 6, 10 },
			{ 1, 5, 4, 1, 4, 8, 1, 8, 2, 2, 8, 7, 2, 7, 6 },
			{ 1, 5, 4, 4, 0, 1, 7, 6, 2, 2, 3, 7 },
			{ 4, 9, 5, 2, 0, 8, 7, 6, 2, 2, 8, 7 },
			{ 5, 4, 9, 3, 7, 6, 6, 2, 3 },
			{ 6, 11, 7, 5, 4, 8, 5, 8, 3, 5, 3, 10, 2, 10, 3 },
			{ 6, 11, 7, 0, 2, 10, 5, 4, 0, 0, 10, 5 },
			{ 2, 10, 1, 9, 5, 4, 11, 7, 6, 0, 8, 3 },
			{ 2, 10, 1, 9, 5, 4, 6, 11, 7 },
			{ 7, 6, 11, 1, 5, 4, 8, 3, 1, 1, 4, 8 },
			{ 7, 6, 11, 1, 5, 4, 4, 0, 1 }, { 0, 8, 3, 11, 7, 6, 4, 9, 5 },
			{ 9, 5, 4, 11, 7, 6 }, { 9, 4, 6, 6, 10, 9 },
			{ 10, 9, 0, 10, 0, 3, 10, 3, 6, 6, 3, 8, 6, 8, 4 },
			{ 0, 4, 6, 6, 10, 0, 10, 1, 0 },
			{ 6, 10, 1, 8, 4, 6, 1, 3, 8, 8, 6, 1 },
			{ 4, 6, 2, 2, 1, 4, 1, 9, 4 },
			{ 0, 1, 9, 6, 2, 3, 8, 4, 6, 6, 3, 8 }, { 2, 0, 4, 4, 6, 2 },
			{ 4, 6, 2, 2, 8, 4, 2, 3, 8 },
			{ 4, 6, 11, 4, 11, 3, 4, 3, 9, 9, 3, 2, 9, 2, 10 },
			{ 4, 6, 11, 11, 8, 4, 2, 10, 9, 9, 0, 2 },
			{ 2, 10, 1, 4, 6, 11, 3, 0, 4, 4, 11, 3 },
			{ 2, 10, 1, 4, 6, 11, 11, 8, 4 },
			{ 9, 4, 6, 3, 1, 9, 6, 11, 3, 9, 6, 3 },
			{ 9, 0, 1, 11, 8, 4, 4, 6, 11 }, { 0, 4, 6, 6, 3, 0, 6, 11, 3 },
			{ 11, 8, 4, 4, 6, 11 }, { 10, 9, 8, 8, 7, 10, 7, 6, 10 },
			{ 0, 3, 7, 10, 9, 0, 7, 6, 10, 0, 7, 10 },
			{ 10, 7, 6, 1, 8, 7, 1, 0, 8, 7, 10, 1 },
			{ 1, 3, 7, 7, 10, 1, 7, 6, 10 },
			{ 8, 7, 6, 1, 9, 8, 6, 2, 1, 1, 8, 6 },
			{ 0, 1, 9, 6, 2, 3, 3, 7, 6 }, { 6, 2, 0, 0, 7, 6, 0, 8, 7 },
			{ 3, 7, 6, 6, 2, 3 }, { 11, 7, 6, 9, 8, 3, 2, 10, 9, 9, 3, 2 },
			{ 6, 11, 7, 0, 2, 10, 10, 9, 0 }, { 6, 11, 7, 8, 3, 0, 2, 10, 1 },
			{ 1, 2, 10, 7, 6, 11 }, { 11, 7, 6, 9, 8, 3, 3, 1, 9 },
			{ 6, 11, 7, 0, 1, 9 }, { 0, 8, 3, 6, 11, 7 }, { 7, 6, 11 },
			{ 11, 6, 7 }, { 3, 11, 6, 6, 0, 3, 6, 7, 8, 8, 0, 6 },
			{ 9, 6, 7, 9, 7, 0, 0, 7, 11, 0, 11, 1, 1, 11, 6, 1, 6, 9 },
			{ 1, 3, 11, 1, 11, 6, 1, 6, 9, 9, 6, 7, 9, 7, 8 },
			{ 10, 6, 7, 7, 1, 10, 7, 11, 2, 2, 1, 7 },
			{ 2, 3, 11, 10, 6, 7, 10, 7, 8, 10, 8, 1, 0, 1, 8 },
			{ 9, 10, 6, 9, 6, 7, 9, 7, 0, 0, 7, 11, 0, 11, 2 },
			{ 11, 2, 3, 9, 10, 6, 7, 8, 9, 9, 6, 7 }, { 6, 7, 3, 3, 2, 6 },
			{ 0, 2, 6, 6, 7, 0, 7, 8, 0 },
			{ 7, 3, 0, 7, 0, 9, 7, 9, 6, 6, 9, 1, 6, 1, 2 },
			{ 9, 1, 2, 7, 8, 9, 2, 6, 7, 7, 9, 2 },
			{ 7, 3, 1, 1, 10, 7, 10, 6, 7 },
			{ 0, 7, 8, 1, 6, 7, 1, 10, 6, 7, 0, 1 },
			{ 6, 7, 3, 9, 10, 6, 3, 0, 9, 6, 3, 9 },
			{ 8, 9, 10, 10, 7, 8, 10, 6, 7 }, { 4, 8, 11, 11, 6, 4 },
			{ 6, 4, 0, 0, 3, 6, 3, 11, 6 },
			{ 6, 4, 9, 6, 9, 1, 6, 1, 11, 11, 1, 0, 11, 0, 8 },
			{ 11, 6, 4, 1, 3, 11, 4, 9, 1, 11, 4, 1 },
			{ 8, 11, 2, 8, 2, 1, 8, 1, 4, 4, 1, 10, 4, 10, 6 },
			{ 2, 3, 11, 4, 0, 1, 10, 6, 4, 4, 1, 10 },
			{ 6, 4, 9, 9, 10, 6, 0, 8, 11, 11, 2, 0 },
			{ 11, 2, 3, 9, 10, 6, 6, 4, 9 }, { 2, 6, 4, 4, 8, 2, 8, 3, 2 },
			{ 6, 4, 0, 0, 2, 6 }, { 0, 8, 3, 6, 4, 9, 1, 2, 6, 6, 9, 1 },
			{ 2, 6, 4, 4, 1, 2, 4, 9, 1 },
			{ 4, 8, 3, 10, 6, 4, 3, 1, 10, 10, 4, 3 },
			{ 6, 4, 0, 0, 10, 6, 0, 1, 10 }, { 0, 8, 3, 6, 4, 9, 9, 10, 6 },
			{ 6, 4, 9, 9, 10, 6 }, { 4, 7, 11, 11, 9, 4, 11, 6, 5, 5, 9, 11 },
			{ 4, 7, 8, 9, 0, 3, 9, 3, 11, 9, 11, 5, 6, 5, 11 },
			{ 0, 4, 7, 0, 7, 11, 0, 11, 1, 1, 11, 6, 1, 6, 5 },
			{ 7, 8, 4, 1, 3, 11, 6, 5, 1, 1, 11, 6 },
			{ 6, 5, 10, 11, 2, 1, 11, 1, 9, 11, 9, 7, 4, 7, 9 },
			{ 1, 9, 0, 8, 4, 7, 10, 6, 5, 3, 11, 2 },
			{ 6, 5, 10, 0, 4, 7, 11, 2, 0, 0, 7, 11 },
			{ 4, 7, 8, 3, 11, 2, 6, 5, 10 },
			{ 2, 6, 5, 2, 5, 9, 2, 9, 3, 3, 9, 4, 3, 4, 7 },
			{ 4, 7, 8, 2, 6, 5, 9, 0, 2, 2, 5, 9 },
			{ 7, 3, 0, 0, 4, 7, 1, 2, 6, 6, 5, 1 },
			{ 4, 7, 8, 2, 6, 5, 5, 1, 2 },
			{ 5, 10, 6, 3, 1, 9, 4, 7, 3, 3, 9, 4 },
			{ 8, 4, 7, 6, 5, 10, 9, 0, 1 }, { 6, 5, 10, 0, 4, 7, 7, 3, 0 },
			{ 8, 4, 7, 10, 6, 5 }, { 9, 8, 11, 11, 6, 9, 6, 5, 9 },
			{ 9, 6, 5, 0, 11, 6, 0, 3, 11, 6, 9, 0 },
			{ 11, 6, 5, 0, 8, 11, 5, 1, 0, 0, 11, 5 },
			{ 5, 1, 3, 3, 6, 5, 3, 11, 6 },
			{ 10, 6, 5, 8, 11, 2, 1, 9, 8, 8, 2, 1 },
			{ 5, 10, 6, 11, 2, 3, 1, 9, 0 }, { 10, 6, 5, 8, 11, 2, 2, 0, 8 },
			{ 3, 11, 2, 5, 10, 6 }, { 3, 2, 6, 9, 8, 3, 6, 5, 9, 3, 6, 9 },
			{ 0, 2, 6, 6, 9, 0, 6, 5, 9 }, { 3, 0, 8, 5, 1, 2, 2, 6, 5 },
			{ 5, 1, 2, 2, 6, 5 }, { 5, 10, 6, 3, 1, 9, 9, 8, 3 },
			{ 0, 1, 9, 6, 5, 10 }, { 5, 10, 6, 3, 0, 8 }, { 5, 10, 6 },
			{ 7, 11, 10, 10, 5, 7 },
			{ 5, 7, 8, 5, 8, 0, 5, 0, 10, 10, 0, 3, 10, 3, 11 },
			{ 11, 10, 1, 11, 1, 0, 11, 0, 7, 7, 0, 9, 7, 9, 5 },
			{ 5, 7, 8, 8, 9, 5, 3, 11, 10, 10, 1, 3 },
			{ 1, 5, 7, 7, 11, 1, 11, 2, 1 },
			{ 3, 11, 2, 5, 7, 8, 0, 1, 5, 5, 8, 0 },
			{ 0, 9, 5, 11, 2, 0, 5, 7, 11, 11, 0, 5 },
			{ 3, 11, 2, 5, 7, 8, 8, 9, 5 }, { 5, 7, 3, 3, 2, 5, 2, 10, 5 },
			{ 8, 0, 2, 5, 7, 8, 2, 10, 5, 8, 2, 5 },
			{ 1, 2, 10, 7, 3, 0, 9, 5, 7, 7, 0, 9 },
			{ 10, 1, 2, 8, 9, 5, 5, 7, 8 }, { 1, 5, 7, 7, 3, 1 },
			{ 1, 5, 7, 7, 0, 1, 7, 8, 0 }, { 5, 7, 3, 3, 9, 5, 3, 0, 9 },
			{ 8, 9, 5, 5, 7, 8 }, { 8, 11, 10, 10, 5, 8, 5, 4, 8 },
			{ 10, 5, 4, 3, 11, 10, 4, 0, 3, 3, 10, 4 },
			{ 9, 5, 4, 11, 10, 1, 0, 8, 11, 11, 1, 0 },
			{ 9, 5, 4, 11, 10, 1, 1, 3, 11 },
			{ 2, 1, 5, 8, 11, 2, 5, 4, 8, 2, 5, 8 },
			{ 2, 3, 11, 4, 0, 1, 1, 5, 4 }, { 4, 9, 5, 2, 0, 8, 8, 11, 2 },
			{ 4, 9, 5, 2, 3, 11 }, { 8, 5, 4, 3, 10, 5, 3, 2, 10, 5, 8, 3 },
			{ 4, 0, 2, 2, 5, 4, 2, 10, 5 }, { 4, 9, 5, 10, 1, 2, 0, 8, 3 },
			{ 2, 10, 1, 4, 9, 5 }, { 3, 1, 5, 5, 8, 3, 5, 4, 8 },
			{ 1, 5, 4, 4, 0, 1 }, { 3, 0, 8, 5, 4, 9 }, { 5, 4, 9 },
			{ 11, 10, 9, 9, 4, 11, 4, 7, 11 },
			{ 8, 4, 7, 10, 9, 0, 3, 11, 10, 10, 0, 3 },
			{ 1, 0, 4, 11, 10, 1, 4, 7, 11, 1, 4, 11 },
			{ 7, 8, 4, 1, 3, 11, 11, 10, 1 },
			{ 11, 4, 7, 2, 9, 4, 2, 1, 9, 4, 11, 2 },
			{ 7, 8, 4, 9, 0, 1, 3, 11, 2 }, { 2, 0, 4, 4, 11, 2, 4, 7, 11 },
			{ 4, 7, 8, 2, 3, 11 }, { 9, 4, 7, 2, 10, 9, 7, 3, 2, 2, 9, 7 },
			{ 8, 4, 7, 10, 9, 0, 0, 2, 10 }, { 1, 2, 10, 7, 3, 0, 0, 4, 7 },
			{ 7, 8, 4, 1, 2, 10 }, { 7, 3, 1, 1, 4, 7, 1, 9, 4 },
			{ 7, 8, 4, 1, 9, 0 }, { 7, 3, 0, 0, 4, 7 }, { 8, 4, 7 },
			{ 10, 9, 8, 8, 11, 10 }, { 11, 10, 9, 9, 3, 11, 9, 0, 3 },
			{ 8, 11, 10, 10, 0, 8, 10, 1, 0 }, { 11, 10, 1, 1, 3, 11 },
			{ 9, 8, 11, 11, 1, 9, 11, 2, 1 }, { 11, 2, 3, 9, 0, 1 },
			{ 8, 11, 2, 2, 0, 8 }, { 2, 3, 11 },
			{ 10, 9, 8, 8, 2, 10, 8, 3, 2 }, { 10, 9, 0, 0, 2, 10 },
			{ 8, 3, 0, 10, 1, 2 }, { 1, 2, 10 }, { 9, 8, 3, 3, 1, 9 },
			{ 0, 1, 9 }, { 3, 0, 8 }, {} };

	public static final int[] aiCubeEdgeFlagsCC618 = new int[] { 0x0, 0x109,
			0x203, 0x30a, 0x406, 0x50f, 0x605, 0x70c, 0x80c, 0x905, 0xa0f,
			0xb06, 0xc0a, 0xd03, 0xe09, 0xf00, 0x190, 0x99, 0x393, 0x29a,
			0x596, 0x49f, 0x795, 0x69c, 0x99c, 0x895, 0xb9f, 0xa96, 0xd9a,
			0xc93, 0xf99, 0xe90, 0x230, 0x339, 0x33, 0x13a, 0x636, 0x73f,
			0x435, 0x53c, 0xa3c, 0xb35, 0x83f, 0x936, 0xe3a, 0xf33, 0xc39,
			0xd30, 0x3a0, 0x2a9, 0x1a3, 0xaa, 0x7a6, 0x6af, 0x5a5, 0x4ac,
			0xbac, 0xaa5, 0x9af, 0x8a6, 0xfaa, 0xea3, 0xda9, 0xca0, 0x460,
			0x569, 0x663, 0x76a, 0x66, 0x16f, 0x265, 0x36c, 0xc6c, 0xd65,
			0xe6f, 0xf66, 0x86a, 0x963, 0xa69, 0xb60, 0x5f0, 0x4f9, 0x7f3,
			0x6fa, 0x1f6, 0xff, 0x3f5, 0x2fc, 0xdfc, 0xcf5, 0xfff, 0xef6,
			0x9fa, 0x8f3, 0xbf9, 0xaf0, 0x650, 0x759, 0x453, 0x55a, 0x256,
			0x35f, 0x55, 0x15c, 0xe5c, 0xf55, 0xc5f, 0xd56, 0xa5a, 0xb53,
			0x859, 0x950, 0x7c0, 0x6c9, 0x5c3, 0x4ca, 0x3c6, 0x2cf, 0x1c5,
			0xcc, 0xfcc, 0xec5, 0xdcf, 0xcc6, 0xbca, 0xac3, 0x9c9, 0x8c0,
			0x8c0, 0x9c9, 0xac3, 0xbca, 0xcc6, 0xdcf, 0xec5, 0xfcc, 0xcc,
			0x1c5, 0x2cf, 0x3c6, 0x4ca, 0x5c3, 0x6c9, 0x7c0, 0x950, 0x859,
			0xb53, 0xa5a, 0xd56, 0xc5f, 0xf55, 0xe5c, 0x15c, 0x55, 0x35f,
			0x256, 0x55a, 0x453, 0x759, 0x650, 0xaf0, 0xbf9, 0x8f3, 0x9fa,
			0xef6, 0xfff, 0xcf5, 0xdfc, 0x2fc, 0x3f5, 0xff, 0x1f6, 0x6fa,
			0x7f3, 0x4f9, 0x5f0, 0xb60, 0xa69, 0x963, 0x86a, 0xf66, 0xe6f,
			0xd65, 0xc6c, 0x36c, 0x265, 0x16f, 0x66, 0x76a, 0x663, 0x569,
			0x460, 0xca0, 0xda9, 0xea3, 0xfaa, 0x8a6, 0x9af, 0xaa5, 0xbac,
			0x4ac, 0x5a5, 0x6af, 0x7a6, 0xaa, 0x1a3, 0x2a9, 0x3a0, 0xd30,
			0xc39, 0xf33, 0xe3a, 0x936, 0x83f, 0xb35, 0xa3c, 0x53c, 0x435,
			0x73f, 0x636, 0x13a, 0x33, 0x339, 0x230, 0xe90, 0xf99, 0xc93,
			0xd9a, 0xa96, 0xb9f, 0x895, 0x99c, 0x69c, 0x795, 0x49f, 0x596,
			0x29a, 0x393, 0x99, 0x190, 0xf00, 0xe09, 0xd03, 0xc0a, 0xb06,
			0xa0f, 0x905, 0x80c, 0x70c, 0x605, 0x50f, 0x406, 0x30a, 0x203,
			0x109, 0x0 };
	public static final int[][] a2iTriangleConnectionTableCC618 = new int[][] {
			{}, { 8, 0, 3 }, { 9, 1, 0 }, { 3, 8, 9, 9, 1, 3 }, { 10, 2, 1 },
			{ 0, 1, 10, 10, 8, 0, 10, 2, 3, 3, 8, 10 }, { 0, 9, 10, 10, 2, 0 },
			{ 8, 9, 10, 10, 2, 8, 2, 3, 8 }, { 11, 3, 2 },
			{ 2, 11, 8, 8, 0, 2 }, { 3, 0, 9, 9, 11, 3, 9, 1, 2, 2, 11, 9 },
			{ 11, 8, 9, 9, 1, 11, 1, 2, 11 }, { 1, 10, 11, 11, 3, 1 },
			{ 10, 11, 8, 8, 0, 10, 0, 1, 10 },
			{ 9, 10, 11, 11, 3, 9, 3, 0, 9 }, { 11, 8, 9, 9, 10, 11 },
			{ 7, 4, 8 }, { 0, 3, 7, 7, 4, 0 },
			{ 4, 9, 1, 1, 7, 4, 1, 0, 8, 8, 7, 1 },
			{ 1, 3, 7, 7, 4, 1, 4, 9, 1 }, { 4, 8, 7, 10, 2, 1 },
			{ 4, 0, 1, 4, 1, 10, 4, 10, 7, 7, 10, 2, 7, 2, 3 },
			{ 2, 0, 8, 2, 8, 7, 2, 7, 10, 10, 7, 4, 10, 4, 9 },
			{ 10, 2, 3, 4, 9, 10, 3, 7, 4, 4, 10, 3 },
			{ 8, 3, 2, 2, 4, 8, 2, 11, 7, 7, 4, 2 },
			{ 4, 0, 2, 2, 11, 4, 11, 7, 4 },
			{ 3, 0, 8, 11, 7, 4, 11, 4, 9, 11, 9, 2, 1, 2, 9 },
			{ 11, 1, 2, 7, 9, 1, 7, 4, 9, 1, 11, 7 },
			{ 10, 11, 7, 10, 7, 4, 10, 4, 1, 1, 4, 8, 1, 8, 3 },
			{ 7, 4, 0, 10, 11, 7, 0, 1, 10, 7, 0, 10 },
			{ 8, 3, 0, 10, 11, 7, 4, 9, 10, 10, 7, 4 },
			{ 9, 10, 11, 11, 4, 9, 11, 7, 4 }, { 9, 4, 5 },
			{ 8, 4, 5, 5, 3, 8, 5, 9, 0, 0, 3, 5 }, { 4, 5, 1, 1, 0, 4 },
			{ 5, 1, 3, 3, 8, 5, 8, 4, 5 },
			{ 1, 9, 4, 4, 2, 1, 4, 5, 10, 10, 2, 4 },
			{ 0, 1, 9, 8, 4, 5, 8, 5, 10, 8, 10, 3, 2, 3, 10 },
			{ 2, 0, 4, 4, 5, 2, 5, 10, 2 },
			{ 5, 8, 4, 10, 3, 8, 10, 2, 3, 8, 5, 10 }, { 5, 9, 4, 11, 3, 2 },
			{ 11, 8, 4, 11, 4, 5, 11, 5, 2, 2, 5, 9, 2, 9, 0 },
			{ 5, 1, 2, 5, 2, 11, 5, 11, 4, 4, 11, 3, 4, 3, 0 },
			{ 4, 5, 1, 11, 8, 4, 1, 2, 11, 4, 1, 11 },
			{ 3, 1, 9, 3, 9, 4, 3, 4, 11, 11, 4, 5, 11, 5, 10 },
			{ 9, 0, 1, 11, 8, 4, 5, 10, 11, 11, 4, 5 },
			{ 11, 3, 0, 5, 10, 11, 0, 4, 5, 5, 11, 0 },
			{ 10, 11, 8, 8, 5, 10, 8, 4, 5 }, { 5, 9, 8, 8, 7, 5 },
			{ 3, 7, 5, 5, 9, 3, 9, 0, 3 }, { 7, 5, 1, 1, 0, 7, 0, 8, 7 },
			{ 5, 1, 3, 3, 7, 5 },
			{ 7, 5, 10, 7, 10, 2, 7, 2, 8, 8, 2, 1, 8, 1, 9 },
			{ 1, 9, 0, 7, 5, 10, 2, 3, 7, 7, 10, 2 },
			{ 10, 2, 0, 7, 5, 10, 0, 8, 7, 10, 0, 7 },
			{ 3, 7, 5, 5, 2, 3, 5, 10, 2 },
			{ 9, 8, 3, 9, 3, 2, 9, 2, 5, 5, 2, 11, 5, 11, 7 },
			{ 2, 11, 7, 9, 0, 2, 7, 5, 9, 9, 2, 7 },
			{ 3, 0, 8, 5, 1, 2, 11, 7, 5, 5, 2, 11 },
			{ 7, 5, 1, 1, 11, 7, 1, 2, 11 },
			{ 7, 5, 10, 10, 11, 7, 1, 9, 8, 8, 3, 1 },
			{ 1, 9, 0, 7, 5, 10, 10, 11, 7 }, { 8, 3, 0, 10, 11, 7, 7, 5, 10 },
			{ 10, 11, 7, 7, 5, 10 }, { 6, 10, 5 }, { 6, 10, 5, 8, 0, 3 },
			{ 9, 5, 6, 6, 0, 9, 6, 10, 1, 1, 0, 6 },
			{ 8, 9, 5, 8, 5, 6, 8, 6, 3, 3, 6, 10, 3, 10, 1 },
			{ 2, 1, 5, 5, 6, 2 },
			{ 6, 2, 3, 6, 3, 8, 6, 8, 5, 5, 8, 0, 5, 0, 1 },
			{ 6, 2, 0, 0, 9, 6, 9, 5, 6 },
			{ 3, 8, 9, 6, 2, 3, 9, 5, 6, 3, 9, 6 },
			{ 2, 10, 5, 5, 3, 2, 5, 6, 11, 11, 3, 5 },
			{ 0, 2, 10, 0, 10, 5, 0, 5, 8, 8, 5, 6, 8, 6, 11 },
			{ 1, 2, 10, 9, 5, 6, 9, 6, 11, 9, 11, 0, 3, 0, 11 },
			{ 10, 1, 2, 8, 9, 5, 6, 11, 8, 8, 5, 6 },
			{ 3, 1, 5, 5, 6, 3, 6, 11, 3 },
			{ 8, 0, 1, 6, 11, 8, 1, 5, 6, 6, 8, 1 },
			{ 9, 3, 0, 5, 11, 3, 5, 6, 11, 3, 9, 5 },
			{ 11, 8, 9, 9, 6, 11, 9, 5, 6 },
			{ 7, 6, 10, 10, 8, 7, 10, 5, 4, 4, 8, 10 },
			{ 3, 7, 6, 3, 6, 10, 3, 10, 0, 0, 10, 5, 0, 5, 4 },
			{ 9, 5, 4, 0, 8, 7, 0, 7, 6, 0, 6, 1, 10, 1, 6 },
			{ 5, 4, 9, 3, 7, 6, 10, 1, 3, 3, 6, 10 },
			{ 1, 5, 4, 1, 4, 8, 1, 8, 2, 2, 8, 7, 2, 7, 6 },
			{ 1, 5, 4, 4, 0, 1, 7, 6, 2, 2, 3, 7 },
			{ 4, 9, 5, 2, 0, 8, 7, 6, 2, 2, 8, 7 },
			{ 5, 4, 9, 3, 7, 6, 6, 2, 3 },
			{ 6, 11, 7, 5, 4, 8, 5, 8, 3, 5, 3, 10, 2, 10, 3 },
			{ 6, 11, 7, 0, 2, 10, 5, 4, 0, 0, 10, 5 },
			{ 2, 10, 1, 9, 5, 4, 11, 7, 6, 0, 8, 3 },
			{ 2, 10, 1, 9, 5, 4, 6, 11, 7 },
			{ 7, 6, 11, 1, 5, 4, 8, 3, 1, 1, 4, 8 },
			{ 7, 6, 11, 1, 5, 4, 4, 0, 1 }, { 0, 8, 3, 11, 7, 6, 4, 9, 5 },
			{ 9, 5, 4, 11, 7, 6 }, { 9, 4, 6, 6, 10, 9 },
			{ 10, 9, 0, 10, 0, 3, 10, 3, 6, 6, 3, 8, 6, 8, 4 },
			{ 0, 4, 6, 6, 10, 0, 10, 1, 0 },
			{ 6, 10, 1, 8, 4, 6, 1, 3, 8, 8, 6, 1 },
			{ 4, 6, 2, 2, 1, 4, 1, 9, 4 },
			{ 0, 1, 9, 6, 2, 3, 8, 4, 6, 6, 3, 8 }, { 2, 0, 4, 4, 6, 2 },
			{ 4, 6, 2, 2, 8, 4, 2, 3, 8 },
			{ 4, 6, 11, 4, 11, 3, 4, 3, 9, 9, 3, 2, 9, 2, 10 },
			{ 4, 6, 11, 11, 8, 4, 2, 10, 9, 9, 0, 2 },
			{ 2, 10, 1, 4, 6, 11, 3, 0, 4, 4, 11, 3 },
			{ 2, 10, 1, 4, 6, 11, 11, 8, 4 },
			{ 9, 4, 6, 3, 1, 9, 6, 11, 3, 9, 6, 3 },
			{ 9, 0, 1, 11, 8, 4, 4, 6, 11 }, { 0, 4, 6, 6, 3, 0, 6, 11, 3 },
			{ 11, 8, 4, 4, 6, 11 }, { 10, 9, 8, 8, 7, 10, 7, 6, 10 },
			{ 0, 3, 7, 10, 9, 0, 7, 6, 10, 0, 7, 10 },
			{ 10, 7, 6, 1, 8, 7, 1, 0, 8, 7, 10, 1 },
			{ 1, 3, 7, 7, 10, 1, 7, 6, 10 },
			{ 8, 7, 6, 1, 9, 8, 6, 2, 1, 1, 8, 6 },
			{ 0, 1, 9, 6, 2, 3, 3, 7, 6 }, { 6, 2, 0, 0, 7, 6, 0, 8, 7 },
			{ 3, 7, 6, 6, 2, 3 }, { 11, 7, 6, 9, 8, 3, 2, 10, 9, 9, 3, 2 },
			{ 6, 11, 7, 0, 2, 10, 10, 9, 0 }, { 6, 11, 7, 8, 3, 0, 2, 10, 1 },
			{ 1, 2, 10, 7, 6, 11 }, { 11, 7, 6, 9, 8, 3, 3, 1, 9 },
			{ 6, 11, 7, 0, 1, 9 }, { 0, 8, 3, 6, 11, 7 }, { 7, 6, 11 },
			{ 11, 6, 7 }, { 3, 11, 6, 6, 0, 3, 6, 7, 8, 8, 0, 6 },
			{ 7, 11, 6, 9, 1, 0 },
			{ 1, 3, 11, 1, 11, 6, 1, 6, 9, 9, 6, 7, 9, 7, 8 },
			{ 10, 6, 7, 7, 1, 10, 7, 11, 2, 2, 1, 7 },
			{ 2, 3, 11, 10, 6, 7, 10, 7, 8, 10, 8, 1, 0, 1, 8 },
			{ 9, 10, 6, 9, 6, 7, 9, 7, 0, 0, 7, 11, 0, 11, 2 },
			{ 11, 2, 3, 9, 10, 6, 7, 8, 9, 9, 6, 7 }, { 6, 7, 3, 3, 2, 6 },
			{ 0, 2, 6, 6, 7, 0, 7, 8, 0 },
			{ 7, 3, 0, 7, 0, 9, 7, 9, 6, 6, 9, 1, 6, 1, 2 },
			{ 9, 1, 2, 7, 8, 9, 2, 6, 7, 7, 9, 2 },
			{ 7, 3, 1, 1, 10, 7, 10, 6, 7 },
			{ 0, 7, 8, 1, 6, 7, 1, 10, 6, 7, 0, 1 },
			{ 6, 7, 3, 9, 10, 6, 3, 0, 9, 6, 3, 9 },
			{ 8, 9, 10, 10, 7, 8, 10, 6, 7 }, { 4, 8, 11, 11, 6, 4 },
			{ 6, 4, 0, 0, 3, 6, 3, 11, 6 },
			{ 6, 4, 9, 6, 9, 1, 6, 1, 11, 11, 1, 0, 11, 0, 8 },
			{ 11, 6, 4, 1, 3, 11, 4, 9, 1, 11, 4, 1 },
			{ 8, 11, 2, 8, 2, 1, 8, 1, 4, 4, 1, 10, 4, 10, 6 },
			{ 2, 3, 11, 4, 0, 1, 10, 6, 4, 4, 1, 10 },
			{ 6, 4, 9, 9, 10, 6, 0, 8, 11, 11, 2, 0 },
			{ 11, 2, 3, 9, 10, 6, 6, 4, 9 }, { 2, 6, 4, 4, 8, 2, 8, 3, 2 },
			{ 6, 4, 0, 0, 2, 6 }, { 0, 8, 3, 6, 4, 9, 1, 2, 6, 6, 9, 1 },
			{ 2, 6, 4, 4, 1, 2, 4, 9, 1 },
			{ 4, 8, 3, 10, 6, 4, 3, 1, 10, 10, 4, 3 },
			{ 6, 4, 0, 0, 10, 6, 0, 1, 10 }, { 0, 8, 3, 6, 4, 9, 9, 10, 6 },
			{ 6, 4, 9, 9, 10, 6 }, { 4, 7, 11, 11, 9, 4, 11, 6, 5, 5, 9, 11 },
			{ 4, 7, 8, 9, 0, 3, 9, 3, 11, 9, 11, 5, 6, 5, 11 },
			{ 0, 4, 7, 0, 7, 11, 0, 11, 1, 1, 11, 6, 1, 6, 5 },
			{ 7, 8, 4, 1, 3, 11, 6, 5, 1, 1, 11, 6 },
			{ 6, 5, 10, 11, 2, 1, 11, 1, 9, 11, 9, 7, 4, 7, 9 },
			{ 1, 9, 0, 8, 4, 7, 10, 6, 5, 3, 11, 2 },
			{ 6, 5, 10, 0, 4, 7, 11, 2, 0, 0, 7, 11 },
			{ 4, 7, 8, 3, 11, 2, 6, 5, 10 },
			{ 2, 6, 5, 2, 5, 9, 2, 9, 3, 3, 9, 4, 3, 4, 7 },
			{ 4, 7, 8, 2, 6, 5, 9, 0, 2, 2, 5, 9 },
			{ 7, 3, 0, 0, 4, 7, 1, 2, 6, 6, 5, 1 },
			{ 4, 7, 8, 2, 6, 5, 5, 1, 2 },
			{ 5, 10, 6, 3, 1, 9, 4, 7, 3, 3, 9, 4 },
			{ 8, 4, 7, 6, 5, 10, 9, 0, 1 }, { 6, 5, 10, 0, 4, 7, 7, 3, 0 },
			{ 8, 4, 7, 10, 6, 5 }, { 9, 8, 11, 11, 6, 9, 6, 5, 9 },
			{ 9, 6, 5, 0, 11, 6, 0, 3, 11, 6, 9, 0 },
			{ 11, 6, 5, 0, 8, 11, 5, 1, 0, 0, 11, 5 },
			{ 5, 1, 3, 3, 6, 5, 3, 11, 6 },
			{ 10, 6, 5, 8, 11, 2, 1, 9, 8, 8, 2, 1 },
			{ 5, 10, 6, 11, 2, 3, 1, 9, 0 }, { 10, 6, 5, 8, 11, 2, 2, 0, 8 },
			{ 3, 11, 2, 5, 10, 6 }, { 3, 2, 6, 9, 8, 3, 6, 5, 9, 3, 6, 9 },
			{ 0, 2, 6, 6, 9, 0, 6, 5, 9 }, { 3, 0, 8, 5, 1, 2, 2, 6, 5 },
			{ 5, 1, 2, 2, 6, 5 }, { 5, 10, 6, 3, 1, 9, 9, 8, 3 },
			{ 0, 1, 9, 6, 5, 10 }, { 5, 10, 6, 3, 0, 8 }, { 5, 10, 6 },
			{ 7, 11, 10, 10, 5, 7 },
			{ 5, 7, 8, 5, 8, 0, 5, 0, 10, 10, 0, 3, 10, 3, 11 },
			{ 11, 10, 1, 11, 1, 0, 11, 0, 7, 7, 0, 9, 7, 9, 5 },
			{ 5, 7, 8, 8, 9, 5, 3, 11, 10, 10, 1, 3 },
			{ 1, 5, 7, 7, 11, 1, 11, 2, 1 },
			{ 3, 11, 2, 5, 7, 8, 0, 1, 5, 5, 8, 0 },
			{ 0, 9, 5, 11, 2, 0, 5, 7, 11, 11, 0, 5 },
			{ 3, 11, 2, 5, 7, 8, 8, 9, 5 }, { 5, 7, 3, 3, 2, 5, 2, 10, 5 },
			{ 8, 0, 2, 5, 7, 8, 2, 10, 5, 8, 2, 5 },
			{ 1, 2, 10, 7, 3, 0, 9, 5, 7, 7, 0, 9 },
			{ 10, 1, 2, 8, 9, 5, 5, 7, 8 }, { 1, 5, 7, 7, 3, 1 },
			{ 1, 5, 7, 7, 0, 1, 7, 8, 0 }, { 5, 7, 3, 3, 9, 5, 3, 0, 9 },
			{ 8, 9, 5, 5, 7, 8 }, { 8, 11, 10, 10, 5, 8, 5, 4, 8 },
			{ 10, 5, 4, 3, 11, 10, 4, 0, 3, 3, 10, 4 },
			{ 9, 5, 4, 11, 10, 1, 0, 8, 11, 11, 1, 0 },
			{ 9, 5, 4, 11, 10, 1, 1, 3, 11 },
			{ 2, 1, 5, 8, 11, 2, 5, 4, 8, 2, 5, 8 },
			{ 2, 3, 11, 4, 0, 1, 1, 5, 4 }, { 4, 9, 5, 2, 0, 8, 8, 11, 2 },
			{ 4, 9, 5, 2, 3, 11 }, { 8, 5, 4, 3, 10, 5, 3, 2, 10, 5, 8, 3 },
			{ 4, 0, 2, 2, 5, 4, 2, 10, 5 }, { 4, 9, 5, 10, 1, 2, 0, 8, 3 },
			{ 2, 10, 1, 4, 9, 5 }, { 3, 1, 5, 5, 8, 3, 5, 4, 8 },
			{ 1, 5, 4, 4, 0, 1 }, { 3, 0, 8, 5, 4, 9 }, { 5, 4, 9 },
			{ 11, 10, 9, 9, 4, 11, 4, 7, 11 },
			{ 8, 4, 7, 10, 9, 0, 3, 11, 10, 10, 0, 3 },
			{ 1, 0, 4, 11, 10, 1, 4, 7, 11, 1, 4, 11 },
			{ 7, 8, 4, 1, 3, 11, 11, 10, 1 },
			{ 11, 4, 7, 2, 9, 4, 2, 1, 9, 4, 11, 2 },
			{ 7, 8, 4, 9, 0, 1, 3, 11, 2 }, { 2, 0, 4, 4, 11, 2, 4, 7, 11 },
			{ 4, 7, 8, 2, 3, 11 }, { 9, 4, 7, 2, 10, 9, 7, 3, 2, 2, 9, 7 },
			{ 8, 4, 7, 10, 9, 0, 0, 2, 10 }, { 1, 2, 10, 7, 3, 0, 0, 4, 7 },
			{ 7, 8, 4, 1, 2, 10 }, { 7, 3, 1, 1, 4, 7, 1, 9, 4 },
			{ 7, 8, 4, 1, 9, 0 }, { 7, 3, 0, 0, 4, 7 }, { 8, 4, 7 },
			{ 10, 9, 8, 8, 11, 10 }, { 11, 10, 9, 9, 3, 11, 9, 0, 3 },
			{ 8, 11, 10, 10, 0, 8, 10, 1, 0 }, { 11, 10, 1, 1, 3, 11 },
			{ 9, 8, 11, 11, 1, 9, 11, 2, 1 }, { 11, 2, 3, 9, 0, 1 },
			{ 8, 11, 2, 2, 0, 8 }, { 2, 3, 11 },
			{ 10, 9, 8, 8, 2, 10, 8, 3, 2 }, { 10, 9, 0, 0, 2, 10 },
			{ 8, 3, 0, 10, 1, 2 }, { 1, 2, 10 }, { 9, 8, 3, 3, 1, 9 },
			{ 0, 1, 9 }, { 3, 0, 8 }, {} };
	/**
	 * For any edge, if one vertex is inside of the surface and the other is
	 * outside of the surface then the edge intersects the surface For each of
	 * the 4 vertices of the tetrahedron can be two possible states : either
	 * inside or outside of the surface For any tetrahedron the are 2^4=16
	 * possible sets of vertex states This table lists the edges intersected by
	 * the surface for all 16 possible vertex states There are 6 edges. For each
	 * entry in the table, if edge #n is intersected, then bit #n is set to 1
	 */
	protected static final int aiTetrahedronEdgeFlags[] = { 0x00, 0x0d, 0x13,
			0x1e, 0x26, 0x2b, 0x35, 0x38, 0x38, 0x35, 0x2b, 0x26, 0x1e, 0x13,
			0x0d, 0x00, };

	/**
	 * For each of the possible vertex states listed in aiTetrahedronEdgeFlags
	 * there is a specific triangulation of the edge intersection points.
	 * a2iTetrahedronTriangles lists all of them in the form of 0-2 edge triples
	 * with the list terminated by the invalid value -1.
	 */
	protected static final int a2iTetrahedronTriangles[][] = {
			{ -1, -1, -1, -1, -1, -1, -1 }, { 0, 3, 2, -1, -1, -1, -1 },
			{ 0, 1, 4, -1, -1, -1, -1 }, { 1, 4, 2, 2, 4, 3, -1 },

			{ 1, 2, 5, -1, -1, -1, -1 }, { 0, 3, 5, 0, 5, 1, -1 },
			{ 0, 2, 5, 0, 5, 4, -1 }, { 5, 4, 3, -1, -1, -1, -1 },

			{ 3, 4, 5, -1, -1, -1, -1 }, { 4, 5, 0, 5, 2, 0, -1 },
			{ 1, 5, 0, 5, 3, 0, -1 }, { 5, 2, 1, -1, -1, -1, -1 },

			{ 3, 4, 2, 2, 4, 1, -1 }, { 4, 1, 0, -1, -1, -1, -1 },
			{ 2, 3, 0, -1, -1, -1, -1 }, { -1, -1, -1, -1, -1, -1, -1 }, };

	/**
	 * Instantiates a new iso surface generator.
	 */
	public IsoSurfaceGenerator() {
		super();
		setLabel("Iso-Surface");
	}

	public IsoSurfaceGenerator(TopologyPreservationRule3D.Rule rule) {
		super();
		this.connectivityRule = rule;
		setLabel("Iso-Surface");
	}

	public IsoSurfaceGenerator(AbstractCalculation parent,
			TopologyPreservationRule3D.Rule rule) {
		super(parent);
		this.connectivityRule = rule;
		setLabel("Iso-Surface");
	}

	/**
	 * Instantiates a new iso surface generator.
	 * 
	 * @param parent
	 *            the parent
	 */
	public IsoSurfaceGenerator(AbstractCalculation parent) {
		super(parent);
		setLabel("Iso-Surface");
	}

	/**
	 * Clamp column index.
	 * 
	 * @param c
	 *            the column
	 * 
	 * @return the column
	 */
	protected int clampColumn(int c) {
		return Math.max(Math.min(c, cols - 1), 0);
	}

	/**
	 * Clamp row.
	 * 
	 * @param r
	 *            the row
	 * 
	 * @return the row
	 */
	protected int clampRow(int r) {
		return Math.max(Math.min(r, rows - 1), 0);
	}

	/**
	 * Clamp slice.
	 * 
	 * @param s
	 *            the slice
	 * 
	 * @return the slice
	 */
	protected int clampSlice(int s) {
		return Math.max(Math.min(s, slices - 1), 0);
	}

	/**
	 * Interpolate position along edge.
	 * 
	 * @param v1
	 *            the lower grid point
	 * @param v2
	 *            the upper grid point
	 * 
	 * @return the position
	 */
	protected float fGetOffset(Point3i v1, Point3i v2) {
		float fValue1 = getValue(v1.x, v1.y, v1.z);
		float fValue2 = getValue(v2.x, v2.y, v2.z);
		double fDelta = fValue2 - fValue1;

		if (fDelta == 0.0) {
			return 0.5f;
		}
		return (float) ((isoLevel - fValue1) / fDelta);
	}

	/**
	 * Get level set value
	 * 
	 * @param i
	 *            row
	 * @param j
	 *            column
	 * @return
	 */
	protected float getValue(int i, int j, int k) {
		float val = 0;
		if (component != -1 && volsMat != null) {
			val = volsMat[i][j][k][component];
			if (nudgeLevelSet) {
				// Push iso-level away from zero level set
				if (val < isoLevel) {
					val = (float) Math.min(val, isoLevel - 0.1);
				} else {
					val = (float) Math.max(val, isoLevel + 0.1);
				}
			}
		} else {
			val = volMat[i][j][k];
			if (nudgeLevelSet) {
				// Push iso-level away from zero level set
				if (val < isoLevel) {
					val = (float) Math.min(val, isoLevel - 0.1);
				} else {
					val = (float) Math.max(val, isoLevel + 0.1);
				}
			}
		}
		return val;
	}

	/**
	 * March tetrahedral.
	 * 
	 * @param splits
	 *            the edge splits
	 * @param triangles
	 *            the triangles
	 * @param pasTetrahedronPosition
	 *            the tetrahedron positions
	 */
	protected void marchTetrahedral(Hashtable<Long, EdgeSplit> splits,
			LinkedList<Triangle> triangles, Point3i[] pasTetrahedronPosition) {
		int iEdge, iVert0, iVert1, iEdgeFlags, iVertex, iFlagIndex = 0;
		int[] asEdgeVertex = new int[6];

		// Find which vertices are inside of the surface and which are outside
		for (iVertex = 0; iVertex < 4; iVertex++) {
			Point3i v = pasTetrahedronPosition[iVertex];
			if (getValue(v.x, v.y, v.z) <= isoLevel)
				iFlagIndex |= 1 << iVertex;
		}

		// Find which edges are intersected by the surface
		iEdgeFlags = aiTetrahedronEdgeFlags[iFlagIndex];

		// If the tetrahedron is entirely inside or outside of the surface, then
		// there will be no intersections
		if (iEdgeFlags == 0) {
			return;
		}
		// Find the point of intersection of the surface with each edge
		// Then find the normal to the surface at those points
		for (iEdge = 0; iEdge < 6; iEdge++) {
			// if there is an intersection on this edge
			if ((iEdgeFlags & (1 << iEdge)) != 0) {
				iVert0 = a2iTetrahedronEdgeConnection[iEdge][0];
				iVert1 = a2iTetrahedronEdgeConnection[iEdge][1];
				EdgeSplit split = new EdgeSplit(pasTetrahedronPosition[iVert0],
						pasTetrahedronPosition[iVert1]);
				long hashValue = split.hashValue();
				EdgeSplit foundSplit = splits.get(hashValue);
				if (foundSplit == null) {
					split.vid = vertCount++;
					Point3f pt3d = new Point3f();
					float fOffset = fGetOffset(split.pt1, split.pt2);
					float fInvOffset = 1.0f - fOffset;
					pt3d.x = resX
							* (fInvOffset * pasTetrahedronPosition[iVert0].x + fOffset
									* pasTetrahedronPosition[iVert1].x);
					pt3d.y = resY
							* (fInvOffset * pasTetrahedronPosition[iVert0].y + fOffset
									* pasTetrahedronPosition[iVert1].y);
					pt3d.z = resZ
							* (fInvOffset * pasTetrahedronPosition[iVert0].z + fOffset
									* pasTetrahedronPosition[iVert1].z);
					split.pt3d = pt3d;
					splits.put(hashValue, split);
				} else {
					split = foundSplit;
				}
				asEdgeVertex[iEdge] = split.vid;
			}
		}
		// Create list of triangles
		for (int iTriangle = 0; iTriangle < 5; iTriangle++) {
			if (a2iTetrahedronTriangles[iFlagIndex][3 * iTriangle] < 0)
				break;
			Triangle tri = new Triangle();
			for (int iCorner = 0; iCorner < 3; iCorner++) {
				iVertex = a2iTetrahedronTriangles[iFlagIndex][3 * iTriangle
						+ iCorner];
				tri.vids[iCorner] = asEdgeVertex[iVertex];
			}
			triangles.add(tri);
		}

	}

	/**
	 * Sets the iso-surface generation method.
	 * 
	 * @param method
	 *            the iso-surface generation method.
	 */
	public void setMethod(Method method) {
		this.method = method;
	}

	/**
	 * Sets the use image resolutions flag.
	 * 
	 * @param value
	 *            the if true, algorithm will use image resolutions
	 */
	public void setUseResolutions(boolean value) {
		this.useResolutions = value;
	}

	/**
	 * Sets the winding order.
	 * 
	 * @param winding
	 *            the winding order
	 */
	public void setWinding(Winding winding) {
		this.winding = winding;
	}

	/**
	 * Solve for iso-surface
	 * 
	 * @param vol
	 *            the volume
	 * @param isoLevel
	 *            the target iso-level
	 * 
	 * @return the iso-surface
	 */
	public EmbeddedSurface solve(ImageDataFloat vol, float isoLevel) {
		return solve(vol, isoLevel, -1);
	}

	/**
	 * Solve for iso-surface
	 * 
	 * @param vol
	 *            the volume
	 * @param isoLevel
	 *            the target iso-level
	 * 
	 * @return the iso-surface
	 */
	public EmbeddedSurface solve(ImageDataFloat vol, float isoLevel,
			int component) {
		volMat = vol.toArray3d();
		volsMat = vol.toArray4d();
		rows = vol.getRows();
		cols = vol.getCols();
		slices = vol.getSlices();
		this.component = component;
		vertCount = 0;
		this.isoLevel = isoLevel;
		// Set mesh resolutions
		if (useResolutions) {
			float[] res = vol.getHeader().getDimResolutions();
			resX = res[0];
			resY = res[1];
			resZ = res[2];
		} else {
			resX = resY = resZ = 1;
		}
		Hashtable<Long, EdgeSplit> splits = new Hashtable<Long, EdgeSplit>();
		LinkedList<Triangle> triangles = new LinkedList<Triangle>();
		if (!silent)
			setTotalUnits(rows);
		// Solve for iso-surface
		if (method == Method.MARCHING_CUBES || connectivityRule != null) {
			if (connectivityRule == null) {
				for (int i = 0; i < rows; i++) {
					for (int j = 0; j < cols; j++) {
						for (int k = 0; k < slices; k++) {
							triangulateUsingMarchingCubes(splits, triangles, i,
									j, k);
						}
					}
					if (!silent)
						incrementCompletedUnits();
				}
			} else {
				for (int i = 0; i < rows; i++) {
					for (int j = 0; j < cols; j++) {
						for (int k = 0; k < slices; k++) {
							triangulateUsingMarchingCubesConnectivityConsistent(
									splits, triangles, i, j, k);
						}
					}
					if (!silent)
						incrementCompletedUnits();
				}
			}
		} else if (method == Method.MARCHING_TETRAHEDRALS) {
			for (int i = 0; i < rows; i++) {
				for (int j = 0; j < cols; j++) {
					for (int k = 0; k < slices; k++) {
						triangulateUsingMarchingTetrahedrals(splits, triangles,
								i, j, k);
					}
				}
				if (!silent)
					incrementCompletedUnits();
			}
		}
		// Generate iso-surface from list of triangles
		int[] indexes = new int[triangles.size() * 3];
		Point3f[] points = new Point3f[splits.size()];
		int index = 0;
		if (winding == Winding.CLOCKWISE) {
			for (Triangle tri : triangles) {
				indexes[index++] = tri.vids[0];
				indexes[index++] = tri.vids[1];
				indexes[index++] = tri.vids[2];
			}
		} else if (winding == Winding.COUNTER_CLOCKWISE) {
			for (Triangle tri : triangles) {
				indexes[index++] = tri.vids[2];
				indexes[index++] = tri.vids[1];
				indexes[index++] = tri.vids[0];
			}
		}
		index = 0;
		for (EdgeSplit split : splits.values()) {
			index = split.vid;
			points[index] = split.pt3d;
		}
		volMat = null;
		// Create surface
		EmbeddedSurface surf = new EmbeddedSurface(points, indexes);
		surf.setName(vol.getName());
		if (!silent)
			markCompleted();
		return surf;
	}

	/**
	 * Triangulate using marching cubes.
	 * 
	 * @param splits
	 *            the edge splits
	 * @param triangles
	 *            the triangles
	 * @param x
	 *            the x position
	 * @param y
	 *            the y position
	 * @param z
	 *            the z position
	 */
	protected void triangulateUsingMarchingCubes(
			Hashtable<Long, EdgeSplit> splits, LinkedList<Triangle> triangles,
			int x, int y, int z) {
		int iVertex = 0;
		Point3i[] afCubeValue = new Point3i[8];
		int iFlagIndex = 0;
		for (iVertex = 0; iVertex < 8; iVertex++) {
			Point3i v = afCubeValue[iVertex] = new Point3i(clampRow(x
					+ a2fVertexOffset[iVertex][0]), clampColumn(y
					+ a2fVertexOffset[iVertex][1]), clampSlice(z
					+ a2fVertexOffset[iVertex][2]));
			// Find which vertices are inside of the surface and which are
			// outside
			if (getValue(v.x, v.y, v.z) <= isoLevel)
				iFlagIndex |= 1 << iVertex;

		}
		// Find which edges are intersected by the surface
		int iEdgeFlags = aiCubeEdgeFlags[iFlagIndex];

		// If the cube is entirely inside or outside of the surface, then there
		// will be no intersections
		if (iEdgeFlags == 0) {
			return;
		}
		EdgeSplit split;
		EdgeSplit[] asEdgeVertex = new EdgeSplit[12];
		// Find the point of intersection of the surface with each edge
		// Then find the normal to the surface at those points
		for (int iEdge = 0; iEdge < 12; iEdge++) {
			// if there is an intersection on this edge

			if ((iEdgeFlags & (1 << iEdge)) != 0) {
				Point3i v = afCubeValue[a2iEdgeConnection[iEdge][0]];
				if (getValue(v.x, v.y, v.z) <= isoLevel) {
					split = new EdgeSplit(
							afCubeValue[a2iEdgeConnection[iEdge][0]],
							afCubeValue[a2iEdgeConnection[iEdge][1]]);
				} else {
					split = new EdgeSplit(
							afCubeValue[a2iEdgeConnection[iEdge][1]],
							afCubeValue[a2iEdgeConnection[iEdge][0]]);
				}
				long hashValue = split.hashValue();
				EdgeSplit foundSplit = splits.get(hashValue);
				if (foundSplit == null) {
					split.vid = vertCount++;
					Point3f pt3d = new Point3f();
					float fOffset = fGetOffset(
							afCubeValue[a2iEdgeConnection[iEdge][0]],
							afCubeValue[a2iEdgeConnection[iEdge][1]]);
					pt3d.x = resX
							* (x + (a2fVertexOffset[a2iEdgeConnection[iEdge][0]][0] + fOffset
									* a2fEdgeDirection[iEdge][0]));
					pt3d.y = resY
							* (y + (a2fVertexOffset[a2iEdgeConnection[iEdge][0]][1] + fOffset
									* a2fEdgeDirection[iEdge][1]));
					pt3d.z = resZ
							* (z + (a2fVertexOffset[a2iEdgeConnection[iEdge][0]][2] + fOffset
									* a2fEdgeDirection[iEdge][2]));
					split.pt3d = pt3d;
					splits.put(hashValue, split);
				} else {
					split = foundSplit;
				}
				asEdgeVertex[iEdge] = split;
			}
		}
		// Generate list of triangles
		for (int iTriangle = 0; iTriangle < 5; iTriangle++) {
			if (a2iTriangleConnectionTable[iFlagIndex][3 * iTriangle] < 0)
				break;
			Triangle tri = new Triangle();
			int sign = 0;
			for (int iCorner = 0; iCorner < 3; iCorner++) {
				iVertex = a2iTriangleConnectionTable[iFlagIndex][3 * iTriangle
						+ iCorner];
				split = asEdgeVertex[iVertex];
				tri.vids[iCorner] = split.vid;
			}
			triangles.add(tri);
		}
	}

	/**
	 * Triangulate using marching cubes.
	 * 
	 * @param splits
	 *            the edge splits
	 * @param triangles
	 *            the triangles
	 * @param x
	 *            the x position
	 * @param y
	 *            the y position
	 * @param z
	 *            the z position
	 */
	protected void triangulateUsingMarchingCubesConnectivityConsistent(
			Hashtable<Long, EdgeSplit> splits, LinkedList<Triangle> triangles,
			int x, int y, int z) {
		int iVertex = 0;
		boolean flip = false;
		int[] aiCubeEdgeFlags = null;
		int[][] a2iTriangleConnectionTable = null;
		switch (connectivityRule) {
		case CONNECT_18_6:
			flip = true;
			aiCubeEdgeFlags = aiCubeEdgeFlagsCC618;
			a2iTriangleConnectionTable = a2iTriangleConnectionTableCC618;
			break;
		case CONNECT_26_6:
			flip = true;
			aiCubeEdgeFlags = aiCubeEdgeFlagsCC626;
			a2iTriangleConnectionTable = a2iTriangleConnectionTableCC626;
			break;
		case CONNECT_6_18:
			flip = false;
			aiCubeEdgeFlags = aiCubeEdgeFlagsCC618;
			a2iTriangleConnectionTable = a2iTriangleConnectionTableCC618;
			break;
		case CONNECT_6_26:
			flip = false;
			aiCubeEdgeFlags = aiCubeEdgeFlagsCC626;
			a2iTriangleConnectionTable = a2iTriangleConnectionTableCC626;
			break;
		}
		Point3i[] afCubeValue = new Point3i[8];
		int iFlagIndex = 0;
		for (iVertex = 0; iVertex < 8; iVertex++) {
			Point3i v = afCubeValue[iVertex] = new Point3i(clampRow(x
					+ a2fVertexOffset[iVertex][0]), clampColumn(y
					+ a2fVertexOffset[iVertex][1]), clampSlice(z
					+ a2fVertexOffset[iVertex][2]));
			// Find which vertices are inside of the surface and which are
			// outside
			if (flip) {
				if (getValue(v.x, v.y, v.z) > isoLevel)
					iFlagIndex |= 1 << iVertex;
			} else {
				if (getValue(v.x, v.y, v.z) <= isoLevel)
					iFlagIndex |= 1 << iVertex;
			}
		}
		// Find which edges are intersected by the surface
		int iEdgeFlags = aiCubeEdgeFlags[iFlagIndex];

		// If the cube is entirely inside or outside of the surface, then there
		// will be no intersections
		if (iEdgeFlags == 0) {
			return;
		}
		EdgeSplit split;
		EdgeSplit[] asEdgeVertex = new EdgeSplit[12];
		// Find the point of intersection of the surface with each edge
		// Then find the normal to the surface at those points
		for (int iEdge = 0; iEdge < 12; iEdge++) {
			// if there is an intersection on this edge

			if ((iEdgeFlags & (1 << iEdge)) != 0) {
				Point3i v = afCubeValue[a2iEdgeConnection[iEdge][0]];
				if (getValue(v.x, v.y, v.z) <= isoLevel) {
					split = new EdgeSplit(
							afCubeValue[a2iEdgeConnection[iEdge][0]],
							afCubeValue[a2iEdgeConnection[iEdge][1]]);
				} else {
					split = new EdgeSplit(
							afCubeValue[a2iEdgeConnection[iEdge][1]],
							afCubeValue[a2iEdgeConnection[iEdge][0]]);
				}
				long hashValue = split.hashValue();
				EdgeSplit foundSplit = splits.get(hashValue);
				if (foundSplit == null) {
					split.vid = vertCount++;
					Point3f pt3d = new Point3f();
					float fOffset = fGetOffset(
							afCubeValue[a2iEdgeConnection[iEdge][0]],
							afCubeValue[a2iEdgeConnection[iEdge][1]]);
					pt3d.x = resX
							* (x + (a2fVertexOffset[a2iEdgeConnection[iEdge][0]][0] + fOffset
									* a2fEdgeDirection[iEdge][0]));
					pt3d.y = resY
							* (y + (a2fVertexOffset[a2iEdgeConnection[iEdge][0]][1] + fOffset
									* a2fEdgeDirection[iEdge][1]));
					pt3d.z = resZ
							* (z + (a2fVertexOffset[a2iEdgeConnection[iEdge][0]][2] + fOffset
									* a2fEdgeDirection[iEdge][2]));
					split.pt3d = pt3d;
					splits.put(hashValue, split);
				} else {
					split = foundSplit;
				}
				asEdgeVertex[iEdge] = split;
			}
		}
		// Generate list of triangles
		for (int iTriangle = 0; iTriangle < 5; iTriangle++) {
			if (3 * iTriangle >= a2iTriangleConnectionTable[iFlagIndex].length)
				break;
			Triangle tri = new Triangle();
			if (flip) {
				for (int iCorner = 0; iCorner < 3; iCorner++) {
					iVertex = a2iTriangleConnectionTable[iFlagIndex][3
							* iTriangle + iCorner];
					split = asEdgeVertex[iVertex];
					tri.vids[iCorner] = split.vid;
				}
			} else {
				for (int iCorner = 0; iCorner < 3; iCorner++) {
					iVertex = a2iTriangleConnectionTable[iFlagIndex][3
							* iTriangle + iCorner];
					split = asEdgeVertex[iVertex];
					if (split == null) {
						System.err.println("Could not find " + iVertex
								+ " for " + iFlagIndex);
					}
					tri.vids[2 - iCorner] = split.vid;
				}
			}
			triangles.add(tri);
		}
	}

	/**
	 * Triangulate using marching tetrahedrals.
	 * 
	 * @param splits
	 *            the edge splits
	 * @param triangles
	 *            the triangles
	 * @param x
	 *            the x position
	 * @param y
	 *            the y position
	 * @param z
	 *            the z position
	 */
	protected void triangulateUsingMarchingTetrahedrals(
			Hashtable<Long, EdgeSplit> splits, LinkedList<Triangle> triangles,
			int x, int y, int z) {
		int iVertex, iTetrahedron, iVertexInACube;
		Point3i[] asTetrahedronPosition = new Point3i[4];
		float[] afCubeValue = new float[8];
		Point3i[] afCubePosition = new Point3i[8];
		for (iVertex = 0; iVertex < 8; iVertex++) {
			Point3i v = afCubePosition[iVertex] = new Point3i(clampRow(x
					+ a2fVertexOffset[iVertex][0]), clampColumn(y
					+ a2fVertexOffset[iVertex][1]), clampSlice(z
					+ a2fVertexOffset[iVertex][2]));
			// Find which vertices are inside of the surface and which are
			// outside
			afCubeValue[iVertex] = getValue(v.x, v.y, v.z);

		}
		Point3i p;
		for (iTetrahedron = 0; iTetrahedron < 6; iTetrahedron++) {
			for (iVertex = 0; iVertex < 4; iVertex++) {
				iVertexInACube = a2iTetrahedronsInACube[iTetrahedron][iVertex];
				asTetrahedronPosition[iVertex] = p = new Point3i();
				p.x = afCubePosition[iVertexInACube].x;
				p.y = afCubePosition[iVertexInACube].y;
				p.z = afCubePosition[iVertexInACube].z;
			}
			marchTetrahedral(splits, triangles, asTetrahedronPosition);
		}
	}
}
