package edu.jhu.ece.iacl.algorithms.registration;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import Jama.Matrix;

import edu.jhmi.rad.medic.libraries.ImageFunctionsPublic;
import edu.jhmi.rad.medic.utilities.NumericsPublic;
import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataByte;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataInt;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;

public class RegistrationUtilities {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.18 $");
	}

	// Buffers
	//public static int i0, j0, k0, i1, j1, k1;
	//public static float dx, dy, dz, hx, hy, hz;
	//public static int ch;
	//public static float xx, yy, zz;

	public static int calculateBin(double interval, double minVal, double val) {
		return (int) Math.round((val - minVal) / interval);
	}

	/* imA is assumed to be from 0 and numBin-1 */
	/* imB is assumed to be from 0 and numBin-1 */
	public static void JointHistogram3D(ImageData imA, ImageData imB,
			int numBin, int[] roi, int[][] jointHist) {
		int i, j, k, binA, binB;
		for (j = 0; j < numBin; j++) {
			for (i = 0; i < numBin; i++) {
				jointHist[j][i] = 0;
			}
		}
		for (k = roi[4]; k <= roi[5]; k++) {
			for (j = roi[2]; j <= roi[3]; j++) {
				for (i = roi[0]; i <= roi[1]; i++) {
					binA = imA.getUByte(i, j, k);
					if (binA >= numBin)
						binA = numBin - 1;
					if (binA < 0)
						binA = 0;
					binB = imB.getUByte(i, j, k);
					if (binB >= numBin)
						binB = numBin - 1;
					if (binB < 0)
						binB = 0;
					jointHist[binA][binB] += 1;
				}
			}
		}
	}

	public static void JointHistogram3D(ImageData imA[], ImageData imB[],
			int channel, int numBin, int[] roi, int[][][] jointHist) {
		JointHistogram3D(imA[channel], imB[channel], numBin, roi,
				jointHist[channel]);
	}

	/* the max value is assumed to be numBin-1, and min value 0 */
	public static void Histogram3D(ImageData im, int numBin, int[] roi,
			int[] hist) {
		int i, j, k;
		int bin;
		for (i = 0; i < numBin; i++)
			hist[i] = 0;
		for (k = roi[4]; k <= roi[5]; k++) {
			for (j = roi[2]; j <= roi[3]; j++) {
				for (i = roi[0]; i <= roi[1]; i++) {
					bin = im.getUByte(i, j, k);
					if (bin >= numBin)
						bin = numBin - 1;
					if (bin < 0)
						bin = 0;
					hist[bin] += 1;					
				}
			}
		}
	}

	public static void Histogram3D(ImageData im[], int channel, int numBin,
			int[] roi, int[][] hist) {
		Histogram3D(im[channel], numBin, roi, hist[channel]);

	}
	
	public static ImageDataFloat DeformImageFloat3D(ImageData im,
			ImageDataFloat DF) {
		ImageDataFloat vol = new ImageDataFloat(im.getName() + "_def", im
				.getRows(), im.getCols(), im.getSlices());
		vol.setHeader(im.getHeader());
		DeformImage3D(im, vol, DF, im.getRows(), im.getCols(), im.getSlices(), RegistrationUtilities.InterpolationType.TRILINEAR);
		return vol;
	}
	 
	public static ImageData DeformImage3D(ImageData im, ImageDataFloat DF, int type) {
		ImageData vol = im.mimic();
		vol.setName(im.getName() + "_def");
		DeformImage3D(im, vol, DF, im.getRows(), im.getCols(), im.getSlices(), type);
		return vol;
	}

	public static void DeformImage3D(ImageData im, ImageData deformedIm,
			ImageDataFloat DF, int sizeX, int sizeY, int sizeZ, int type) {
		int i, j, k;
		float[][][][] DFM = DF.toArray4d();
		
		
		System.out.format("Deforming Image Now\n");
		for (i = 0; i < sizeX; i++) for (j = 0; j < sizeY; j++) for (k = 0; k < sizeZ; k++) {
			if (DFM[i][j][k][0] != 0 || DFM[i][j][k][1] != 0 || (DF.getComponents() > 2 && DFM[i][j][k][2] != 0)) {
				
				if(DF.getComponents() == 2){ //Def field is 2D in 3D format
					deformedIm.set(i, j, 0, Interpolation(im, sizeX, sizeY, sizeZ, i + DFM[i][j][k][0], 
							j + DFM[i][j][k][1], 0, type));
				}
				else if(DF.getComponents() == 3){ //Def field is 3D 
					deformedIm.set(i, j, k, Interpolation(im, sizeX, sizeY, sizeZ, i + DFM[i][j][k][0], 
							j + DFM[i][j][k][1], k+ DFM[i][j][k][2], type));
				}
			} else {
				deformedIm.set(i, j, k, im.getDouble(i, j, k));
			}
		}
	}

	public static ImageData DeformImage(ImageData im, ImageDataFloat DF, int type) {
		
		ImageData vol = im.clone();
		vol.setName(im.getName() + "_def");

		//System.out.format("check" +  DF.getComponents() +" " +DF.getSlices() + " " + im.getSlices());
		//check if in 2D or 3D format
		if(DF.getComponents() <= 1 && DF.getSlices() == 2 && im.getSlices() <= 1)
			DeformImage2D(im, vol, DF, im.getRows(), im.getCols(), type);
		else 
			DeformImage3D(im, vol, DF, im.getRows(), im.getCols(), im.getSlices(), type);

		return vol;
	}
	
	public static void DeformImage2D(ImageData im, ImageData deformedIm,
			ImageDataFloat DF, int sizeX, int sizeY, int type) {
		int i, j, k;
		float[][][] DFM = DF.toArray3d();
		for (i = 0; i < sizeX; i++) for (j = 0; j < sizeY; j++){
			if (DFM[i][j][0] != 0 || DFM[i][j][1] != 0) {
				deformedIm.set(i, j, 0, Interpolation(im, sizeX, sizeY, 1, i + DFM[i][j][0], j
						+ DFM[i][j][1], 0, type));
			} else {
				deformedIm.set(i, j, im.getDouble(i, j));
			}
		}
	}
	
	public static void DeformImage1D(double[] im, double[] deformedIm,
			double[] DF) {
		int i;
		
		for (i = 0; i < im.length; i++){
			if (DF[i] != 0) {
				deformedIm[i] = TrilinearInterpolation1D(im, i + DF[i]);
			} else {
				deformedIm[i] = im[i];
			}
		}
	}
	
	//newDF is the output deformation field
	public static void DeformationFieldResample3DM(float[][][][] oldDF,
			ImageDataFloat newDF) {

		int oldSizeX = oldDF.length;
		int oldSizeY = oldDF[0].length;
		int oldSizeZ = oldDF[0][0].length;


		int newSizeX = newDF.getRows();
		int newSizeY = newDF.getCols();
		int newSizeZ = newDF.getSlices();

		DeformationFieldResample3DM(oldDF, newDF, oldSizeX, oldSizeY, oldSizeZ,
				newSizeX, newSizeY, newSizeZ);

	}
	
	//newDF is the output deformation field
	public static void DeformationFieldResample3DM(float[][][][] oldDF,
			ImageDataFloat newDF, int oldSizeX, int oldSizeY, int oldSizeZ,
			int newSizeX, int newSizeY, int newSizeZ) {

		int i, j, k, dim;
		float x, y, z;
		float ratio[] = new float[3];
		//		ratio[0] = (float) Math.floor((float) newSizeX / oldSizeX + 0.5);
		//		ratio[1] = (float) Math.floor((float) newSizeY / oldSizeY + 0.5);
		//		ratio[2] = (float) Math.floor((float) newSizeZ / oldSizeZ + 0.5);

		ratio[0] = (float) newSizeX / oldSizeX;
		ratio[1] = (float) newSizeY / oldSizeY;
		ratio[2] = (float) newSizeZ / oldSizeZ;
		float[][][][] newDFM = newDF.toArray4d();


		float[][][] tmpX = new float[oldSizeX][oldSizeY][oldSizeZ];
		float[][][] tmpY = new float[oldSizeX][oldSizeY][oldSizeZ];
		float[][][] tmpZ = new float[oldSizeX][oldSizeY][oldSizeZ];
		for(i=0; i < oldSizeX; i++) for(j=0; j < oldSizeY; j++) for(k=0; k < oldSizeZ; k++){
			tmpX[i][j][k] = oldDF[i][j][k][0];
			tmpY[i][j][k] = oldDF[i][j][k][1];
			tmpZ[i][j][k] = oldDF[i][j][k][2];
		}
		ImageDataFloat oldDFX = new ImageDataFloat(tmpX);
		ImageDataFloat oldDFY = new ImageDataFloat(tmpY);
		ImageDataFloat oldDFZ = new ImageDataFloat(tmpZ);

		for (i = 0; i < newSizeX; i++) {
			x = ((i+1) / ratio[0])-1;
			for (j = 0; j < newSizeY; j++) {
				y = ((j+1) / ratio[1])-1;
				for (k = 0; k < newSizeZ; k++) {
					z = ((k+1) / ratio[2])-1;
					newDFM[i][j][k][0] = (float) (Interpolation(oldDFX, oldSizeX, oldSizeY, oldSizeZ, x, y, z, InterpolationType.TRILINEAR) * ratio[0]);
					newDFM[i][j][k][1] = (float) (Interpolation(oldDFY, oldSizeX, oldSizeY, oldSizeZ, x, y, z, InterpolationType.TRILINEAR) * ratio[1]);
					newDFM[i][j][k][2] = (float) (Interpolation(oldDFZ, oldSizeX, oldSizeY, oldSizeZ, x, y, z,InterpolationType.TRILINEAR) * ratio[2]);
				}
			}
		}
	}

	//Scatter point interp using Shepard's method
	//scatterpoint list array should represent {xLocation, yLocation, zLocation, xComponentOfVector,yComponentOfVector,zComponentOfVector}
	public static void scatterPointInterp(ImageData outField, List<double[]> scatterPointsList,int p, double R){

		double[] newVec = new double[3];
		double weightSum;
		double currentWeight=0;
		double[] currentPoint;
		Iterator<double[]> itr;
		//int listN = scatterPointsList.size();
		for(int i = 0; i < outField.getRows(); i++)
			for(int j = 0; j < outField.getCols(); j++)
				for(int k = 0; k < outField.getSlices(); k++){

					weightSum=0;	
					for(int c = 0; c < 3; c++) newVec[c] = 0;
					itr = scatterPointsList.iterator();
					while(itr.hasNext()){
						currentPoint = itr.next();

						//use p-norm
						currentWeight = Math.pow(Math.pow((i-currentPoint[0]),p) 
								+ Math.pow((j-currentPoint[1]),p) 
								+ Math.pow((k-currentPoint[2]),p),1/(double)p);

						if(currentWeight == 0){
							for(int c = 0; c < 3; c++) outField.set(i,j,k,c, currentPoint[c+3]);
							break;
						}else if(currentWeight < R){
							//}else if(true){
							//if(q<2){
							//	System.out.format(q+"\n");
							//	System.out.format(currentWeight+"\n");
							//System.out.format((i-currentPoint[0])+" "+(j-currentPoint[1])+" "+(k-currentPoint[2])+"\n");
							//System.out.format(Math.pow((i-currentPoint[0]),p)+" "+Math.pow((j-currentPoint[1]),p)+" "+Math.pow((k-currentPoint[2]),p)+"\n");

							//}

							//if the location exist in list, then set field as that
							//currentWeight = 1/currentWeight;

							currentWeight = Math.pow((R-currentWeight)/R*currentWeight , 2);
							weightSum += currentWeight;							
							for(int c = 0; c < 3; c++) newVec[c] += currentWeight*currentPoint[c+3];
						}

					}

					if(currentWeight != 0 && weightSum !=0){
						for(int c = 0; c < 3; c++) outField.set(i,j,k,c, newVec[c]/weightSum);
					}

				}

	}



	public static float[][][] computeJacobianDet(float[][][][] def){
		int nx = def.length;
		int ny = def[0].length;
		int nz = def[0][0].length;
		float[][][] jacmap = new float[nx][ny][nz];
		float[][] jac = new float[3][3];
		float Dmx, Dmy, Dmz, Dpx, Dpy, Dpz, D0x, D0y, D0z;
		for(int x=0; x<nx; x++) for(int y=0; y<ny; y++) for(int z=0; z<nz; z++){
			// derivatives of the x comp wrt x, y, and z
			int LY = (y == 0) ? 1 : 0;
			int HY = (y == (ny - 1)) ? 1 : 0;
			int LZ = (z == 0) ? 1 : 0;
			int HZ = (z == (nz - 1)) ? 1 : 0;
			int LX = (x == 0) ? 1 : 0;
			int HX = (x == (nx - 1)) ? 1 : 0;

			// DERIVATIVES OF THE X COMPONENT 
			/*	Dmx = def[x][y][z][0] - def[x - 1 + LX][y][z][0];
			Dpx = def[x + 1 - HX][y][z][0] - def[x][y][z][0];
			Dmy = def[x][y][z][0] - def[x][y - 1 + LY][z][0];
			Dpy = def[x][y + 1 - HY][z][0] - def[x][y][z][0];
			Dmz = def[x][y][z][0] - def[x][y][z - 1 + LZ][0];
			Dpz = def[x][y][z + 1 - HZ][0] - def[x][y][z][0];*/
			// central differences
			D0x = (def[x + 1 - HX][y][z][0] - def[x - 1 + LX][y][z][0]) / 2;
			D0y = (def[x][y + 1 - HY][z][0] - def[x][y - 1 + LY][z][0]) / 2;
			D0z = (def[x][y][z + 1 - HZ][0] - def[x][y][z - 1 + LZ][0]) / 2;

			//set the appropriate values in the jacobian matrix
			//we add one before we are working with displacement field, but we want the jacobian of the deformation
			jac[0][0]=1+D0x;
			jac[0][1]=D0y;
			jac[0][2]=D0z;

			// DERIVATIVES OF THE Y COMPONENT 
			/*	Dmx = def[x][y][z][1] - def[x - 1 + LX][y][z][1];
			Dpx = def[x + 1 - HX][y][z][1] - def[x][y][z][1];
			Dmy = def[x][y][z][1] - def[x][y - 1 + LY][z][1];
			Dpy = def[x][y + 1 - HY][z][1] - def[x][y][z][1];
			Dmz = def[x][y][z][1] - def[x][y][z - 1 + LZ][1];
			Dpz = def[x][y][z + 1 - HZ][1] - def[x][y][z][1];*/
			// central differences
			D0x = (def[x + 1 - HX][y][z][1] - def[x - 1 + LX][y][z][1]) / 2;
			D0y = (def[x][y + 1 - HY][z][1] - def[x][y - 1 + LY][z][1]) / 2;
			D0z = (def[x][y][z + 1 - HZ][1] - def[x][y][z - 1 + LZ][1]) / 2;
			//set the appropriate values in the jacobian matrix
			//we add one before we are working with displacement field, but we want the jacobian of the deformation
			jac[1][0]=D0x;
			jac[1][1]=1+D0y;
			jac[1][2]=D0z;

			// DERIVATIVES OF THE Z COMPONENT 
			/*	Dmx = def[x][y][z][2] - def[x - 1 + LX][y][z][2];
			Dpx = def[x + 1 - HX][y][z][2] - def[x][y][z][2];
			Dmy = def[x][y][z][2] - def[x][y - 1 + LY][z][2];
			Dpy = def[x][y + 1 - HY][z][2] - def[x][y][z][2];
			Dmz = def[x][y][z][2] - def[x][y][z - 1 + LZ][2];
			Dpz = def[x][y][z + 1 - HZ][2] - def[x][y][z][2];*/
			// central differences
			D0x = (def[x + 1 - HX][y][z][2] - def[x - 1 + LX][y][z][2]) / 2;
			D0y = (def[x][y + 1 - HY][z][2] - def[x][y - 1 + LY][z][2]) / 2;
			D0z = (def[x][y][z + 1 - HZ][2] - def[x][y][z - 1 + LZ][2]) / 2;

			//set the appropriate values in the jacobian matrix
			//we add one before we are working with displacement field, but we want the jacobian of the deformation
			jac[2][0]=D0x;
			jac[2][1]=D0y;
			jac[2][2]=1+D0z;

			//set the value equal to the determinant
			jacmap[x][y][z]= NumericsPublic.determinant3D(jac);

		}
		return jacmap;
	}

	public static float computeMaxJacobianDetRBF(float[][][] def, double[] localCoarseGradient){


		int nx = def.length;
		int ny = def[0].length;
		int nz = def[0][0].length;
		float max=0;
		float maxdiff=0;
		float temp;
		float[][] jac = new float[3][3];
		float Dmx, Dmy, Dmz, Dpx, Dpy, Dpz, D0x, D0y, D0z;
		for(int x=0; x<nx; x++) for(int y=0; y<ny; y++) for(int z=0; z<nz; z++){
			// derivatives of the x comp wrt x, y, and z
			int LY = (y == 0) ? 1 : 0;
			int HY = (y == (ny - 1)) ? 1 : 0;
			int LZ = (z == 0) ? 1 : 0;
			int HZ = (z == (nz - 1)) ? 1 : 0;
			int LX = (x == 0) ? 1 : 0;
			int HX = (x == (nx - 1)) ? 1 : 0;

			// DERIVATIVES OF THE X COMPONENT 
			// central differences
			D0x = (def[x + 1 - HX][y][z] - def[x - 1 + LX][y][z]) / 2;
			D0y = (def[x][y + 1 - HY][z] - def[x][y - 1 + LY][z]) / 2;
			D0z = (def[x][y][z + 1 - HZ] - def[x][y][z - 1 + LZ]) / 2;


			//set the appropriate values in the jacobian matrix
			jac[0][0]=D0x*(float)localCoarseGradient[0];
			jac[0][1]=D0y*(float)localCoarseGradient[0];
			jac[0][2]=D0z*(float)localCoarseGradient[0];

			//set the appropriate values in the jacobian matrix
			jac[1][0]=D0x*(float)localCoarseGradient[1];
			jac[1][1]=D0y*(float)localCoarseGradient[1];
			jac[1][2]=D0z*(float)localCoarseGradient[1];

			//set the appropriate values in the jacobian matrix
			jac[2][0]=D0x*(float)localCoarseGradient[2];
			jac[2][1]=D0y*(float)localCoarseGradient[2];
			jac[2][2]=D0z*(float)localCoarseGradient[2];


			//set the value equal to the determinant
			temp= Math.abs(NumericsPublic.determinant3D(jac));
			/*
			if(Math.abs(D0x) > maxdiff){
				maxdiff = D0x;
				System.out.format(maxdiff+" "+temp+"\n");
				for(int i =0; i < 3; i++){ for(int j =0; j < 3; j++){
					System.out.format(jac[i][j]+" ");}
					System.out.format("\n");
					}

			}

			if(Math.abs(D0y) > maxdiff){
				maxdiff = D0y;
				System.out.format(maxdiff+" "+temp+"\n");
				for(int i =0; i < 3; i++){ for(int j =0; j < 3; j++){
					System.out.format(jac[i][j]+" ");}
					System.out.format("\n");
					}

			}

			if(Math.abs(D0z) > maxdiff){
				maxdiff = D0z;
				System.out.format(maxdiff+" "+temp+"\n");
				for(int i =0; i < 3; i++){ for(int j =0; j < 3; j++){
					System.out.format(jac[i][j]+" ");}
					System.out.format("\n");
					}

			}
			 */

			if (temp > max) max = temp;

		}
		return max;

	}

	public static float[] computeMaxAndMinJacobianDet(float[][][][] def){

		int nx = def.length;
		int ny = def[0].length;
		int nz = def[0][0].length;
		float[] maxAndMin = new float[4];
		maxAndMin[0] = Float.MIN_VALUE;
		maxAndMin[1] = Float.MAX_VALUE;
		maxAndMin[2] = Float.MIN_VALUE;
		maxAndMin[3] = Float.MAX_VALUE;
		float temp;
		float[][] jac = new float[3][3];
		for(int x=0; x<nx; x++) for(int y=0; y<ny; y++) for(int z=0; z<nz; z++){
			// derivatives of the x comp wrt x, y, and z
			int LY = (y == 0) ? 1 : 0;
			int HY = (y == (ny - 1)) ? 1 : 0;
			int LZ = (z == 0) ? 1 : 0;
			int HZ = (z == (nz - 1)) ? 1 : 0;
			int LX = (x == 0) ? 1 : 0;
			int HX = (x == (nx - 1)) ? 1 : 0;

			// central differences
			jac[0][0] = (def[x + 1 - HX][y][z][0] - def[x - 1 + LX][y][z][0]) / 2;
			jac[0][1] = (def[x][y + 1 - HY][z][0] - def[x][y - 1 + LY][z][0]) / 2;
			jac[0][2] = (def[x][y][z + 1 - HZ][0] - def[x][y][z - 1 + LZ][0]) / 2;

			jac[1][0] = (def[x + 1 - HX][y][z][1] - def[x - 1 + LX][y][z][1]) / 2;
			jac[1][1] = (def[x][y + 1 - HY][z][1] - def[x][y - 1 + LY][z][1]) / 2;
			jac[1][2] = (def[x][y][z + 1 - HZ][1] - def[x][y][z - 1 + LZ][1]) / 2;

			jac[2][0] = (def[x + 1 - HX][y][z][2] - def[x - 1 + LX][y][z][2]) / 2;
			jac[2][1] = (def[x][y + 1 - HY][z][2] - def[x][y - 1 + LY][z][2]) / 2;
			jac[2][2] = (def[x][y][z + 1 - HZ][2] - def[x][y][z - 1 + LZ][2]) / 2;

			//set the value equal to the determinant
			temp= NumericsPublic.determinant3D(jac);
			if (temp > maxAndMin[0]) maxAndMin[0] = temp;
			if (temp < maxAndMin[1]) maxAndMin[1] = temp;

			jac[0][0]+=1;
			jac[1][1]+=1;
			jac[2][2]+=1;

			//set the value equal to the determinant
			temp= NumericsPublic.determinant3D(jac);
			if (temp > maxAndMin[2]) maxAndMin[2] = temp;
			if (temp < maxAndMin[3]) maxAndMin[3] = temp;

		}
		return maxAndMin;
	}


	public static double DoubleDistance(double z0, double y0, double x0,
			double z1, double y1, double x1) {
		double tmp = 0.0;

		tmp += (x0 - x1) * (x0 - x1);
		tmp += (y0 - y1) * (y0 - y1);
		tmp += (z0 - z1) * (z0 - z1);

		return Math.sqrt(tmp);
	}


	//This RBF comess from the following reference:
	//See: Rohde, G.K. et al. (TMI 2003) - "The adaptive bases algorithm for intensity-based nonrigid image registration"
	public static float RBF(float r) {
		if (r > 1 || r < 0)
			return 0;
		return (float) (Math.pow(1 - r, 4) * (3 * Math.pow(r, 3) + 12 * r * r
				+ 16 * r + 4));
	}

	public static float RBF3D(int cx, int cy, int cz, int x, int y, int z,
			float scale) {

		return RBF((float) (Math.sqrt((cx - x) * (cx - x) + (cy - y) * (cy - y)
				+ (cz - z) * (cz - z)) / scale));
	}


	/*
	public static double Interpolation(float[][][][] oldVM,	int XN, int YN, int ZN, 
			int dim, double x, double y, double z, int type) {

		return Interpolation(oldV, XN, YN, ZN, x, y,z, type);
	}
	 */

	public static double Interpolation(ImageData oldV, double x, double y, double z, int type) {
		return Interpolation(oldV, oldV.getRows(), oldV.getCols(), oldV.getSlices(),x,y, z, type);
	}


	public static double Interpolation(ImageData oldV, int XN, int YN,
			int ZN, double x, double y, double z, int type) {
		switch(type){

		case InterpolationType.TRILINEAR: //Trilinear
			return TrilinearInterpolation(oldV, XN, YN,	ZN, x, y, z);
		case InterpolationType.NEAREST_NEIGHTBOR: //Nearest Neighbor
			return NNInterpolation(oldV, XN, YN, ZN, x, y, z);
		default:
			return 0;
		}

	}

	public static class InterpolationType{
		public static final int TRILINEAR = 0;
		public static final int NEAREST_NEIGHTBOR = 1;
	}

	// Nearest Neighbor interpolation
	public static double NNInterpolation(ImageData oldV, int XN, int YN, int ZN,
			double x, double y, double z) {
		
		
		if (!((x < 0) || (x > (XN - 1)) || (y < 0) || (y > (YN - 1))
				|| (z < 0) || (z > (ZN - 1)))) {
		double d000 = 0.0, d001 = 0.0, d010 = 0.0, d011 = 0.0;
		double d100 = 0.0, d101 = 0.0, d110 = 0.0, d111 = 0.0;
		double value = 0.0, dist = 0.0;
		int x0 = (int) x, y0 = (int) y, z0 = (int) z;
		int x1 = x0 + 1, y1 = y0 + 1, z1 = z0 + 1;

		
		//If exactly on the boundary, then we set it as the boundary
		if (x == (double) (XN - 1))
			x1 = XN - 1;
		if (y == (double) (YN - 1))
			y1 = YN - 1;
		if (z == (double) (ZN - 1))
			z1 = ZN - 1;

		d000 = DoubleDistance(z, y, x, (double) z0, (double) y0,
					(double) x0);
			d100 = DoubleDistance(z, y, x, (double) z0, (double) y0,
					(double) x1);
			d010 = DoubleDistance(z, y, x, (double) z0, (double) y1,
					(double) x0);
			d110 = DoubleDistance(z, y, x, (double) z0, (double) y1,
					(double) x1);

			d001 = DoubleDistance(z, y, x, (double) z1, (double) y0,
					(double) x0);
			d101 = DoubleDistance(z, y, x, (double) z1, (double) y0,
					(double) x1);
			d011 = DoubleDistance(z, y, x, (double) z1, (double) y1,
					(double) x0);
			d111 = DoubleDistance(z, y, x, (double) z1, (double) y1,
					(double) x1);

			dist = d000;
			value = oldV.getDouble(x0, y0, z0);

			if (dist > d001) {
				dist = d001;
				value = oldV.getDouble(x0, y0, z1);
			}

			if (dist > d010) {
				dist = d010;
				value = oldV.getDouble(x0, y1, z0);
			}

			if (dist > d011) {
				dist = d011;
				value = oldV.getDouble(x0, y1, z1);
			}

			if (dist > d100) {
				dist = d100;
				value = oldV.getDouble(x1, y0, z0);
			}

			if (dist > d101) {
				dist = d101;
				value = oldV.getDouble(x1, y0, z1);
			}

			if (dist > d110) {
				dist = d110;
				value = oldV.getDouble(x1, y1, z0);
			}

			if (dist > d111) {
				dist = d111;
				value = oldV.getDouble(x1, y1, z1);
			}

			return value;
		} else {
			return 0;
		}
	}	

	// Nearest Neighbor interpolation Boolean
	public static boolean NNInterpolationBool(boolean[][][] oldV, int XN, int YN, int ZN,
			double x, double y, double z) {
		
		if (!((x < 0) || (x > (XN - 1)) || (y < 0) || (y > (YN - 1))
				|| (z < 0) || (z > (ZN - 1)))) {
			
		double d000 = 0.0, d001 = 0.0, d010 = 0.0, d011 = 0.0;
		double d100 = 0.0, d101 = 0.0, d110 = 0.0, d111 = 0.0;
		boolean value = false;
		double dist = 0.0;
		int x0 = (int) x, y0 = (int) y, z0 = (int) z;
		int x1 = x0 + 1, y1 = y0 + 1, z1 = z0 + 1;

		/*
		if (x == (double) (XN - 1))
			x1 = XN - 1;
		if (y == (double) (YN - 1))
			y1 = YN - 1;
		if (z == (double) (ZN - 1))
			z1 = ZN - 1;
		 */


			d000 = DoubleDistance(z, y, x, (double) z0, (double) y0,
					(double) x0);
			d100 = DoubleDistance(z, y, x, (double) z0, (double) y0,
					(double) x1);
			d010 = DoubleDistance(z, y, x, (double) z0, (double) y1,
					(double) x0);
			d110 = DoubleDistance(z, y, x, (double) z0, (double) y1,
					(double) x1);

			d001 = DoubleDistance(z, y, x, (double) z1, (double) y0,
					(double) x0);
			d101 = DoubleDistance(z, y, x, (double) z1, (double) y0,
					(double) x1);
			d011 = DoubleDistance(z, y, x, (double) z1, (double) y1,
					(double) x0);
			d111 = DoubleDistance(z, y, x, (double) z1, (double) y1,
					(double) x1);

			dist = d000;
			value = oldV[x0][y0][z0];

			if (dist > d001) {
				dist = d001;
				value = oldV[x0][y0][z1];
			}

			if (dist > d010) {
				dist = d010;
				value = oldV[x0][y1][z0];
			}

			if (dist > d011) {
				dist = d011;
				value = oldV[x0][y1][z1];
			}

			if (dist > d100) {
				dist = d100;
				value = oldV[x1][y0][z0];
			}

			if (dist > d101) {
				dist = d101;
				value = oldV[x1][y0][z1];
			}

			if (dist > d110) {
				dist = d110;
				value = oldV[x1][y1][z0];
			}

			if (dist > d111) {
				dist = d111;
				value = oldV[x1][y1][z1];
			}

			return value;
		} else {
			return false;
		}
	}	


	public static int[] NNInterpolationLoc(ImageData oldV, int XN, int YN, int ZN,
			double x, double y, double z) {
		int[] valAndLoc = new int[4];
		double d000 = 0.0, d001 = 0.0, d010 = 0.0, d011 = 0.0;
		double d100 = 0.0, d101 = 0.0, d110 = 0.0, d111 = 0.0;
		double dist = 0.0;
		int x0 = (int) x, y0 = (int) y, z0 = (int) z;
		int x1 = x0 + 1, y1 = y0 + 1, z1 = z0 + 1;

		/*
		if (x == (double) (XN - 1))
			x1 = XN - 1;
		if (y == (double) (YN - 1))
			y1 = YN - 1;
		if (z == (double) (ZN - 1))
			z1 = ZN - 1;
		 */

		d000 = DoubleDistance(z, y, x, (double) z0, (double) y0,
				(double) x0);
		d100 = DoubleDistance(z, y, x, (double) z0, (double) y0,
				(double) x1);
		d010 = DoubleDistance(z, y, x, (double) z0, (double) y0,
				(double) x0);
		d110 = DoubleDistance(z, y, x, (double) z0, (double) y1,
				(double) x1);

		d001 = DoubleDistance(z, y, x, (double) z1, (double) y0,
				(double) x0);
		d101 = DoubleDistance(z, y, x, (double) z1, (double) y0,
				(double) x1);
		d011 = DoubleDistance(z, y, x, (double) z1, (double) y0,
				(double) x0);
		d111 = DoubleDistance(z, y, x, (double) z1, (double) y1,
				(double) x1);

		dist = d000;
		valAndLoc[1] = x0;
		valAndLoc[2] = y0;
		valAndLoc[3] = z0;

		if (dist > d001) {
			dist = d001;
			valAndLoc[1] = x0;
			valAndLoc[2] = y0;
			valAndLoc[3] = z1;
		}

		if (dist > d010) {
			dist = d010;
			valAndLoc[1] = x0;
			valAndLoc[2] = y1;
			valAndLoc[3] = z0;
		}

		if (dist > d011) {
			dist = d011;
			valAndLoc[1] = x0;
			valAndLoc[2] = y1;
			valAndLoc[3] = z1;
		}

		if (dist > d100) {
			dist = d100;
			valAndLoc[1] = x1;
			valAndLoc[2] = y0;
			valAndLoc[3] = z0;
		}

		if (dist > d101) {
			dist = d101;
			valAndLoc[1] = x1;
			valAndLoc[2] = y0;
			valAndLoc[3] = z1;
		}

		if (dist > d110) {
			dist = d110;
			valAndLoc[1] = x1;
			valAndLoc[2] = y1;
			valAndLoc[3] = z0;
		}

		if (dist > d111) {
			dist = d111;
			valAndLoc[1] = x1;
			valAndLoc[2] = y1;
			valAndLoc[3] = z1;
		}


		if (((x0 < 0) || (x1 > (XN - 1)) || (y0 < 0) || (y1 > (YN - 1))
				|| (z0 < 0) || (z1 > (ZN - 1)))) {
			if(x0 < 0) valAndLoc[1] = 0;
			if(x1 > (XN - 1)) valAndLoc[1] = XN-1;
			if(y0 < 0) valAndLoc[2] = 0;
			if(y1 > (YN - 1)) valAndLoc[2] = YN-1;
			if(z0 < 0) valAndLoc[3] = 0;
			if(z1 > (ZN - 1))valAndLoc[3] = ZN -1;
		}

		valAndLoc[0] = oldV.getInt(valAndLoc[1],valAndLoc[2],valAndLoc[3]);

		return valAndLoc;
	}	

	public static double TrilinearInterpolation(ImageData oldV, int XN, int YN,
			int ZN, double x, double y, double z) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		if (x < 0 || x > (XN - 1) || y < 0 || y > (YN - 1) || z < 0
				|| z > (ZN - 1)) {
			return 0;
		} else {
			j1 = (int) Math.ceil(x);
			i1 = (int) Math.ceil(y);
			k1 = (int) Math.ceil(z);
			j0 = (int) Math.floor(x);
			i0 = (int) Math.floor(y);
			k0 = (int) Math.floor(z);
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;

			// Introduce more variables to reduce computation
			hx = 1.0f - dx;
			hy = 1.0f - dy;
			hz = 1.0f - dz;
			// Optimized below
			return   (((oldV.getDouble(j0, i0, k0) * hx + oldV.getDouble(j1, i0, k0) * dx) * hy 
					+ (oldV.getDouble(j0, i1, k0) * hx + oldV.getDouble(j1, i1, k0) * dx) * dy) * hz 
					+ ((oldV.getDouble(j0, i0, k1) * hx + oldV.getDouble(j1, i0, k1) * dx) * hy 
							+ (oldV.getDouble(j0, i1, k1) * hx + oldV.getDouble(j1, i1, k1) * dx) * dy)* dz);
		}
	}

	
	public static double TrilinearInterpolation1D(double[] oldV, double x) {
		int j0, j1;
		double dx, hx;
		if (x < 0 || x > (oldV.length - 1)) {
			return 0;
		} else {
			j1 = (int) Math.ceil(x);
			j0 = (int) Math.floor(x);
			dx = x - j0;
			hx = 1.0f - dx;
			return oldV[j0] * hx + oldV[j1] * dx;
		}
	}

	public static double TrilinearInterpolateDefField(ImageData defField, int XN, int YN,
			int ZN, double x, double y, double z, int c) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		if (x < 0 || x > (XN - 1) || y < 0 || y > (YN - 1) || z < 0
				|| z > (ZN - 1)) {
			return 0;
		} else {
			j1 = (int) Math.ceil(x);
			i1 = (int) Math.ceil(y);
			k1 = (int) Math.ceil(z);
			j0 = (int) Math.floor(x);
			i0 = (int) Math.floor(y);
			k0 = (int) Math.floor(z);
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;

			// Introduce more variables to reduce computation
			hx = 1.0f - dx;
			hy = 1.0f - dy;
			hz = 1.0f - dz;
			// Optimized below
			return   (((defField.getDouble(j0, i0, k0,c) * hx + defField.getDouble(j1, i0, k0,c) * dx) * hy 
					+ (defField.getDouble(j0, i1, k0,c) * hx + defField.getDouble(j1, i1, k0,c) * dx) * dy) * hz 
					+ ((defField.getDouble(j0, i0, k1,c) * hx + defField.getDouble(j1, i0, k1,c) * dx) * hy 
							+ (defField.getDouble(j0, i1, k1,c) * hx + defField.getDouble(j1, i1, k1,c) * dx) * dy)* dz);

		}
	}

	public static class IndexedFloat implements Comparable<IndexedFloat> {
		public int index;
		public float val;

		public IndexedFloat(float val, int index) {
			this.index = index;
			this.val = val;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Comparable#compareTo(java.lang.Object)
		 */
		public int compareTo(IndexedFloat o) {
			return (int) Math.signum(val - o.val);
		}
	}

	public static int[] QKSort2(float[] arr) {
		ArrayList<IndexedFloat> list = new ArrayList<IndexedFloat>(arr.length);
		int index = 0;
		for (int i = 0; i < arr.length; i++) {
			list.add(new RegistrationUtilities.IndexedFloat(arr[i], i));
		}
		Collections.sort(list);
		int[] brr = new int[arr.length];
		for (IndexedFloat in : list) {
			arr[index] = in.val;
			brr[index++] = in.index;
		}
		return brr;
	}

	public static double VectorNormalization(double[] v, int n) {
		int i;
		double ret = 0;
		double thre = 0.0000001;
		for (i = 0; i < n; i++) {
			ret += v[i] * v[i];
		}
		ret = Math.sqrt(ret);
		if (ret < thre) {
			for (i = 0; i < n; i++) {
				v[i] = 0;
				ret = 0;
			}
		} else {
			for (i = 0; i < n; i++) {
				v[i] /= ret;
			}
		}
		return ret;
	}

	public static double NMI(int[][] histA, int[][] histB, int[][][] histAB,
			int channel, int numBin) {

		return NMI(histA[channel], histB[channel], histAB[channel], numBin);
	}

	public static double NMI(int[] histA, int[] histB, int[][] histAB,
			int numBin) {
		int i = 0, j = 0;
		double HA = 0, HB = 0, HAB = 0;
		int numVoxelA = 0, numVoxelB = 0, numVoxelAB = 0;
		double tmp = 0;
		for (i = 0; i < numBin; i++) {
			//System.out.format(histA[i] + "\n");
			numVoxelA += histA[i];
			numVoxelB += histB[i];
			for (j = 0; j < numBin; j++) {
				numVoxelAB += histAB[j][i];
			}
		}
		for (i = 0; i < numBin; i++) {
			if (histA[i] > 0) {
				tmp = ((double) histA[i]) / numVoxelA;
				HA -= tmp * Math.log(tmp);
			}
			if (histB[i] > 0) {
				tmp = ((double) histB[i]) / numVoxelB;
				HB -= tmp * Math.log(tmp);
			}
			for (j = 0; j < numBin; j++) {
				if (histAB[j][i] > 0) {
					tmp = ((double) histAB[j][i]) / numVoxelAB;
					HAB -= tmp * Math.log(tmp);
				}
			}
		}

		//System.out.format(HA+" "+HB+" "+HAB+" "+numVoxelAB+"\n");
		if (HA == 0 && HB == 0 && HAB == 0)
			return 2;
		else
			return (HA + HB) / HAB;
	}

	public static double MI(int[] histA, int[] histB, int[][] histAB,
			int numBin) {
		int i = 0, j = 0;
		double HA = 0, HB = 0, HAB = 0;
		int numVoxelA = 0, numVoxelB = 0, numVoxelAB = 0;
		double tmp = 0;
		for (i = 0; i < numBin; i++) {
			//System.out.format(histA[i] + "\n");
			numVoxelA += histA[i];
			numVoxelB += histB[i];
			for (j = 0; j < numBin; j++) {
				numVoxelAB += histAB[j][i];
			}
		}
		for (i = 0; i < numBin; i++) {
			if (histA[i] > 0) {
				tmp = ((double) histA[i]) / numVoxelA;
				HA -= tmp * Math.log(tmp);
			}
			if (histB[i] > 0) {
				tmp = ((double) histB[i]) / numVoxelB;
				HB -= tmp * Math.log(tmp);
			}
			for (j = 0; j < numBin; j++) {
				if (histAB[j][i] > 0) {
					tmp = ((double) histAB[j][i]) / numVoxelAB;
					HAB -= tmp * Math.log(tmp);
				}
			}
		}

		//System.out.format(HA+" "+HB+" "+HAB+" "+numVoxelAB+"\n");
		if (HA == 0 && HB == 0 && HAB == 0)
			return 2;
		else
			return (HA + HB) - HAB;
	}

	//calculate bounding box of the two images
	static public int[] calculateBoundingBox(ImageData sub, ImageData tar, int ext) {
		int k, j, i;
		int XN, YN, ZN;
		int boundingBox[] = new int[6];
		XN = sub.getRows();
		YN = sub.getCols();
		ZN = sub.getSlices();

		boundingBox[1] = 0;
		boundingBox[0] = XN;
		boundingBox[3] = 0;
		boundingBox[2] = YN;
		boundingBox[5] = 0;
		boundingBox[4] = ZN;

		int count=0;
		for (i = 0; i < XN; i++){
			for (j = 0; j < YN; j++){
				for (k = 0; k < ZN; k++){
					// if (subject.data[k][j][i][ch] > 0 ||
					// target.data[k][j][i][ch] > 0)
					if ((Math.abs(sub.getDouble(i, j, k)) > 0.0000001) ||
							(Math.abs(tar.getDouble(i, j, k)) > 0.0000001))
						// changed to match Xiao Feng's abra implementation
					{
						count++;
						if (i < boundingBox[0])
							boundingBox[0] = i;
						if (i > boundingBox[1])
							boundingBox[1] = i;
						if (j < boundingBox[2])
							boundingBox[2] = j;
						if (j > boundingBox[3])
							boundingBox[3] = j;
						if (k < boundingBox[4])
							boundingBox[4] = k;
						if (k > boundingBox[5])
							boundingBox[5] = k;
					}
				}
			}
		}



		boundingBox[0]=Math.max(0, boundingBox[0]-ext); 			
		boundingBox[1]=Math.min(XN-1, boundingBox[1]+ext);
		boundingBox[2]=Math.max(0, boundingBox[2]-ext); 			
		boundingBox[3]=Math.min(YN-1, boundingBox[3]+ext);
		boundingBox[4]=Math.max(0, boundingBox[4]-ext); 			
		boundingBox[5]=Math.min(ZN-1, boundingBox[5]+ext);

		return boundingBox;

		/*for (i = 0; i < 6; i++) {
			System.out.format("bb[%d]:%d\n",i,boundingBox[i]);
		}*/

	}

	// rescale the image data to fall between 0 and bins-1
	static public void normalize(ImageData sub, double minVals, double maxVals, int bins) {
		int k, j, i;
		float intervals = ((float) (maxVals - minVals)) / (bins-1);
		int XN = sub.getRows();
		int YN = sub.getCols();
		int ZN = sub.getSlices();

		for (i = 0; i < XN; i++) {
			for (j = 0; j < YN; j++) {
				for (k = 0; k < ZN; k++) {
					sub.set(i, j, k, RegistrationUtilities.calculateBin(intervals,
							minVals, sub.getDouble(i, j, k)));
				}
			}
		}
	}

	//Calculate Max and Min of the two images
	static public double[] calculateMinAndMaxVals(ImageData sub, ImageData tar) {

		int ch;
		int CH;
		int i, j, k;
		int x = 0, y = 0, z = 0;
		int mx = 0, my = 0, mz = 0;

		int XN, YN, ZN;
		XN = sub.getRows();
		YN = sub.getCols();
		ZN = sub.getSlices();
		double MinandMaxValsD[] = new double[2];
		double max = Double.NEGATIVE_INFINITY;
		double min = Double.POSITIVE_INFINITY;
		for (i = 0; i < XN; i++) {
			for (j = 0; j < YN; j++) {
				for (k = 0; k < ZN; k++) {

					if (sub.getDouble(i, j, k) > max) {
						max = sub.getDouble(i, j, k);
						mx = i;
						my = j;
						mz = k;
					}
					if (sub.getDouble(i, j, k) < min) {
						min = sub.getDouble(i, j, k);
						x = i;
						y = j;
						z = k;
					}
					if (tar.getDouble(i, j, k) > max) {
						max = tar.getDouble(i, j, k);
						mx = i;
						my = j;
						mz = k;
					}
					if (tar.getDouble(i, j, k) < min) {
						min = tar.getDouble(i, j, k);
						x = i;
						y = j;
						z = k;
					}

				}
			}
		}

		MinandMaxValsD[0] = min;
		MinandMaxValsD[1] = max;
		//System.out.format("Max: " + MinandMaxValsD[0] + " Min" + MinandMaxValsD[1] + "\n");
		return MinandMaxValsD;

	}

	public static ImageData[] split4DImageDataIntoArray(ImageData volIn){

		int XN = volIn.getRows();
		int YN = volIn.getCols();
		int ZN = volIn.getSlices();
		int CH = Math.max(1,volIn.getComponents());

		ImageData[] volInArray = new ImageData[CH];

		for (int c = 0; c < CH; c++){
			if(volIn.getType() == VoxelType.BOOLEAN){

				volInArray[c] = new ImageDataInt(volIn.getName(),XN, YN, ZN);
				for (int i = 0; i < XN; i++) for (int j = 0; j < YN; j++)for (int k = 0; k < ZN; k++){
					if(volIn.getBoolean(i, j, k, c))volInArray[c].set(i, j, k, 1);
					else volInArray[c].set(i, j, k, 0);
				}
			}
			else{
				volInArray[c] = new ImageDataMipav(volIn.getName(),volIn.getType(),XN, YN, ZN);
				for (int i = 0; i < XN; i++) for (int j = 0; j < YN; j++)for (int k = 0; k < ZN; k++){
					volInArray[c].set(i, j, k, volIn.get(i, j, k, c));
				}
			}
			volInArray[c].setHeader(volIn.getHeader());
		}

		return volInArray;
	}

	public static ImageData combineImageDataArrayTo4D(ImageData volArray[]){

		int XN = volArray[0].getRows();
		int YN = volArray[0].getCols();
		int ZN = volArray[0].getSlices();
		int CH = volArray.length;

		ImageData volOut = new ImageDataMipav(volArray[0].getName(),volArray[0].getType(),XN, YN, ZN, CH);
		for (int c = 0; c < CH; c++){
			for (int i = 0; i < XN; i++) for (int j = 0; j < YN; j++)for (int k = 0; k < ZN; k++){
				volOut.set(i, j, k, c, volArray[c].getDouble(i, j, k, c));
			}
		}
		volOut.setHeader(volArray[0].getHeader());
		return volOut;
	}


	//Reorder the dimensions in the array
	public static ImageData reorderDimension(ImageData inField, boolean changeXYZTtoTXYZ){
		int XN = inField.getRows();
		int YN = inField.getCols();
		int ZN = inField.getSlices();
		int TN = inField.getComponents();
		ImageData outField;
		if(changeXYZTtoTXYZ)
			outField = new ImageDataFloat(TN,XN,YN,ZN);
		else
			outField = new ImageDataFloat(YN,ZN,TN,XN);
		for (int i = 0; i < XN; i ++)
			for (int j = 0; j < YN; j ++)
				for (int k = 0; k < ZN; k ++){
					for (int c = 0; c < TN; c ++){
						if(changeXYZTtoTXYZ)
							outField.set(c,i,j,k, inField.getFloat(i, j, k,c));
						else
							outField.set(j,k,c,i, inField.getFloat(i, j, k,c));
					}
				}
		
		outField.setHeader(inField.getHeader());
		return outField;
	}

	//converts Deformation to Displacement field, and vice-verse
	public static void convertField(ImageData field, boolean defToDisp){
		int convertFactor = 1;
		//changing from deformation to Displacement
		if (defToDisp) convertFactor = -1;

		for (int i = 0; i < field.getRows(); i ++)
			for (int j = 0; j < field.getCols(); j ++)
				for (int k = 0; k < field.getSlices(); k ++){
					field.set(i,j,k,0, field.getFloat(i, j, k, 0)+ convertFactor*i);
					field.set(i,j,k,1, field.getFloat(i, j, k, 1)+ convertFactor*j);
					field.set(i,j,k,2, field.getFloat(i, j, k, 2)+ convertFactor*k);
				}
	}

	public static ImageData findFieldMagnitude(ImageData inField){
		int XN = inField.getRows();
		int YN = inField.getCols();
		int ZN = inField.getSlices();
		int CN = inField.getComponents();
		ImageData outField = new ImageDataFloat(XN, YN, ZN);
		double sumOfSq;
		for (int i = 0; i < XN; i ++)
			for (int j = 0; j < YN; j ++)
				for (int k = 0; k < ZN; k ++){
					sumOfSq = 0;
					for (int c = 0; c < CN; c ++){
						sumOfSq += inField.getDouble(i, j, k, c) * inField.getDouble(i, j, k, c);
					}
					outField.set(i,j,k, Math.sqrt(sumOfSq));
				}
		return outField;
	}
	
	
	//Finds Matrix Exponential of the field using "Scale and Squaring" approach using N Steps
	public static ImageData findFieldExponentialSS(ImageData inField, int inNSteps){
		int XN = inField.getRows();
		int YN = inField.getCols();
		int ZN = inField.getSlices();
		int CN = inField.getComponents();
		ImageData outField = inField.mimic();
		//first scale by 1/(2^N)
		double scaleVal = Math.pow(2,inNSteps);
		for (int i = 0; i < XN; i ++)
			for (int j = 0; j < YN; j ++)
				for (int k = 0; k < ZN; k ++)
					for (int c = 0; c < CN; c ++){
						outField.set(i,j,k,c,inField.getDouble(i, j, k, c)/scaleVal);
					}
		//then square(composition operator) N times. 
		for (int i = 0 ; i < inNSteps; i++ ){
			outField = composeDeformations(outField, outField);
		}
		return outField;
	}
	
	//Finds Matrix Logarithm of the field using "Inverse Scale and Squaring" approach using N Steps
	public static ImageData findFieldLogarithmISS(ImageData inField, int inNSteps){
		int XN = inField.getRows();
		int YN = inField.getCols();
		int ZN = inField.getSlices();
		int CN = inField.getComponents();
		ImageData outField = inField.mimic();
		
		//for inField(F) first square root(composition operator) N times to get F^(2^-N) 
		for (int i = 0 ; i < inNSteps; i++ ){
			outField = defFieldSquareRoot(outField);
		}

		
		//Since F^(2^-N) is close to zero, approximate log(F^(2^-N)) = F^(2^-N) - ID
		
		//then scale by 2^N
		double scaleVal = Math.pow(2,inNSteps);
		for (int i = 0; i < XN; i ++)
			for (int j = 0; j < YN; j ++)
				for (int k = 0; k < ZN; k ++)
					for (int c = 0; c < CN; c ++){
						outField.set(i,j,k,c,inField.getDouble(i, j, k, c)/scaleVal);
					}
					
					
		return outField;
	}

	//use gradient descent to solve for square root (composition wise) of deformation field
	public static ImageData defFieldSquareRoot(ImageData inField){
		double maxItr = 100;
		double stepSize = .01;
		ImageData oldField = inField.mimic();
		ImageData newField = inField.mimic();
		
		int XN = inField.getRows();
		int YN = inField.getCols();
		int ZN = inField.getSlices();
		int CN = 3;
		
		
		for(int itr = 0; itr < maxItr; itr++){
			
			for(int i = 0; i < XN; i++)
				for(int j = 0; j < YN; j++)
					for(int k = 0; k < ZN; k++)
						for(int c = 0; c < CN; c++){
							oldField.set(i,j,k, c, newField.get(i, j, k, c));
							
						
					}
			
		}
		
		return newField;
	}
	
	//Inverts a deformation field
	public static void invertDisplacementField(ImageData disp, ImageData inv){

		int maxIter, iter, XN, YN, ZN;
		float threshold, deltax, deltay, deltaz, deltanorm, deltanorm_old, x, y, z, xnew, ynew, znew;

		ImageData[] dispFieldArray = split4DImageDataIntoArray(disp);

		System.out.format("Start iter\n");
		maxIter = 10000;
		threshold = 0.0001f;
		double[] v= new double[3];
		double[] v_new= new double[3];
		XN = disp.getRows();
		YN = disp.getCols();
		ZN = disp.getSlices();
		int[] previousConverged = new int[3];
		for(int i = 0; i < XN; i++){
			for(int j = 0; j < YN; j++){ 
				for(int k = 0; k < ZN; k++){

					/*
					iter=0;
					deltanorm = 1;
					while (deltanorm > .001 && iter != maxIter){
						v_new[0] = -TrilinearInterpolation(dispFieldArray[0], XN, YN, ZN, i+v[0], j+v[1], k+v[2]); 
						v_new[1] = -TrilinearInterpolation(dispFieldArray[1], XN, YN, ZN, i+v[0], j+v[1], k+v[2]); 
						v_new[2] = -TrilinearInterpolation(dispFieldArray[2], XN, YN, ZN, i+v[0], j+v[1], k+v[2]);
						deltanorm = 0;
						for(int c=0; c <3; c++){
							deltanorm += Math.abs(v[c]-v_new[c]);
							v[c]=v_new[c];
						}
						iter++;
					}
					if(iter == maxIter){
						System.out.format("Reached MaxIter "+"delta:" + deltanorm+"itr"+iter+"\n");
						iter=0;
						deltanorm = 1;
						v[0] = inv.getDouble(previousConverged[0] ,previousConverged[1] ,previousConverged[2],0);
						v[1] = inv.getDouble(previousConverged[0] ,previousConverged[1] ,previousConverged[2],1);
						v[2] = inv.getDouble(previousConverged[0] ,previousConverged[1] ,previousConverged[2],2);
						while (deltanorm > .001 && iter != maxIter){
							v_new[0] = -TrilinearInterpolation(dispFieldArray[0], XN, YN, ZN, i+v[0], j+v[1], k+v[2]); 
							v_new[1] = -TrilinearInterpolation(dispFieldArray[1], XN, YN, ZN, i+v[0], j+v[1], k+v[2]); 
							v_new[2] = -TrilinearInterpolation(dispFieldArray[2], XN, YN, ZN, i+v[0], j+v[1], k+v[2]);
							deltanorm = 0;
							for(int c=0; c <3; c++){
								deltanorm += Math.abs(v[c]-v_new[c]);
								v[c]=v_new[c];
							}
							iter++;
						}
						System.out.format("Redo "+"delta:" + deltanorm+"itr"+iter+"\n");
					}else{
						previousConverged[0] =i;
						previousConverged[1] =j;
						previousConverged[2] =k;
					}
					for(int c=0; c <3; c++){
						inv.set(i,j,k,c,v[c]);
						v[c]=0;
						iter=0;
					}
					 */
					//Lotta's
					//initialize variables, we start with 1 so that the while loop doesn't shut down immediately
					deltax = 1.0f; 
					deltay = 1.0f;  
					deltaz = 1.0f; 
					deltanorm = (float) Math.sqrt(deltax*deltax + deltay*deltay + deltaz*deltaz); 
					deltanorm_old = deltanorm;
					x = (float)i - dispFieldArray[0].getFloat(i, j, k, 0);
					y = (float)j - dispFieldArray[1].getFloat(i, j, k, 1);  
					z = (float)k - dispFieldArray[2].getFloat(i, j, k, 2);
					iter = 0; 
					//////////////////////

					
					//keep iterating until the magnitude of delta is below some threshold
					while(deltanorm > threshold){   
					//while((deltanorm > threshold || (deltanorm_old - deltanorm >= 0.0f && deltanorm > 0.01))){   
						xnew = x; 
						ynew = y; 
						znew = z; 

						if(xnew > 0.0f && ynew > 0.0f && znew > 0.0f && xnew < (float)XN-1 && ynew < (float)YN-1 && znew < (float)ZN-1){
							deltax = (float)i -(xnew + (float)TrilinearInterpolation(dispFieldArray[0], XN, YN, ZN, xnew, ynew, znew)); 
							deltay = (float)j -(ynew + (float)TrilinearInterpolation(dispFieldArray[1], XN, YN, ZN, xnew, ynew, znew)); 
							deltaz = (float)k -(znew +  (float)TrilinearInterpolation(dispFieldArray[2], XN, YN, ZN, xnew, ynew, znew)); 
						} 
						else{ 
							//might want to consider doing a mirror? or extend? instead of set as zero
							deltax = 0.0f; 
							deltay = 0.0f;  
							deltaz = 0.0f; 
						} 

						deltanorm_old = deltanorm;
						deltanorm = (float) Math.sqrt(deltax*deltax + deltay*deltay + deltaz*deltaz); 
						x += deltax/2.0f; 
						y += deltay/2.0f;  
						z += deltaz/2.0f;  
						iter ++; 
						if(iter == maxIter){  
							//	    printf("Algorithm did not converge, deltanorm = %5.4f\n", deltanorm);	
							deltanorm_old = deltanorm;		
							xnew = (float)i + inv.getFloat(previousConverged[0],previousConverged[1],previousConverged[2],0);
							ynew = (float)j + inv.getFloat(previousConverged[0],previousConverged[1],previousConverged[2],1);
							znew = (float)k + inv.getFloat(previousConverged[0],previousConverged[1],previousConverged[2],2);
							deltax = (float)i -(xnew + (float)TrilinearInterpolation(dispFieldArray[0], XN, YN, ZN, xnew, ynew, znew)); 
							deltay = (float)j -(ynew + (float)TrilinearInterpolation(dispFieldArray[1], XN, YN, ZN, xnew, ynew, znew)); 
							deltaz = (float)k -(znew +  (float)TrilinearInterpolation(dispFieldArray[2], XN, YN, ZN, xnew, ynew, znew));
							deltanorm = (float) Math.sqrt(deltax*deltax + deltay*deltay + deltaz*deltaz); 
							if(deltanorm < deltanorm_old){
								x = xnew;
								y = ynew;
								z = znew;
								//	      printf("new used\n");
							}
							//	    printf("deltanorm = %5.4f\n", deltanorm);			
							deltanorm = 0.0f;
						}else{
							previousConverged[0] =i;
							previousConverged[1] =j;
							previousConverged[2] =k;
						}
					} 

					inv.set(i,j,k,0,x- (float)i); 
					inv.set(i,j,k,1,y - (float)j); 
					inv.set(i,j,k,2,z - (float)k);

				}
			}
		}

	}	

	//compose two (pullback) deformations field
	//return composeDef(X) = firstDef(secondDef(x))
	//Remember that we are working with pullback fields!
	//Composed field is a pull back that takes a point in the subject space of firstDef(x) and pulls back to target space of secondDef(x) 
	//For image I, applying composeDef(x) is equivalent to applying firstDef(x) and then applying secondDef(x) consecutively 
	public static ImageData composeDeformations(ImageData firstDef, ImageData secondDef){
		int chN = 3;		
		int XN = firstDef.getRows();
		int YN = firstDef.getCols();
		int ZN = firstDef.getSlices();
		double[] firstVec = new double[chN];
		double[] secondVec = new double[chN];

		ImageData newDef = firstDef.clone();
		
		//First split deformation field into three 3D arrays, for easy interpolation
		ImageData[] firstDefArray = split4DImageDataIntoArray(firstDef);

		for(int i = 0; i < XN; i++)
			for(int j = 0; j < YN; j++)
				for(int k = 0; k < ZN; k++){

					for(int c = 0; c < chN; c++) secondVec[c] = secondDef.getDouble(i, j, k, c); 

					//get first deformation vector at where the second deformation vector is pointing
					for(int c = 0; c < chN; c++) firstVec[c] = RegistrationUtilities.Interpolation(firstDefArray[c], XN, YN, ZN, 
							secondVec[0]+i, secondVec[1]+j, secondVec[2]+k, InterpolationType.TRILINEAR); 

					for(int c = 0; c < chN; c++) newDef.set(i,j,k, c, secondVec[c]+firstVec[c]);

				}
		return newDef;			

	}

	/*
	public static ImageData approxHomeoDefFieldoFTopAtlas(ImageData inDeformationField, ImageDataByte topologyAtlas, String connectivity){
		
		ImageData[] inDeformationFieldSplit = RegistrationUtilities.split4DImageDataIntoArray(inDeformationField);
				
				DigitalHomeomorphism homeoDefApprox = new DigitalHomeomorphism(topologyAtlas.toArray3d(), 
						new ImageDataFloat(atlasToSubDefSplit[0]).toArray3d(),
						new ImageDataFloat(atlasToSubDefSplit[1]).toArray3d(),
						new ImageDataFloat(atlasToSubDefSplit[2]).toArray3d(),
						topAtlas.getRows(), topAtlas.getCols(), topAtlas.getSlices(),
						topAtlas.getHeader().getDimResolutions()[0], 
						topAtlas.getHeader().getDimResolutions()[1],
						topAtlas.getHeader().getDimResolutions()[2],
						connMap.get(inParamConnectivity.getValue()),
						null);
				homeoDefApprox.computeHomeomorphicTransformAndImage();
				atlasToSubDefSplit[0] = new ImageDataFloat(homeoDefApprox.getTransformedFieldX());
				atlasToSubDefSplit[1] = new ImageDataFloat(homeoDefApprox.getTransformedFieldY());
				atlasToSubDefSplit[2] = new ImageDataFloat(homeoDefApprox.getTransformedFieldZ());
				atlasToSubDef = RegistrationUtilities.combineImageDataArrayTo4D(atlasToSubDefSplit);
		
		
	}
	*/
	//combines a transformation matrix and a deformation fields
	public static ImageData combineTransAndDef(Matrix inTransformMatrix, ImageData inDeformationField){
		
		ImageData currentDefField;
		//create deformation from transformation matrix
		currentDefField=RegistrationUtilities.createDefFieldFromTransMatrix(inTransformMatrix, inDeformationField);
		//combine with deformation field
		currentDefField=RegistrationUtilities.composeDeformations(currentDefField, inDeformationField);
		return currentDefField;
		
	}
	
	
	public static ImageData createDefFieldFromTransMatrix(Matrix newTrans, ImageData referenceImage){
		
		int XN = referenceImage.getRows();
		int YN = referenceImage.getCols();
		int ZN = referenceImage.getSlices();
		int chN = 3;
		float[] dimRes = referenceImage.getHeader().getDimResolutions();
		
		
		//double[] currentVec = new double[chN];
		double[] newVec = new double[chN];
		ImageData newDef = new ImageDataFloat(XN,YN,ZN, chN);
		newDef.setHeader(referenceImage.getHeader());
		for(int i = 0; i < XN; i++)
			for(int j = 0; j < YN; j++)
				for(int k = 0; k < ZN; k++){
					
					newVec[0] = i*dimRes[0];
					newVec[1] = j*dimRes[1];
					newVec[2] = k*dimRes[2];
					                	
                    Matrix v = new Matrix(new double[]{newVec[0],newVec[1],newVec[2],1},4);
                    Matrix vp = newTrans.solve(v);
                    for(int c = 0; c < chN; c++) {
                        newVec[c] = vp.get(c,0) - newVec[c]; 
                    }
                    
					for(int c = 0; c < chN; c++) newDef.set(i,j,k,c, newVec[c]/dimRes[c]);
					
				}
		
		
		
		return newDef;
		
	}
	
	//Finds all labels in an Image. 
	//Input: Labeled/Segmentation Image
	//Output: List of Labels
	public static ArrayList<Integer> findLabels(ImageData truth){
		ArrayList<Integer> labels = new ArrayList<Integer>();
		if(truth!=null){
			int rows = truth.getRows();
			int cols = truth.getCols();
			int slcs = truth.getSlices();
			for(int i=0; i<rows; i++){
				for(int j=0; j<cols; j++){
					for(int k=0; k<slcs; k++){
						if(truth.getComponents() > 1){
							for(int c=0; c<truth.getComponents(); c++){
								if(!labels.contains(new Integer(truth.getInt(i,j,k,c)))){
									labels.add(new Integer(truth.getInt(i,j,k,c)));
								}	
							}
						}
						else{
							if(!labels.contains(new Integer(truth.getInt(i,j,k)))){
								labels.add(new Integer(truth.getInt(i,j,k)));
							}
						}
					}
				}
			}
		}else{
			System.err.println("No data!");
		}
		labels.trimToSize();
		System.out.println("Found Labels: ");
		System.out.println(labels);
		return labels;
	}

	
	//Pad with Pad Method
	//padMethods=0 ->"Extend Zeros"
	//padMethods=1 -> "Extend Ends", 
	//padMethods=2 -> "Mirror Edges"
	public static List<ImageData> padImageList(List<ImageData> imgsToPad, int padSize, int padMethod){
		int rows, cols, slcs, i, j, k;

		ImageData newVol, origVol;
		List<ImageData> paddedImages = new ArrayList<ImageData>();
		for (int ch = 0; ch < imgsToPad.size(); ch ++) {
			origVol = imgsToPad.get(ch);
			rows = origVol.getRows();
			cols = origVol.getCols();
			slcs = origVol.getSlices();
			newVol = origVol.mimic(rows + 2*padSize, cols + 2*padSize, slcs + 2*padSize, origVol.getComponents());
			for (i = 0; i < rows; i++) for (j = 0; j < cols; j++) for (k = 0; k < slcs; k++) {
				newVol.set(i + padSize, j + padSize, k + padSize, origVol.getDouble(i, j, k));
			}

			
			//Pad with Pad Method
			if(padMethod != 0){
				for (i = 0; i < rows + 2*padSize; i++) for (j = 0; j < cols + 2*padSize; j++) for (k = 0; k < slcs + 2*padSize; k++) {
					if(j >= padSize && j < cols+padSize && k >= padSize && k < slcs+padSize){
						if(padMethod == 1){
							if (i < padSize) newVol.set(i , j , k , origVol.getDouble(0, j-padSize, k-padSize));
							if (i >= rows+padSize) newVol.set(i , j , k , origVol.getDouble(rows-1, j-padSize, k-padSize));

							if (j < padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, 0, k-padSize));
							if (j >= cols+padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, cols-1, k-padSize));

							if (k < padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, j-padSize, 0));
							if (k >= slcs+padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, j-padSize, slcs-1));
						}
						else if(padMethod == 2){
							if (i < padSize) newVol.set(i , j , k , origVol.getDouble(padSize-1-i, j-padSize, k-padSize));
							if (i >= rows+padSize) newVol.set(i , j , k , origVol.getDouble(2*rows+padSize-1-i, j-padSize, k-padSize));
							
							if (j < padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, padSize-1-j, k-padSize));
							if (j >= cols+padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, 2*cols+padSize-1-j, k-padSize));
							
							if (k < padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, j-padSize, padSize-1-k));
							if (k >= slcs+padSize) newVol.set(i , j , k , origVol.getDouble(i-padSize, j-padSize, 2*slcs+padSize-1-k));

						}
					}
				}
			}

			newVol.setHeader(origVol.getHeader());
			newVol.setName(origVol.getName() + "_pad");
			origVol.dispose();
			paddedImages.add(newVol);
		}

		return paddedImages;
	}
	
	public static void threshAtRobustMaxAndMin(List<ImageData> imgVec, float RobustMinThresh, float RobustMaxThresh) {

		float robustMax, robustMin;
		int XN, YN, ZN, i, j, k;
		

		ImageData currentImage;
		for (int ch = 0; ch < imgVec.size(); ch++) {
			currentImage = imgVec.get(ch);
			XN = currentImage.getRows();
			YN = currentImage.getCols();
			ZN = currentImage.getSlices();

			if (RobustMaxThresh > 0.00001f) {
				robustMax = ImageFunctionsPublic.robustMaximum(currentImage, RobustMaxThresh, 2, XN, YN, ZN);
				for (i = 0; i < XN; i++) for (j = 0; j < YN; j++) for (k = 0; k < ZN; k++) {
					if (currentImage.getFloat(i, j, k) > robustMax) {
						currentImage.set(i, j, k, robustMax);
					}
				}
			}

			if (RobustMinThresh > 0.00001f) {
				robustMin = ImageFunctionsPublic.robustMinimum(currentImage, RobustMinThresh, 2, XN, YN, ZN);
				for (i = 0; i < XN; i++) for (j = 0; j < YN; j++) for (k = 0; k < ZN; k++) {
					if (currentImage.getFloat(i, j, k) < robustMin) {
						currentImage.set(i, j, k, robustMin);
					}
				}
			}
		}
	}
	
	//SyN- Vectors elements are split into scalar maps at each components so c1=xxxx... c2=yyyy... c3=zzzz...
	//Uses header info.
	//Similar to VABRA except scaled by resolution
	//First direction is negative, because Syn reads in images R-L
	public static ImageData synToVABRA(ImageData synField){
		ImageData out = synField.mimic();
		
		int XN = out.getRows();
		int YN = out.getCols();
		int ZN = out.getSlices();
		
		float[] dimRes= synField.getHeader().getDimResolutions();
		
		for (int i = 0; i < XN; i++) for (int j = 0; j < YN; j++) for (int k = 0; k < ZN; k++) {
			for (int c = 0; c < 3; c++){
				//    if(c==0){
				//    	out.set(i, j, k, c, -synField.getDouble(i, j, k, c)/dimRes[c]);
				//    }else{
				    	out.set(i, j, k, c, synField.getDouble(i, j, k, c)/dimRes[c]);
				//    }
				    
			}
			
		}
		return out;
	}
	
	
	//DRAMMS Vectors are presented sequentially. X/Y components flipped. So stored as YXZYXZYXZYXZ...
	//does not use header info.
	public static ImageData drammsToVABRA(ImageData drammsField){
		
		ImageData out = drammsField.mimic();
		int XN = out.getRows();
		int YN = out.getCols();
		int ZN = out.getSlices();
		int counter=0;
		int ii,jj,kk,cc;

		for (int c = 0; c < 3; c++){
			for (int k = 0; k < ZN; k++) for (int j = 0; j < YN; j++) for (int i = 0; i < XN; i++){
				
				//changing from sequence indexing into volume indexing
				ii= (counter/3)%XN;
				jj=(counter/(XN*3))%YN;
				kk=(counter/(XN*YN*3))%ZN;
				cc= counter%3;
				//X/Y are flipped in DRAMMS fields, so we have to swap x component(c=0) and y components(c=1) 
				//when making ours:				
				if (cc==1)cc=0;
				else if (cc==0)cc=1;
				
			out.set(ii,jj,kk,cc, drammsField.getDouble(i, j, k, c));
			counter++;
			
			}
		}
		return out;
	}
	
	/*

	public static ImageData DeformImage3DNN(ImageData im, ImageDataFloat DF) {
		ImageData vol = im.mimic();
		vol.setName(im.getName() + "_def");
		DeformImage3DNN(im, vol, DF, im.getRows(), im.getCols(), im.getSlices());
		return vol;
	}

	public static void DeformImage3DNN(ImageData im, ImageData deformedIm,
			ImageDataFloat DF, int sizeX, int sizeY, int sizeZ) {
		int i, j, k;
		float[][][][] DFM = DF.toArray4d();
		for (i = 0; i < sizeX; i++) for (j = 0; j < sizeY; j++) for (k = 0; k < sizeZ; k++) {
					if (DFM[i][j][k][0] != 0 || DFM[i][j][k][1] != 0
							|| DFM[i][j][k][2] != 0) {
						deformedIm.set(i, j, k, Interpolation(im, sizeX, sizeY,	sizeZ, i + DFM[i][j][k][0],
								j + DFM[i][j][k][1], k + DFM[i][j][k][2], interpolationType.NearestNeighbor));
					} else {
						deformedIm.set(i, j, k, im.getDouble(i, j, k));
					}
				}
	}
	 */

	/*
	public static double Interpolation(ImageData oldV, int XN, int YN,
			int ZN, float x, float y, float z, int type) {

		return Interpolation(oldV, XN, YN, ZN, (double)x, (double)y, (double)z, type);
	}
	 */

	/*
	public static double TrilinearInterpolationFloatM(float[][][][] oldVM,
			int XO, int YO, int ZO, int dim, double x, double y, double z) {
		int i0, j0, k0, i1, j1, k1;
		float dx, dy, dz, hx, hy, hz;
		float xx, yy, zz;
		xx = (float) x;
		yy = (float) y;
		zz = (float) z;
		if (xx < 0 || xx > (XO - 1) || yy < 0 || yy > (YO - 1) || zz < 0
				|| zz > (ZO - 1)) {
			return 0;
		} else {

			j1 = (int) Math.ceil(xx);
			i1 = (int) Math.ceil(yy);
			k1 = (int) Math.ceil(zz);
			j0 = (int) Math.floor(xx);
			i0 = (int) Math.floor(yy);
			k0 = (int) Math.floor(zz);
			dx = xx - j0;
			dy = yy - i0;
			dz = zz - k0;

			// Introduce more variables to reduce computation
			hx = 1.0f - dx;
			hy = 1.0f - dy;
			hz = 1.0f - dz;
			return ((oldVM[j0][i0][k0][dim] * hx + oldVM[j1][i0][k0][dim] * dx)
	 * hy + (oldVM[j0][i1][k0][dim] * hx + oldVM[j1][i1][k0][dim]
	 * dx)
	 * dy)
	 * hz
					+ ((oldVM[j0][i0][k1][dim] * hx + oldVM[j1][i0][k1][dim]
	 * dx)
	 * hy + (oldVM[j0][i1][k1][dim] * hx + oldVM[j1][i1][k1][dim]
	 * dx)
	 * dy) * dz;
		}
	}
	 */
	/*
	public static double TrilinearInterpolation(ImageData oldV, int XN, int YN,
			int ZN, float x, float y, float z) {

		return TrilinearInterpolation(oldV, XN, YN,	ZN, (double)x, (double)y, (double)z);
	}
	 */

	/*
	// buffers to reduce memory allocation
	public static void TrilinearInterpolation(ImageData oldV[], int XO, int YO,
			int ZO, int channels, double x, double y, double z, double[] res) {
		//int i0, j0, k0, i1, j1, k1;
		//float dx, dy, dz, hx, hy, hz;
		float xx, yy, zz;

		int ch;
		xx = (float) x;
		yy = (float) y;
		zz = (float) z;

	//	if (xx < 0 || xx > (XO - 1) || yy < 0 || yy > (YO - 1) || zz < 0
		//		|| zz > (ZO - 1)) {
		//	for (ch = 0; ch < channels; ch++) {
		//		res[ch] = 0;
		//	}
	//	} else {
	//		j1 = (int) Math.ceil(xx);
		//	i1 = (int) Math.ceil(yy);
	//	k1 = (int) Math.ceil(zz);
		//	j0 = (int) Math.floor(xx);
		//	i0 = (int) Math.floor(yy);
		//	k0 = (int) Math.floor(zz);
	//		dx = xx - j0;
		//	dy = yy - i0;
		//	dz = zz - k0;

			// Introduce more variables to reduce computation
		//	hx = 1.0f - dx;
		//	hy = 1.0f - dy;
		//	hz = 1.0f - dz;
			for (ch = 0; ch < channels; ch++) {

				res[ch] = Interpolation(oldV[ch], XO, YO, ZO, xx, yy, zz, interpolationType.Trilinear);
					//(int) Math
					//	.floor(((oldV[ch].getDouble(j0, i0, k0) * hx + oldV[ch]
					//			.getDouble(j1, i0, k0)
					//			* dx)
					//			* hy + (oldV[ch].getDouble(j0, i1, k0) * hx + oldV[ch]
					//			.getDouble(j1, i1, k0)
					//			* dx)
					//			* dy)
					//			* hz
					//			+ ((oldV[ch].getDouble(j0, i0, k1) * hx + oldV[ch]
					//					.getDouble(j1, i0, k1)
					//					* dx)
					//					* hy + (oldV[ch].getDouble(j0, i1, k1)
					//					* hx + oldV[ch].getDouble(j1, i1, k1)
					//					* dx)
					//					* dy) * dz + 0.5);
		//	}
		}
	}
	 */
}
