package edu.jhu.ece.iacl.algorithms.vabra;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import edu.jhmi.rad.medic.structures.CriticalPointLUT;
import edu.jhmi.rad.medic.utilities.NumericsPublic;
import edu.jhu.ece.iacl.algorithms.registration.RegistrationUtilities;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;

public class VabraSubjectTargetPairs extends AbstractCalculation{

	protected VabraHistograms hist;

	protected int maxDimensions;

	protected VabraVolumeCollection subject;
	protected VabraVolumeCollection deformedSubject;
	protected VabraVolumeCollection target;

	protected VabraVolumeCollection targetBinned;
	protected VabraVolumeCollection deformedSubjectBinned;

	protected int currentDownSampleFactor;

	protected int boundingBox[] = new int[6];

	protected ImageDataFloat currentDeformField; //deformation field 
	protected float[][][][] currentDeformFieldM; //deformation field as an array for easy access
	protected ImageDataFloat totalDeformField; //deformation field 
	protected float[][][][] totalDeformFieldM; //deformation field as an array for easy access
	protected List<RBFPoint> allRBFPoints;

	protected List<ImageData> origSubjectList;
	protected List<ImageData> origTargetList;

	protected int[] chInterpType;
	protected int costFunctionType;
	protected int numOfBins;
	protected int numOfSub;
	protected int numOfTar;
	protected boolean forceDiffeomorphismInRecon;
	protected int defFieldUpdateMode;



	//Homeomorphic deformation variables
	private ImageData labelsToMaintainHomeo;//Labels input to maintain digital homeomorphism
	private float[][][] vx, vy, vz; //approximated def field changes
	private float[][][] ux, uy, uz;//desired rbf to add
	private float[][][] order;
	private boolean[][][] finished;
	private boolean[][][] nonsimple;
	private boolean[][][] free;
	private byte[][][] homeoTransformedLabels;//transformed labels as rbfs as are added
	private	CriticalPointLUT	lut;// the LUT for critical points



	public class RBFPoint implements Comparable<RBFPoint>{

		int[] centerPoint;
		int scale;
		double[] coeff;

		public RBFPoint(int[] centerPoint, int rbfScale, int resolutionScale, double[] gradientCoeff, double globalCoeff){
			this.centerPoint = new int[3];
			this.coeff = new double[3];
			for(int i = 0; i < 3; i++){
				this.coeff[i] = globalCoeff*gradientCoeff[i]*resolutionScale;
				this.centerPoint[i] = centerPoint[i]*(int)resolutionScale;
			}

			this.scale = rbfScale*resolutionScale;

		}

		public int compareTo(RBFPoint o){
			return (int)(this.scale - o.scale);
		}

	}



	public void dispose() {

		currentDeformField = null;
		currentDeformFieldM = null;
		totalDeformField = null;
		//totalDeformFieldM = null;
		subject=null;
		deformedSubject=null;
		origSubjectList=null;
		origTargetList=null;
		targetBinned = null;
		deformedSubjectBinned = null;
		hist.dispose();
	}

	/*
	public VabraSubjectTargetPairs(List<ImageData> subjectVols, List<ImageData> targetVols,
			AbstractCalculation parent, int[] InterpType,boolean useMNMI) {
		this(subjectVols, targetVols, parent, 0.000f, 0.000f, VabraHistograms.defaultBins, InterpType,useMNMI, false, null);
	}
	 */
	public VabraSubjectTargetPairs(List<ImageData> subjectVols, List<ImageData> targetVols, AbstractCalculation parent,
			int numBins, int[] interpType,int costFunctionType, boolean forceDiffeomorphism, ImageData labelsToMaintainHomeo, int defFieldUpdateMode) {
		super(parent);
		this.defFieldUpdateMode = defFieldUpdateMode;
		this.numOfBins = numBins;
		this.origSubjectList=subjectVols;
		this.origTargetList=targetVols;
		this.chInterpType = interpType;
		
		//SSD & NCC intensity workaround (To fix individual normalization)
		//TODO: Fix cost function structures so that we don't have to rescale to bin size for NCC and SSD
		if(costFunctionType == 1 || costFunctionType == 3){
			//We want to set the number of bins equal to number of intensity levels
			this.numOfBins = 10000;//default to something high
		}
		
		this.target=new VabraVolumeCollection(targetVols, chInterpType, numOfBins);
		this.subject=new VabraVolumeCollection(subjectVols, chInterpType, numOfBins);
		deformedSubject=this.subject.clone();
		
		//SSD & NCC intensity workaround (To fix individual normalization)
		//TODO: Fix cost function structures so that we don't have to do this
		if(costFunctionType == 1 || costFunctionType == 3){
			//set max and min for subject image to be the same as Target's
			//Prevents improper rescaling when normalizing to bin size
			subject.setMaxandMinVals(target.maxValsD, target.minValsD);
			deformedSubject.setMaxandMinVals(target.maxValsD, target.minValsD);
		}
		
		
		forceDiffeomorphismInRecon = forceDiffeomorphism;
		numOfSub = origSubjectList.size();
		numOfTar = origTargetList.size();

		//histogram structures
		targetBins = new int[numOfTar];
		subjectBins = new int[numOfSub];
		testBin = new int[numOfSub];
		interpValsD  = new double[numOfSub];

		this.labelsToMaintainHomeo = labelsToMaintainHomeo;

		calculateBoundingBox();

		incrementCompletedUnits();
		targetBinned = target.clone();
		targetBinned.rescaleToBins();
		
		
		this.costFunctionType = costFunctionType;

		switch(costFunctionType){
		case 0:
			System.out.format("Using MI Cost function\n");
			hist = new VabraHistogramsMultCh(numOfSub, numOfTar, numOfBins);
			break;
		case 1:
			System.out.format("Using SSD Cost function\n");
			hist = new VabraHistogramsSSD(numOfSub, numOfTar, numOfBins);
			break;
		case 2:
			System.out.format("Using M-MI Cost function\n");
			hist = new VabraHistogramsMNMI(numOfSub, numOfTar, numOfBins);
			break;
		case 3:
			System.out.format("Using NCC Cost function\n");
			hist = new VabraHistogramsNCC(numOfSub, numOfTar, numOfBins);
			break;
		}

		incrementCompletedUnits();

		totalDeformFieldSplit = new ImageData[3];
		allRBFPoints = new ArrayList<RBFPoint>();

		return;
	}

	private void initDigitalHomeomorphismVars(){
		int XN=labelsToMaintainHomeo.getRows();
		int YN=labelsToMaintainHomeo.getCols();
		int ZN=labelsToMaintainHomeo.getSlices();

		vx = new float[XN][YN][ZN];
		vy = new float[XN][YN][ZN]; 
		vz = new float[XN][YN][ZN];
		ux = new float[XN][YN][ZN];
		uy = new float[XN][YN][ZN];
		uz = new float[XN][YN][ZN];
		order = new float[XN][YN][ZN];
		finished = new boolean[XN][YN][ZN];
		nonsimple = new boolean[XN][YN][ZN];
		free = new boolean[XN][YN][ZN];
		homeoTransformedLabels = new byte[XN][YN][ZN];
		for(int i=0; i < XN; i++)for(int j=0; j < YN; j++)for(int k=0; k < ZN; k++){
			homeoTransformedLabels[i][j][k] = labelsToMaintainHomeo.getByte(i, j, k);
		}

		lut = new CriticalPointLUT("critical626LUT.raw.gz",200);				// the LUT for critical points
		lut.loadCompressedPattern();

	}

	private void disposeDigitalHomeomorphismVars(){
		vx = null;
		vy = null; 
		vz = null;
		ux = null;
		uy = null;
		uz = null;
		order = null;
		finished = null;
		nonsimple = null;
		free = null;
		homeoTransformedLabels = null;
		lut = null;
	}

	public void addRBFPoint(int[] centerPoint,int rbfScale, int resolutionScale, double[] gradientCoeff, double globalCoeff){
		allRBFPoints.add(new RBFPoint(centerPoint, rbfScale, resolutionScale, gradientCoeff, globalCoeff));
	}

	public ImageDataFloat reConstructDeformationFromRBFPoints(int downSampleFactor){

		Collections.sort(allRBFPoints);
		int currentScale=1;
		int[] localROI = new int[6];
		int origXN = origSubjectList.get(0).getRows()/downSampleFactor;
		int origYN = origSubjectList.get(0).getCols()/downSampleFactor;
		int origZN = origSubjectList.get(0).getSlices()/downSampleFactor;
		VabraRBF rbf = new VabraRBF(origXN,origYN,origZN);
		ImageDataFloat deformField = new ImageDataFloat(origXN,origYN,origZN,3);
		float[][][][] deformFieldM = deformField.toArray4d();

		double[] directionsOptmizationWeight = new double[3];
		for (int i = 0; i<3; i++) directionsOptmizationWeight[i] = 1;

		List<RBFPoint> remainingRBFPoints = new ArrayList<RBFPoint>(allRBFPoints);
		Collections.copy(remainingRBFPoints, allRBFPoints);
		boolean changed = false;
		boolean endLoop = false;
		int itr = 0;

		if(labelsToMaintainHomeo != null)initDigitalHomeomorphismVars();

		while(!remainingRBFPoints.isEmpty() && !endLoop){
			//for(Iterator<RBFPoint> itr = allRBFPoints.iterator(); itr.hasNext();){
			//RBFPoint currentPoint = itr.next();

			RBFPoint currentPoint = remainingRBFPoints.get(itr);

			if (currentPoint.scale/downSampleFactor != currentScale){
				currentScale = currentPoint.scale/downSampleFactor;
				rbf.setScale(currentScale);
			}

			int[] currentCenterPoint = new int[3];
			double[] currentCoeff = new double[3];

			for (int i = 0; i < 3; i++){
				currentCenterPoint[i] = currentPoint.centerPoint[i]/downSampleFactor;
				currentCoeff[i] = (float)currentPoint.coeff[i]/(float)downSampleFactor;
			}

			localROI[0] = Math.max(0, currentCenterPoint[0] - rbf.getScale());
			localROI[1] = Math.min(origXN - 1, currentCenterPoint[0] + rbf.getScale());
			localROI[2] = Math.max(0, currentCenterPoint[1] - rbf.getScale());
			localROI[3] = Math.min(origYN - 1, currentCenterPoint[1] + rbf.getScale());
			localROI[4] = Math.max(0, currentCenterPoint[2] - rbf.getScale());
			localROI[5] = Math.min(origZN - 1, currentCenterPoint[2] + rbf.getScale());
			//System.out.format(currentPoint.scale + " X:" + localROI[0] +"to" + localROI[1]+ " Y:" + localROI[2] +"to" + localROI[3]+  "Z:" + localROI[4] +"to" + localROI[5]+"\n");

			if(!forceDiffeomorphismInRecon || checkPositiveJacobian(deformField, 1, currentCoeff, currentCenterPoint, localROI, rbf, directionsOptmizationWeight )){

				if(labelsToMaintainHomeo == null || downSampleFactor != 1){
					addRBFToField(deformFieldM, localROI, currentCenterPoint, currentCoeff, rbf);
				}else{
					addRBFToFieldHomeomorphic(deformFieldM, localROI, currentCenterPoint, currentCoeff, rbf);
				}

				remainingRBFPoints.remove(itr);
				changed = true;
			}else{
				System.out.format("RBF Rejected During Reconstruction\n");
				itr++;
			}

			//check if itr is at end of list
			if(remainingRBFPoints.size() == itr){
				if(changed){
					itr = 0;
					changed = false;
				}else
					endLoop = true;
			}


		} 

		if(remainingRBFPoints.isEmpty())System.out.format("Ended with empty List");
		else if(endLoop)System.out.format("Ended with List Non-Empty");

		//float[] maxMin = RegistrationUtilities.computeMaxAndMinJacobianDet(deformFieldM);
		//System.out.format("Reconstruction Max-Mins:");
		//for(int i = 0; i < 4; i++)System.out.format(maxMin[i] + " ");
		//System.out.format("\n");

		if(labelsToMaintainHomeo != null) disposeDigitalHomeomorphismVars();

		return deformField;

	}

	public void addRBFToField(float[][][][] deformFieldM, int[] localROI, int[] centerPoint, double[] coeff, VabraRBF rbf ){

		float rbfVal;
		int ox_valRBFoff,oy_valRBFoff,oz_valRBFoff;

		for (int i = localROI[0]; i <= localROI[1]; i++) 
			for (int j = localROI[2]; j <= localROI[3]; j++) 
				for (int k = localROI[4]; k <= localROI[5]; k++) {

					//(ox, oy, oz) are coordinates of (i, j) relative to region center				
					ox_valRBFoff = i - centerPoint[0] + rbf.getOffsetX();
					oy_valRBFoff = j - centerPoint[1] + rbf.getOffsetY();
					oz_valRBFoff = k - centerPoint[2] + rbf.getOffsetZ();

					rbfVal = rbf.values[ox_valRBFoff][oy_valRBFoff][oz_valRBFoff];

					deformFieldM[i][j][k][0] +=  coeff[0] * rbfVal;
					deformFieldM[i][j][k][1] +=  coeff[1] * rbfVal;
					deformFieldM[i][j][k][2] +=  coeff[2] * rbfVal;
				}

	}

	//Adds and rbf while maintaining digital homeomorphism
	private void addRBFToFieldHomeomorphic(float[][][][] deformFieldM, int[] localROI, int[] centerPoint, double[] coeff, VabraRBF rbf){


		boolean candidates, finish;
		float maxorder,prevmax;
		int itermax = 1000;
		float rbfVal;
		int ox_valRBFoff,oy_valRBFoff,oz_valRBFoff;

		//clear ROI and init desired rbf to add
		for (int x = localROI[0]; x <= localROI[1]; x++) 
			for (int y = localROI[2]; y <= localROI[3]; y++) 
				for (int z = localROI[4]; z <= localROI[5]; z++) {			// current transform: zero


					//current approximation is zero
					vx[x][y][z] = 0.0f;
					vy[x][y][z] = 0.0f;
					vz[x][y][z] = 0.0f;

					//(ox, oy, oz) are coordinates of (i, j) relative to region center				
					ox_valRBFoff = x - centerPoint[0] + rbf.getOffsetX();
					oy_valRBFoff = y - centerPoint[1] + rbf.getOffsetY();
					oz_valRBFoff = z - centerPoint[2] + rbf.getOffsetZ();

					rbfVal = rbf.values[ox_valRBFoff][oy_valRBFoff][oz_valRBFoff];

					ux[x][y][z] = (float) coeff[0] * rbfVal;
					uy[x][y][z] = (float) coeff[1] * rbfVal;
					uz[x][y][z] = (float) coeff[2] * rbfVal;


					// order by increasing sizes ?
					order[x][y][z] = ux[x][y][z]*ux[x][y][z]+uy[x][y][z]*uy[x][y][z]+uz[x][y][z]*uz[x][y][z];

					finished[x][y][z] = false;
					nonsimple[x][y][z] = false;
					free[x][y][z] = true;
				}

		int iter=0;
		candidates = true;
		finish = false;
		maxorder = 0.0f;
		prevmax = 0.0f;

		//traverse bounds in all directions
		int[][] bounds = {
				{localROI[0],localROI[1],localROI[2],localROI[3],localROI[4],localROI[5]},
				{localROI[0],localROI[1],localROI[2],localROI[3],-localROI[5],-localROI[4]},
				{localROI[0],localROI[1],-localROI[3],-localROI[2],localROI[4],localROI[5]},
				{localROI[0],localROI[1],-localROI[3],-localROI[2],-localROI[5],-localROI[4]},
				{-localROI[1],-localROI[0],localROI[2],localROI[3],localROI[4],localROI[5]},
				{-localROI[1],-localROI[0],localROI[2],localROI[3],-localROI[5],-localROI[4]},
				{-localROI[1],-localROI[0],-localROI[3],-localROI[2],localROI[4],localROI[5]},
				{-localROI[1],-localROI[0],-localROI[3],-localROI[2],-localROI[5],-localROI[4]}
		};
		int x,y,z;
		while (candidates) {
			candidates = false;
			iter++;
			prevmax = maxorder;
			for (int b = 0; b < 8; b++){
				maxorder = 0.0f;
				for (int i = bounds[b][0]; i <= bounds[b][1]; i++) 
					for (int j = bounds[b][2]; j <= bounds[b][3]; j++) 
						for (int k = bounds[b][4]; k <= bounds[b][5]; k++) {
							x=Math.abs(i);
							y=Math.abs(j);
							z=Math.abs(k);
							if (updateTransformAt(x,y,z, deformFieldM)) {
								candidates = true;
							}
							if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
						}
			}
			if(maxorder > 0){
				System.out.println("iteration "+iter+": "+maxorder);
				if (!candidates) System.out.println("no possible update left");
			}
			//if (maxorder==prevmax) candidates = false;
			if (iter>itermax) candidates = false;
			if ( ((maxorder==prevmax) || (maxorder<2)) && !finish) {
				finish = true;
				itermax = iter + 5;
			}

		}// while there are changes



		//commit changes to def field
		for (int i = localROI[0]; i <= localROI[1]; i++) 
			for (int j = localROI[2]; j <= localROI[3]; j++) 
				for (int k = localROI[4]; k <= localROI[5]; k++) {

					//if(order[i][j][k] != 0) System.out.format(i+" " + j + " " + k + " " + order[i][j][k]+"\n");

					deformFieldM[i][j][k][0] += vx[i][j][k];
					deformFieldM[i][j][k][1] += vy[i][j][k];
					deformFieldM[i][j][k][2] += vz[i][j][k];
				}


	}

	public double getCurrentCost(){
		return hist.getCurrentCost();
	}

	public void resetCost(){
		hist.resetCurrentHistograms();
	}

	public void resetCostGradient(){
		hist.resetGradientHistograms();
	}

	public void getCurrentCostGradient(double[] currentCostGradient, double[] deltaC){
		hist.getCostGradients(currentCostGradient, deltaC);
	}


	private int[] targetBins;
	private int[] subjectBins;
	private int[] testBin;
	private double[] interpValsD;


	public void updateCostGradientAtPointWithDef(int i, int j, int k, double defX, double defY,double defZ){
		double x, y, z;
		VabraVolumeCollection referenceSubject;

		//Find values(normalized) of subject and target at the point 
		for (int ch = 0; ch < numOfSub; ch++) {
			subjectBins[ch] = deformedSubjectBinned.data[ch].getInt(i,j, k);
			//System.out.format(subjectBins[ch]+"\n");
		}
		for (int ch = 0; ch < numOfTar; ch++) {
			targetBins[ch] = targetBinned.data[ch].getInt(i, j, k);
			//System.out.format(targetBins[ch]+"\n");
		}

		//Check which type of deformation update method to use(see "defFieldUPdateMode" variable declaration for details)
		if (defFieldUpdateMode == 0){
			x = i + totalDeformFieldM[i][j][k][0];
			y = j + totalDeformFieldM[i][j][k][1];
			z = k + totalDeformFieldM[i][j][k][2];
			referenceSubject = subject;	
		}else{
			x = i;
			y = j;
			z = k;
			referenceSubject = deformedSubject;
		}

		//Adjusts histograms bins in plus/minus x, y and z directions to find NMI gradient
		hist.adjustAllGradientBins(referenceSubject, x, y, z, defX, defY, defZ, targetBins, subjectBins);


	}

	public void updateCurrentCostAtPointWithDef(int i, int j, int k, double defX, double defY,double defZ, boolean commitUpdate){
		double x, y, z;
		VabraVolumeCollection referenceSubject;
		//Check which type of deformation update method to use(see "defFieldUPdateMode" variable declaration for details)
		if (defFieldUpdateMode == 0){
			x = i + totalDeformFieldM[i][j][k][0];
			y = j + totalDeformFieldM[i][j][k][1];
			z = k + totalDeformFieldM[i][j][k][2];
			referenceSubject = subject;	
		}else{
			x = i;
			y = j;
			z = k;
			referenceSubject = deformedSubjectCopy;
		}

		for (int ch = 0; ch < numOfSub; ch++) {
			subjectBins[ch] = deformedSubjectBinned.data[ch].getInt(i,j, k);
			interpValsD[ch] = RegistrationUtilities.Interpolation(referenceSubject.data[ch], totalDeformField.getRows(), totalDeformField.getCols(), totalDeformField.getSlices(), x + defX, y + defY,z + defZ, chInterpType[ch]);
			testBin[ch] = referenceSubject.calculateBin(interpValsD[ch], ch);
			//if(coeff == 0 && subjectBins[ch] != testBin[ch]) System.out.format(subjectBins[ch]+" "+interpValsD[ch]+" "+ testBin[ch]+"\n");
		}
		for (int ch = 0; ch < numOfTar; ch++) {
			targetBins[ch] = targetBinned.data[ch].getInt(i, j, k);
		}
		//System.out.format("interp"+interpValsD[0]+"submax"+referenceSubject.maxValsD[0]+"\n");
		//System.out.format("sub"+subjectBins[0]+"test"+testBin[0]+"tar"+targetBins[0]+"\n");
		hist.adjustCurrentBins(subjectBins, targetBins,testBin);

		if(commitUpdate){			
			//Check which type of deformation update method to use(see "defFieldUPdateMode" variable declaration for details)
			if (defFieldUpdateMode == 0){
				totalDeformFieldM[i][j][k][0] += (float)defX;
				totalDeformFieldM[i][j][k][1] += (float)defY;
				totalDeformFieldM[i][j][k][2] += (float)defZ;
			}else{
				currentDeformFieldM[i][j][k][0] = (float)defX;
				currentDeformFieldM[i][j][k][1] = (float)defY;
				currentDeformFieldM[i][j][k][2] = (float)defZ;
			}

			//change images and real histograms if actually updating
			for (int ch = 0; ch < numOfSub; ch++) {
				deformedSubject.data[ch].set(i, j, k, interpValsD[ch]);
				deformedSubjectBinned.data[ch].set(i, j, k, testBin[ch]);
			}
			hist.adjustOrigBins(subjectBins, targetBins,testBin);
		}

	}

	public void finalDefFieldUpdate(int[] regionToUpdate){

		//Check which type of deformation update method to use(see "defFieldUPdateMode" variable declaration for details)
		//System.out.format("Before:"+ -imgSubTarPairs.hist.getMNMI(imgSubTarPairs.hist.currentDeformedSubject, imgSubTarPairs.hist.origTarget, imgSubTarPairs.hist.currentJointST)+"\n");
		hist.commitCurrentJointHistogram();
		//System.out.format("After:"+ -imgSubTarPairs.hist.getMNMI(imgSubTarPairs.hist.currentDeformedSubject, imgSubTarPairs.hist.origTarget, imgSubTarPairs.hist.currentJointST)+"\n");
		if (defFieldUpdateMode == 1){
			for(int ch = 0; ch < numOfSub; ch++){
				for (int i = regionToUpdate[0]; i <= regionToUpdate[1]; i++) 
					for (int j = regionToUpdate[2]; j <= regionToUpdate[3]; j++) 
						for (int k = regionToUpdate[4]; k <= regionToUpdate[5]; k++) 
							deformedSubjectCopy.data[ch].set(i, j, k, deformedSubject.data[ch].getDouble(i, j, k));
			}

			updateDefField(regionToUpdate);
		}
		//for (int ch = 0; ch < imgSubTarPairs.numOfCh; ch++){
		//nmiValD -= imgSubTarPairs.chWeights[ch] * RegistrationUtilities.NMI(imgSubTarPairs.hist.currentDeformedSubject, imgSubTarPairs.hist.origTarget,
		//	imgSubTarPairs.hist.currentJointST, ch, imgSubTarPairs.numOfBins);
		//if(commitUpdate)System.out.format("Committing" + imgSubTarPairs.hist.getCurrentNMI() + "\n");
		//if(commitUpdate)System.out.format("Committed:" + imgSubTarPairs.hist.getOrigNMI() + "\n");
		//}

	}

	public void updateDefField(){
		int[] regionToUpdate = new int[6];
		regionToUpdate[0] = 0;
		regionToUpdate[1] = subject.getXN() -1;
		regionToUpdate[2] = 0;
		regionToUpdate[3] = subject.getYN() -1;
		regionToUpdate[4] = 0;
		regionToUpdate[5] = subject.getZN() -1;
		updateDefField(regionToUpdate);
	}

	public void updateDefField(int[] regionToUpdate){

		double newVec, oldVec, x, y, z;
		int XN = totalDeformField.getRows();
		int YN = totalDeformField.getCols();
		int ZN = totalDeformField.getSlices();

		for(int i = regionToUpdate[0]; i <= regionToUpdate[1]; i++)
			for(int j = regionToUpdate[2]; j <= regionToUpdate[3]; j++)
				for(int k = regionToUpdate[4]; k <= regionToUpdate[5]; k++){
					if(currentDeformFieldM[i][j][k][0] != 0 || currentDeformFieldM[i][j][k][1] != 0 || currentDeformFieldM[i][j][k][2] != 0 ){
						x = i + currentDeformFieldM[i][j][k][0];
						y = j + currentDeformFieldM[i][j][k][1];
						z = k + currentDeformFieldM[i][j][k][2];

						for(int c = 0; c < 3; c++){
							oldVec = RegistrationUtilities.Interpolation(totalDeformFieldSplit[c], XN, YN, ZN, x, y, z, 0);
							newVec = currentDeformFieldM[i][j][k][c] + oldVec;
							totalDeformField.set(i, j, k, c, newVec);
							currentDeformFieldM[i][j][k][c] = 0;
						}
					}
				}

		for(int c = 0; c < totalDeformField.getComponents(); c++)
		{
			for(int i = regionToUpdate[0]; i <= regionToUpdate[1]; i++)
				for(int j = regionToUpdate[2]; j <= regionToUpdate[3]; j++)
					for(int k = regionToUpdate[4]; k <= regionToUpdate[5]; k++){
						totalDeformFieldSplit[c].set(i, j, k, totalDeformField.getDouble(i, j, k, c));
					}
		}

	}

	private final boolean updateTransformAt(int x, int y, int z, float[][][][] deformFieldM) {
		boolean changed = false;
		float xt,yt,zt;
		float i,j,l;
		byte targetLabel;
		float diffx, diffy, diffz;

		if (!finished[x][y][z]) {
			// check size: if too small we're done
			if (order[x][y][z] == 0) {
				finished[x][y][z] = true;
			} else {
				// find the position in original image
				xt = x + deformFieldM[x][y][z][0] + vx[x][y][z];
				yt = y + deformFieldM[x][y][z][1] + vy[x][y][z];
				zt = z + deformFieldM[x][y][z][2] + vz[x][y][z];

				i=0;
				j=0;
				l=0;

				diffx = ux[x][y][z]-vx[x][y][z];
				diffy = uy[x][y][z]-vy[x][y][z];
				diffz = uz[x][y][z]-vz[x][y][z];

				// find the next 6-c point to be reached
				if ( (Math.abs(diffx)>=Math.abs(diffy))&& (Math.abs(diffx)>=Math.abs(diffz)) ) {
					if(Math.abs(diffx) < 1)
						i = diffx;
					else
						i = Math.signum(diffx);
				} else if ( (Math.abs(diffy)>=Math.abs(diffx))&&(Math.abs(diffy)>=Math.abs(diffz)) ) {
					if(Math.abs(diffy) < 1)
						j = diffy;
					else
						j = Math.signum(diffy);
				} else {
					if(Math.abs(diffz) < 1)
						l = diffz;
					else
						l = Math.signum(diffz);
				}

				if  ( (xt+i<=0) || (xt+i>=ux.length) || (yt+j<=0) || (yt+j>=ux[0].length) || (zt+l<=0) || (zt+l>=ux[0][0].length) ) {
					// the origin point is out of the image: no need for constraint
					finished[x][y][z] = true;
					order[x][y][z] = 0;					
				} else if ( (i==0) && (j==0) && (l==0) ) {
					// not going any further
					finished[x][y][z] = true;
					order[x][y][z] = 0;
				} else {
					// first, check if the spot is taken by another object ?? 
					// nope, only one arrival point possible
					// check if the point can switch labels
					targetLabel = (byte)RegistrationUtilities.Interpolation(labelsToMaintainHomeo,xt+i,yt+j,zt+l,RegistrationUtilities.InterpolationType.NEAREST_NEIGHTBOR);
					if (simplePoint(homeoTransformedLabels, x,y,z, targetLabel)) {
						// change the label
						homeoTransformedLabels[x][y][z] = targetLabel;
						// update the transform : composition of transforms ??
						vx[x][y][z] += i;
						vy[x][y][z] += j;
						vz[x][y][z] += l;
						// recompute the ordering on the new object
						order[x][y][z] = ((ux[x][y][z])-vx[x][y][z])*((ux[x][y][z])-vx[x][y][z])
						+((uy[x][y][z])-vy[x][y][z])*((uy[x][y][z])-vy[x][y][z])
						+((uz[x][y][z])-vz[x][y][z])*((uz[x][y][z])-vz[x][y][z]);

						// changes: go for a new loop
						changed = true;
					} else {
						// not possible to change the point: wait until one 
						//of the neighbors is changed
						nonsimple[x][y][z] = true;
					}// simple point
				}// boundary
			}// transform size
		}// finished
		return changed;
	}

	/**
	 *  critical relation detection: groups objects with relations
	 */
	private final boolean simplePoint(byte[][][] img, int x, int y, int z, int k) {
		// is the new object regular ? not the growing object, the original one!!
		//if (!isRegular(x,y,z,k)) return false;

		// inside the original object ?
		if (img[x][y][z]==k) return true;

		boolean [][][] obj = new boolean[3][3][3];

		// does it change the topology of the new object ?
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if(x+i < 0 || x+i >= img.length || y+j < 0 || y+j >= img[0].length || z+l < 0 || z+l >= img[0][0].length){
				//if out of bounds then label is zero
				if (0==k) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}else{
				if (img[x+i][y+j][z+l]==k) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}

		}
		obj[1][1][1] = true;

		// check well-composedness for new object(s)
		///if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;

		if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;

		// does it change the topology of the object it modifies ?
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if(x+i < 0 || x+i >= img.length || y+j < 0 || y+j >= img[0].length || z+l < 0 || z+l >= img[0][0].length){
				//if out of bounds then label is zero
				if (0==img[x][y][z]) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}else{
				if (img[x+i][y+j][z+l]==img[x][y][z]) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
		}
		obj[1][1][1] = false;

		if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;

		// does it change the topology of a relation between the modified object and its neighbors ?
		int  Nconfiguration = 0;
		byte[] lb = new byte[26];
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if ( (i*i+j*j+l*l>0) 
					&& (img[x+i][y+j][z+l]!=k) 
					&& (img[x+i][y+j][z+l]!=img[x][y][z]) ) {
				boolean found = false;
				for (int n=0;n<Nconfiguration;n++) 
					if (img[x+i][y+j][z+l]==lb[n]) { found = true; break; }

				if (!found) {
					lb[Nconfiguration] = img[x+i][y+j][z+l];
					Nconfiguration++;
				}
			}
		}

		// pairs
		for (int n=0;n<Nconfiguration;n++) {
			// in relation with previous object
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if ( (img[x+i][y+j][z+l]==img[x][y][z])
						|| (img[x+i][y+j][z+l]==lb[n]) ) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
			obj[1][1][1] = false;
			if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
			// in relation with new object
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if ( (img[x+i][y+j][z+l]==k)
						|| (img[x+i][y+j][z+l]==lb[n]) ) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
			obj[1][1][1] = true;
			//if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
			if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
		}
		// triplets
		for (int n=0;n<Nconfiguration;n++) {
			for (int m=n+1;m<Nconfiguration;m++) {
				// in relation with previous object
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if ( (img[x+i][y+j][z+l]==img[x][y][z])
							|| (img[x+i][y+j][z+l]==lb[n])
							|| (img[x+i][y+j][z+l]==lb[m]) ) {
						obj[1+i][1+j][1+l] = true;
					} else {
						obj[1+i][1+j][1+l] = false;
					}
				}
				obj[1][1][1] = false;
				if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
				// in relation with new object
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if ( (img[x+i][y+j][z+l]==k)
							|| (img[x+i][y+j][z+l]==lb[n]) 
							|| (img[x+i][y+j][z+l]==lb[m]) ) {
						obj[1+i][1+j][1+l] = true;
					} else {
						obj[1+i][1+j][1+l] = false;
					}
				}
				obj[1][1][1] = true;
				//				if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
				if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
			}
		}
		// else, it works
		return true;
	}

	public boolean checkPositiveJacobian(ImageDataFloat defField, double coeff, double[] localCoarseGradient, int[] coarseLocalRegionCenter, int[] localROI, VabraRBF rbf, double[] directionsOptmizationWeight ){
		double rbfVal;
		double coeff_x, coeff_y, coeff_z;

		int ox_valRBFoff, oy_valRBFoff, oz_valRBFoff;
		double defX, defY, defZ;
		double tlambda;
		//int i, j, k, ch;
		boolean posNeg= true; 
		//temp start
		int XN =  defField.getRows();
		int YN =  defField.getCols();
		int ZN =  defField.getSlices();
		float[][][][] defFieldM = defField.toArray4d();
		//temp end


		tlambda = (Math.round((double) coeff * 10000.0)) / 10000.0;

		coeff_x = ((double) tlambda) * localCoarseGradient[0];
		coeff_y = ((double) tlambda) * localCoarseGradient[1];
		coeff_z = ((double) tlambda) * localCoarseGradient[2];

		float[][][][] def = new float[XN][YN][ZN][3];

		//System.out.format("Bounds:"+XN +" " + YN + " " + ZN +"\n");
		//System.out.format("Center:"+coarseLocalRegionCenter[0] +" " + coarseLocalRegionCenter[1] + " " + coarseLocalRegionCenter[2] +"\n");

		//optimized by putting outside and incrementing.
		for (int i = 0; i < XN; i++) 
			for (int j = 0; j < YN; j++) 
				for (int k = 0; k < ZN; k++) {

					def[i][j][k][0] = defFieldM[i][j][k][0];
					def[i][j][k][1] = defFieldM[i][j][k][1];
					def[i][j][k][2] = defFieldM[i][j][k][2];

					if( i >= localROI[0] && i <= localROI[1] && j >= localROI[2]&& j <= localROI[3]&& k >= localROI[4]&& k <= localROI[5]) {

						//(ox, oy, oz) are coordinates of (i, j) relative to region center				
						ox_valRBFoff = i - coarseLocalRegionCenter[0] + rbf.getOffsetX();
						oy_valRBFoff = j - coarseLocalRegionCenter[1] + rbf.getOffsetY();
						oz_valRBFoff = k - coarseLocalRegionCenter[2] + rbf.getOffsetZ();

						rbfVal = rbf.values[ox_valRBFoff][oy_valRBFoff][oz_valRBFoff];

						if (rbfVal != 0) {
							//Amount to adjust with RBF if not constrained
							defX = directionsOptmizationWeight[0]*coeff_x*rbfVal;
							defY = directionsOptmizationWeight[1]*coeff_y*rbfVal;
							defZ = directionsOptmizationWeight[2]*coeff_z*rbfVal;

							def[i][j][k][0] += (float)defX;
							def[i][j][k][1] += (float)defY;
							def[i][j][k][2] += (float)defZ;


						}
					}
				}
		float[][] jac = new float[3][3];
		for(int x=0; x<XN; x++) for(int y=0; y<YN; y++) for(int z=0; z<ZN; z++){
			// derivatives of the x comp wrt x, y, and z
			int LY = (y == 0) ? 1 : 0;
			int HY = (y == (YN - 1)) ? 1 : 0;
			int LZ = (z == 0) ? 1 : 0;
			int HZ = (z == (ZN - 1)) ? 1 : 0;
			int LX = (x == 0) ? 1 : 0;
			int HX = (x == (XN - 1)) ? 1 : 0;

			// central differences
			jac[0][0] = (def[x + 1 - HX][y][z][0] - def[x - 1 + LX][y][z][0]) / 2;
			jac[0][1] = (def[x][y + 1 - HY][z][0] - def[x][y - 1 + LY][z][0]) / 2;
			jac[0][2] = (def[x][y][z + 1 - HZ][0] - def[x][y][z - 1 + LZ][0]) / 2;

			jac[1][0] = (def[x + 1 - HX][y][z][1] - def[x - 1 + LX][y][z][1]) / 2;
			jac[1][1] = (def[x][y + 1 - HY][z][1] - def[x][y - 1 + LY][z][1]) / 2;
			jac[1][2] = (def[x][y][z + 1 - HZ][1] - def[x][y][z - 1 + LZ][1]) / 2;

			jac[2][0] = (def[x + 1 - HX][y][z][2] - def[x - 1 + LX][y][z][2]) / 2;
			jac[2][1] = (def[x][y + 1 - HY][z][2] - def[x][y - 1 + LY][z][2]) / 2;
			jac[2][2] = (def[x][y][z + 1 - HZ][2] - def[x][y][z - 1 + LZ][2]) / 2;


			jac[0][0]+=1;
			jac[1][1]+=1;
			jac[2][2]+=1;

			//check if determinant if determinant is negative:
			float jacob = NumericsPublic.determinant3D(jac);



			if(jacob < 0){
				System.out.format("Jacobian is negative:" + jacob + "Loc:"+x +" "+y+" "+z+"\n");
				return false;
				//posNeg = false;
			}
		}



		return posNeg;

	}

	//public void calculateMaxAndMinVals(){

	//	subject.calculateMaxAndMinVals();
	//	deformedSubject.calculateMaxAndMinVals();
	//	target.calculateMaxAndMinVals();


	//}

	/*//don't need anymore, performing normalization in individual collections if needed
	public void normalizeImagesToMax() {
		for (int ch = 0; ch < numOfSub; ch++) {
			ImageDataMath.normalizeToUnitIntensity(subject.data[ch]);
			ImageDataMath.scaleFloatValue(subject.data[ch],(float) maxValsD);

			ImageDataMath.normalizeToUnitIntensity(deformedSubject.data[ch]);
			ImageDataMath.scaleFloatValue(deformedSubject.data[ch], (float) maxValsD);
		}
		for (int ch = 0; ch < numOfTar; ch++) {
			ImageDataMath.normalizeToUnitIntensity(target.data[ch]);
			ImageDataMath.scaleFloatValue(target.data[ch], (float) maxValsD);
		}
	}
	 */

	public void prepareForNextLevel() {
		// System.out.println(getClass().getCanonicalName()+"\t"+"START PREPARE FOR NEXT LEVEL");
		double costVal = hist.getOrigCost();
		System.out.println(getClass().getCanonicalName()+"\t"+"Cost before prepForNextLevelParent=" + costVal);

		deformedSubject = subject.returnDeformedCopy(totalDeformField);
		deformedSubjectCopy = deformedSubject.clone();

		// find binned copies of target and deformed subject
		targetBinned = target.clone();
		deformedSubjectBinned = deformedSubject.clone();

		
		//SSD & NCC intensity workaround (To fix individual normalization)
		//TODO: Fix cost function structures so that we don't have to do this
		if(costFunctionType == 1 || costFunctionType == 3){
			//set max and min for subject image to be the same as Targets
			//Prevents improper rescaling when normalizing to bin size
			deformedSubject.setMaxandMinVals(target.maxValsD, target.minValsD);
			deformedSubjectCopy.setMaxandMinVals(target.maxValsD, target.minValsD);
			deformedSubjectBinned.setMaxandMinVals(target.maxValsD, target.minValsD);
		}
		
		//rescale max/min to bin size
		targetBinned.rescaleToBins();
		deformedSubjectBinned.rescaleToBins();

		calculateBoundingBox();
		
		hist.updateHistograms(targetBinned,deformedSubjectBinned,boundingBox);
		costVal = hist.getOrigCost();
		//nmiVal = -RegistrationUtilities.NMI(hist.origDeformedSubject, hist.origTarget, hist.origJointST, 0, numOfBins);
		System.out.format("Current COST=%f\n", costVal);
		// System.out.format("I'm bothered that NMI goes down a little bit in
		// between levels... it has something to do with hist.origJointogram
		// not being consitent\n");

	}


	public void setResolution(int downSampleFactor) {

		int newX, newY, newZ;
		int oldX, oldY, oldZ;
		double sigma = 1.0;

		System.out.format("Setting resolution to" + 1/downSampleFactor +"of original image.\n" );
		// System.out.format("imagepairChild down sample factor %f\n",
		// downSampleFactor);
		//if (currentDownSampleFactor == downSampleFactor)
		//return;

		oldX = subject.getXN();
		oldY = subject.getYN();
		oldZ = subject.getZN();

		currentDownSampleFactor = downSampleFactor;
		reintializeFromFile();

		if (currentDownSampleFactor != 1.0) {
			newX = (int) (subject.getXN() / currentDownSampleFactor);
			newY = (int) (subject.getYN() / currentDownSampleFactor);
			newZ = (int) (subject.getZN() / currentDownSampleFactor);
			subject.downSample(newX, newY, newZ, sigma);
			target.downSample(newX, newY, newZ, sigma);
		} else {
			newX = (int) (subject.getXN());
			newY = (int) (subject.getYN());
			newZ = (int) (subject.getZN());
		}

		totalDeformField = reConstructDeformationFromRBFPoints(downSampleFactor);
		totalDeformFieldM = totalDeformField.toArray4d();
		//totalDeformField = new ImageDataFloat(subject.getXN(), subject.getYN(), subject.getZN(), 3);
		/*if (totalDeformField == null) {
			// setup deformation fields for first time
			totalDeformField = new ImageDataFloat(subject.getXN(), subject.getYN(), subject.getZN(), 3);
		} else {
			// resample deformation fields
			totalDeformField = new ImageDataFloat(subject.getXN(), subject.getYN(), subject.getZN(), 3);
			RegistrationUtilities.DeformationFieldResample3DM(totalDeformFieldM, totalDeformField, oldX, oldY, oldZ, newX, newY, newZ);
		}*/


		//RegistrationUtilities.DeformationFieldResample3DM(totalDeformFieldM, totalDeformField);

		currentDeformField = new ImageDataFloat(subject.getXN(), subject.getYN(), subject.getZN(), 3);
		currentDeformFieldM = currentDeformField.toArray4d();		

		for(int c = 0; c < totalDeformField.getComponents(); c++){
			totalDeformFieldSplit[c] = new ImageDataFloat(subject.getXN(), subject.getYN(), subject.getZN());
			for(int i = 0; i < subject.getXN(); i++)
				for(int j = 0; j < subject.getYN(); j++)
					for(int k = 0; k < subject.getZN(); k++){
						totalDeformFieldSplit[c].set(i, j, k, totalDeformField.getDouble(i, j, k, c));
					}
		}
	}

	VabraVolumeCollection deformedSubjectCopy;
	ImageData[]  totalDeformFieldSplit;



	/*
	public double totalDeformFieldInterpolate(double x, double y, double z, int dim){
		int XN = totalDeformField.getRows();
		int YN = totalDeformField.getCols();
		int ZN = totalDeformField.getSlices();
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		if (x < 0 || x > (XN - 1) || y < 0 || y > (YN - 1) || z < 0
				|| z > (ZN - 1)) {
			return 0;
		} else {
			j1 = (int) Math.ceil(x);
			i1 = (int) Math.ceil(y);
			k1 = (int) Math.ceil(z);
			j0 = (int) Math.floor(x);
			i0 = (int) Math.floor(y);
			k0 = (int) Math.floor(z);
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;

			// Introduce more variables to reduce computation
			hx = 1.0f - dx;
			hy = 1.0f - dy;
			hz = 1.0f - dz;
			// Optimized below
			return   (((totalDeformField.getDouble(j0, i0, k0, dim) * hx + totalDeformField.getDouble(j1, i0, k0, dim) * dx) * hy 
					+ (totalDeformField.getDouble(j0, i1, k0, dim) * hx + totalDeformField.getDouble(j1, i1, k0, dim) * dx) * dy) * hz 
					+ ((totalDeformField.getDouble(j0, i0, k1, dim) * hx + totalDeformField.getDouble(j1, i0, k1, dim) * dx) * hy 
							+ (totalDeformField.getDouble(j0, i1, k1, dim) * hx + totalDeformField.getDouble(j1, i1, k1, dim) * dx) * dy)* dz);

		}
	}

	 */
	public int coarseGradientParameters() {
		return 3; 
	}

	void reintializeFromFile() {
		this.subject=new VabraVolumeCollection(origSubjectList, chInterpType, numOfBins);
		this.target=new VabraVolumeCollection(origTargetList, chInterpType, numOfBins);
		
		//SSD & NCC intensity workaround (To fix individual normalization)
		//TODO: Fix cost function structures so that we don't have to do this
		if(costFunctionType == 1 || costFunctionType == 3){
			//set max and min for subject image to be the same as Targets
			//Prevents improper rescaling when normalizing to bin size
			subject.setMaxandMinVals(target.maxValsD, target.minValsD);
		}
		
		System.gc();
		//calculateBoundingBox();
	}

	void calculateBoundingBox() {
		int k, j, i, ch;
		int XN, YN, ZN, CH, ext;
		XN = subject.getXN();
		YN = subject.getYN();
		ZN = subject.getZN();
		//	CH = numOfCh;
		ext = 5;

		boundingBox[1] = 0;
		boundingBox[0] = subject.getXN();
		boundingBox[3] = 0;
		boundingBox[2] = subject.getYN();
		boundingBox[5] = 0;
		boundingBox[4] = subject.getZN();

		int count=0;
		for (ch = 0; ch < Math.min(numOfSub,numOfTar); ch++) {
			ImageData sub=deformedSubject.data[ch];
			ImageData tar=target.data[ch];
			for (i = 0; i < XN; i++) for (j = 0; j < YN; j++) for (k = 0; k < ZN; k++){
				if ((Math.abs(sub.getDouble(i, j, k)) > 0.0000001) || (Math.abs(tar.getDouble(i, j, k)) > 0.0000001))
				{
					count++;
					if (i < boundingBox[0]) boundingBox[0] = i;
					if (i > boundingBox[1]) boundingBox[1] = i;
					if (j < boundingBox[2]) boundingBox[2] = j;
					if (j > boundingBox[3]) boundingBox[3] = j;
					if (k < boundingBox[4]) boundingBox[4] = k;
					if (k > boundingBox[5]) boundingBox[5] = k;
				}
			}
		}

		boundingBox[0]=Math.max(0, boundingBox[0]-ext); 			
		boundingBox[1]=Math.min(XN-1, boundingBox[1]+ext);
		boundingBox[2]=Math.max(0, boundingBox[2]-ext); 			
		boundingBox[3]=Math.min(YN-1, boundingBox[3]+ext);
		boundingBox[4]=Math.max(0, boundingBox[4]-ext); 			
		boundingBox[5]=Math.min(ZN-1, boundingBox[5]+ext);

		maxDimensions = Math.max(Math.max(boundingBox[1]-boundingBox[0], boundingBox[3]-boundingBox[2]), boundingBox[5]-boundingBox[4]);
	}


	List<ImageData> getDeformedSubject(){
		ArrayList<ImageData> out = new ArrayList<ImageData>();
		ImageData defSub, origSub;
		for (int i = 0; i < numOfSub; i++){
			origSub = origSubjectList.get(i);
			defSub = origSub.clone();
			RegistrationUtilities.DeformImage3D(origSub, defSub, totalDeformField, origSub.getRows(),
					origSub.getCols(), origSub.getSlices(), chInterpType[i]); 
			defSub.setHeader(origSubjectList.get(i).getHeader());
			defSub.setName(origSubjectList.get(i).getName() + "_reg");
			out.add(defSub);
		}
		return out;
	}

	ImageDataFloat getDeformationField(){
		totalDeformField.setHeader(origSubjectList.get(0).getHeader());
		totalDeformField.setName(origSubjectList.get(0).getName()+"_def_field");
		return totalDeformField;
	}

	public int[] getBoundingBox() {
		calculateBoundingBox();
		return boundingBox;
	}

}
