package edu.jhu.ece.iacl.plugins.dti.tractography;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Vector;

import javax.vecmath.Point3d;

import edu.jhmi.rad.medic.utilities.MedicUtilPublic;
import edu.jhu.ece.iacl.algorithms.dti.tractography.FiberStatistics;
import edu.jhu.ece.iacl.jist.io.ArrayDoubleTxtReaderWriter;
import edu.jhu.ece.iacl.jist.io.CurveVtkReaderWriter;
import edu.jhu.ece.iacl.jist.io.FiberCollectionReaderWriter;
import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamPointDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.fiber.FiberCollection;


public class MedicAlgorithmFiberBundleStatisticsFromVolume extends ProcessingAlgorithm{
	//input params
	private ParamFileCollection fibers;
	private ParamVolume volume;
	private ParamFloat minLength;
	private ParamString statFile;
	
	

	private FiberCollectionReaderWriter fcrw = FiberCollectionReaderWriter.getInstance();

	//private ArrayDoubleTxtReaderWriter arw = ArrayDoubleTxtReaderWriter.getInstance();

	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Compute the tract statistic over the specified volume and save the results in the given csv file.";
	private static final String longDescription = "";


	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(fibers=new ParamFileCollection("Fibers", new FileExtensionFilter(new String[]{"dat"})));
		inputParams.add(volume=new ParamVolume("Volume",null,-1,-1,-1,-1));
		volume.setMandatory(false);
		inputParams.add(minLength = new ParamFloat("Minimum Fiber Lenght ", 0.0f));
		minLength.setDescription("Minimum length for a fiber to be icnluded in the statistics");
		inputParams.add(statFile=new ParamString("Output Statistic File", ""));
//		inputParams.add(comparison = new ParamOption("Connectivity Matrix Statistic",new String[]{"All", "Mean", "Max", "Min", "Median", "None"}));
//		comparison.setDescription("");


		inputParams.setPackage("IACL");
		inputParams.setCategory("DTI.Fiber");
		inputParams.setLabel("Fiber Bundle Stats from Volume");
		inputParams.setName("Fiber_Bundle_Stats_from_Volume");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("");
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.RC);
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		//does not have any output. The results are saved in the specified file.
	}


	protected void execute(CalculationMonitor monitor) {
		File dir = new File(this.getOutputDirectory()+File.separator+edu.jhu.ece.iacl.jist.utility.FileUtil.forceSafeFilename(this.getAlgorithmName()));
		try{
			if(!dir.isDirectory()){
				(new File(dir.getCanonicalPath())).mkdir();
			}
		}catch(IOException e){
			e.printStackTrace();
		}


//		DecimalFormat dist = new DecimalFormat("###.#######");
		FiberStatistics fstat = new FiberStatistics();
		ArrayList<Double> lens;
		ArrayList<Integer> counts;
		ArrayList<Double> sum;
		ArrayList<Double> max;
		ArrayList<Double> min;
		ArrayList<Vector<Double>> data;
		ArrayList<String> outputStat = new ArrayList<String>();
		String currentStat;
		for(File f : fibers.getValue()){
			FiberCollection allfibers = fcrw.read(f);
			fstat.setFibers(allfibers);
			if(fstat.getRes()==null && volume.getValue()!=null){
				fstat.setRes(volume.getImageData().getHeader().getDimResolutions()[0],
						volume.getImageData().getHeader().getDimResolutions()[1],
						volume.getImageData().getHeader().getDimResolutions()[2]);
			}
			fstat.computeLengths();
			fstat.computeNumPts();
			lens = fstat.getFiberLengths();
			counts = fstat.getFiberPointsCount();
			int totalNodes =0;
			int totalFibers =0;
			double sumLength = 0;
			//all the statistics should have the same size, if not, something is wrong, throw an exception
			if (counts.size()!=lens.size()){
				MedicUtilPublic.displayError("Some statistics have not been computed for all fibers!");
			}
			for (int i=0; i<lens.size(); i++) if (lens.get(i)>minLength.getFloat()){
				sumLength += lens.get(i);
				totalNodes += counts.get(i);
				totalFibers++;
			}
			currentStat = f.getName()+","+sumLength/totalFibers +","+totalFibers; 
			//all the statistics should have the same size, if not, something is wrong, throw an exception			
			if(volume.getValue()!=null){
				fstat.findTractVolumeStats(volume.getImageData());

				sum = fstat.getSumFromVolume();
				max = fstat.getMaxFromVolume();
				min = fstat.getMaxFromVolume();
				data = fstat.getDataFromVolume();
				if(sum.size()!=lens.size() || max.size() != lens.size() || min.size() != lens.size() || data.size() != lens.size())
					MedicUtilPublic.displayError("Some statistics have not been computed for all fibers!");
				double avgout =0;
				double sumout = 0;
				double maxout = Double.MIN_VALUE;
				double minout = Double.MAX_VALUE;
				double medout = 0;
				ArrayList<Double> allData = new ArrayList<Double>(totalNodes);
				for (int i=0; i<lens.size(); i++) if (lens.get(i)>minLength.getFloat()){
					sumout+=sum.get(i);
					if (maxout<max.get(i))
						maxout=max.get(i);
					if (minout>min.get(i))
						minout=min.get(i);
					allData.addAll(data.get(i));
				}
				avgout = sumout/totalNodes;
				Collections.sort(allData);
				if (allData.size() == 0)
					medout = Double.NaN;
				else if (allData.size()==1)
					medout = allData.get(1);
				else if (allData.size() % 2 ==0)
					medout = 0.5 * (allData.get(allData.size()/2) + allData.get(allData.size()/2-1));
				else
					medout = allData.get(allData.size()/2-1);
				allData = null;
				System.gc();
				currentStat += ","+avgout+","+medout+","+sumout+","+minout+","+maxout+"\n";
				outputStat.add(currentStat);
				
			}
		}
		writeStatisToCSV(outputStat, statFile.getValue(), "Subject,Length,Count,Mean,Median,Sum,Minimum,Maximum");
		
	}
	
	private final void writeStatisToCSV (ArrayList<String> stats,String statFile, String signature){
		
		File destdir = null;
		File resultFile;
		try {
			if (!statFile.equals("")){
				resultFile = new File(statFile);
			}else {
				destdir = new File(this.getOutputDirectory().getCanonicalFile()+File.separator+this.getAlgorithmName());
				if(!destdir.isDirectory()){
					(new File(destdir.getCanonicalPath())).mkdir();
				}
				resultFile = new File(destdir+File.separator+"Statistics.csv");
				System.out.println("No statistic file is given. A new file is generated: "+destdir+File.separator+"SegmentationStatistics.csv");
			}


			ArrayList<String> list = new ArrayList<String>();
			File f;
			if(resultFile.isFile()){
				System.out.println("reading previous statistic file: "+resultFile.getName());
				f = new File(resultFile.getAbsolutePath());
				FileReader fr = new FileReader(f);
				BufferedReader br = new BufferedReader(fr);
				//read info line
				String line = br.readLine();

				// Exact corresponding template for first line ?
				if (!line.startsWith(signature)) {
					System.out.println("not a proper segmentation statistics file");
					br.close();
					fr.close();
				}
				//Now read stats
				line = br.readLine();
				while (line!=null) {
					list.add(line+"\n");
					line = br.readLine();
				}
				br.close();
				fr.close();
			}

			f = new File(resultFile.getAbsolutePath());
			FileWriter fw = new FileWriter(f);
			PrintWriter pw = new PrintWriter( fw );
			pw.write(signature+"\n");
			for (int n=0;n<list.size();n++) {
				pw.write(list.get(n));
			}
			for (int n=0;n<stats.size();n++) {
				pw.write(stats.get(n));
			}
			pw.close();
			fw.close();
		}
		catch (FileNotFoundException e) {
			System.out.println(e.getMessage());
		}
		catch (IOException e) {
			System.out.println(e.getMessage());
		} 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
		}
	}


	
}
