package edu.vanderbilt.VUIIS.plugins;


import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;

import edu.jhu.bme.smile.commons.math.Spline;
import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.VUIIS.algorithms.CESTSmooth;


public class ASYMCalculation extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	ParamVolume maskVol;	// 3-D Volume containing Mask Data [0 or 1]
	ParamVolume CESTVol;	// 4-D Volume containing CEST data with a range of offsets
	ParamFile freqOffset; // .txt file containing a list of the frequency offset values
	ParamInteger windowsize; // Size of the window for smoothing function
	ParamInteger interpnum;	// Number of Points for interpolation
//	ParamOption smooth; // Option to smooth the right-hand side, the left-hand side, both or none
	ParamBoolean interpolate;	// Choose to interpolate spectrum
	ParamOption ppm;
		
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
//	ParamVolumeCollection ASYMVol;	// An Image Volume containing Asymmetry Data
	ParamVolumeCollection ASYMVols;	// An Image Volume Collection containing Asymmetry Data

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Asymmetry Calculation: Calculates the Asymmetry Between Negative and Positive Offsets for a 4-D CESTR Volume.";
	private static final String longDescription = "\nInput 1: 4-D CESTR Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequences]" +
	"\n\tNote 1: Volume must contain equal numbers of negative and positive offset frequencies as well as a zero offset." +
	"\nInput 2: List of Offset Frequencies (.txt)" +
	"\n\tNote: if the 4-D WASSR Volume contains reference (S_0) images, then" +
	"\n\tlabel the 'offset frequency' associated with the S_0 images as 'NaN'." +
	"\nInput 3: 3-D Mask Volume (x,y,z): " +
	" [rows,columns,slices]"+
	"\n\tNote 1: The Mask Voxel should be [0 or 1] indicating whether the voxel is included or not"+
	"\n\tNote 2: If no Mask Volume is defined a mask of all 1's (all true) will be used."+
	"\nInput 4: Interpolate Data? (Checkbox)" +
	"\n\tNote: Interpolates frequency data over a given number of points (Interpolation Number)" +
	"\nInput 5: Interpolation Number" +
	"\n\tNumber of Points Used in Interpolation" +
//	"\nInput 6: Smoothing Option" +
//	"\n\tAllows the user to smooth the right-hand, left-hand," +
//	"\n\tor both sides of the frequency spectrum (None will not" +
//	"\n\tsmooth the graph)" +
	"\nInput 7: Size of Averaging Window" +
	"\n\tThe window size used in the shifting average calculation (MUST BE ODD)" +
	"\n\tused by the smoothing function" +
//	"\nOutput 1: 4-D Asymmetry Volume (x,y,z,t): " +
//	" [rows,columns,slices,offset frequences]" +
//	"\n\tWhere t is the number of number of negative (or positive) S volumes in the CEST data set [(TOTAL-1)/2]."+
	"\nOutput 1: 4-D Asymmetry Volume Collection (x,y,z,t): " +
	" [rows,columns,slices,offset frequences]" +
	"\n\tWhere t is the number of number of negative (or positive) S volumes in the CEST data set [(TOTAL-1)/2]." +
	"\n\tEach Volume represents the indicated smoothing region (none,left,right,both)"+
	"\nOutput 2: Execution Time [sec]; " +
	"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("CEST");
		inputParams.setLabel("Asymmetry Calculation");
		inputParams.setName("Asymmetry Calculation");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt University");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.add(new Citation(""));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);

		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		inputParams.add(CESTVol = new ParamVolume("CESTR Volume (4D)",null,-1,-1,-1,-1));
		inputParams.add(freqOffset=new ParamFile("List of Offset Frequencies (.txt)",new FileExtensionFilter(new String[]{"txt"})));
		inputParams.add(ppm = new ParamOption("X-Axis Points", new String[]{"HZ","ppm"}));
		inputParams.add(maskVol = new ParamVolume("Mask Volume Map (3D)",null,-1,-1,-1,1));
		maskVol.setMandatory(false);
		inputParams.add(interpolate = new ParamBoolean("Interpolate Data?",true)); 
		inputParams.add(interpnum = new ParamInteger("Number of Interpolation Points",50));
		interpnum.setMandatory(false);
//		inputParams.add(smooth = new ParamOption("Smoothing Option", new String[] {"None","Right","Left","Both"}));
		inputParams.add(windowsize = new ParamInteger("Size of Averaging Window",7));
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
//		outputParams.add(ASYMVol = new ParamVolume("Asymmetry Data (4D)",VoxelType.FLOAT,-1,-1,-1,-1));
		outputParams.add(ASYMVols = new ParamVolumeCollection("Asymmetry Data (4D)",VoxelType.FLOAT,-1,-1,-1,-1));
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}


	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute() {
			this.setLabel("PARSING INPUTS");
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUT DATA");
			int window = windowsize.getInt();
			if(window % 2 == 0){ // Checks that the window size is odd
				JistLogger.logError(JistLogger.SEVERE, "Window Size Must Be ODD. Aborting.");
				return;
			}
			if(interpolate.getValue() && interpnum == null){ // Checks that interpnum is given if interpolation is required
				JistLogger.logError(JistLogger.SEVERE, "Number of Interpolation Points is required for Interpolation. Aborting.");
				return;
			}
			this.setLabel("PARSING INPUTS");
			ImageData cest = new ImageDataFloat(CESTVol.getImageData());	// Extract CEST data from volume
			int rcest = cest.getRows(); int ccest = cest.getCols(); int scest = cest.getSlices(); int tcest = cest.getComponents();	// Assign variables to CEST volume dimensions
			this.setTotalUnits(rcest);
			
			ImageData mask = null;	// Initialize Mask variable
			int rmask = 0; int cmask = 0; int smask = 0;	// Initialize Mask dimension variables
			
			if(maskVol.getImageData() != null){	// Only set mask variable if mask is not empty
				mask = new ImageDataFloat(maskVol.getImageData());	// Extract Mask Data from volume
				rmask = mask.getRows(); cmask = mask.getCols(); smask = mask.getSlices();	// Assign Mask Dimension Variables
				JistLogger.logOutput(JistLogger.CONFIG, "MASK VOLUMES ADDED");
				// Check if img and mask sizes match
				if(rcest != rmask || ccest != cmask || scest != smask){
					JistLogger.logError(JistLogger.SEVERE,"Image and Mask Volume Sizes do not match. Aborting.");
					return;
				}
			}
			
			Vector<Boolean> baseline = new Vector<Boolean>();	// Declare a Boolean Vector to Determine Baseline Values
			Vector<Double> offsetVal = new Vector<Double>();	// Declare a Double Vector to Store Read Values
			int ignoreCnt=0;	// Initialize the Counter for Ignored Values
			try{
				BufferedReader rdr = new BufferedReader(new FileReader(freqOffset.getValue()));	// Creates a FileReader from Input File
				String thisline = " "; // Initialize Line Variable
				thisline = rdr.readLine(); // Reads Next Line to Line Variable
				while(thisline!=null && !thisline.isEmpty()){	// Reads all Lines until End
					Double val = Double.valueOf(thisline);
					if(val==null)
						val = Double.NaN;
					if(val.isNaN() || val.isInfinite()) {	// Adds True Ignore Vector and Ignore Counter if NaN
						val=Double.NaN;
						baseline.add(true);
						ignoreCnt++;
					} 
					else	// Adds False to Ignore Vector (Doesn't Add to Ignore Counter)
						baseline.add(false);
					offsetVal.add(val);	// Adds value to Read Values Vector
					JistLogger.logOutput(JistLogger.CONFIG,"Shift: "+val);
					thisline = rdr.readLine(); // Reads Next Line and Repeats Loop
				}
			}
			catch(IOException e){
				JistLogger.logError(JistLogger.SEVERE, "Cannot parse input shift file");	// Displays Error if Cannot Parse		
			}
			JistLogger.logOutput(JistLogger.CONFIG, "OFFSET FREQUENCIES ADDED");
			
			boolean []baselineArray= new boolean[baseline.size()];	// Declares Ignore Array
			double[]offsetArray= new double[baseline.size()-ignoreCnt];	// Declares Array of Frequency Offset Values
			int idx =0;
			boolean zero = false;
			for(int i=0;i<baselineArray.length;i++) {	// Creates Ignore Array from Ignore Vector
				baselineArray[i] = baseline.get(i);
				if(!baselineArray[i]) {	// Assigns Values for Offsets if not Ignored
					offsetArray[idx]=offsetVal.get(i);
					if(offsetArray[idx] == 0)
						zero = true;
					idx++;
				}
			}
			
			int tasym = 0;
			if(!zero){ // Checks that tcest is odd (symmetry of offsets)
				tasym = tcest/2;
			}	
			else{
				tasym = (tcest-1)/2;	// Calculate size of each side
			}
			
			JistLogger.logOutput(JistLogger.CONFIG, "ARRAYS CREATED");
			// Check if t and shiftArray.length are the same
			JistLogger.logOutput(JistLogger.CONFIG, "Number of Image Volumes: "+tcest+"("+ignoreCnt+")"+"\nNumber of Offset Frequencies found in file: "+offsetArray.length );
			if (offsetArray.length != tcest){
				JistLogger.logError(JistLogger.SEVERE, "The number of Image Volumes and Offset Frequencies do not match. Aborting.");
				return;
			}
			
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
//			ImageData ASYMestimate = new ImageDataFloat(rcest,ccest,scest,tasym);	// Create ImageData variable for Asymmetry Volume (Only Components for (t-1)/2)
			ImageData ASYMestimateNone = new ImageDataFloat(rcest,ccest,scest,tasym);	// Create ImageData variable for Asymmetry Volume (Only Components for (t-1)/2) No Smoothing
			ImageData ASYMestimateLeft = new ImageDataFloat(rcest,ccest,scest,tasym);	// Create ImageData variable for Asymmetry Volume (Only Components for (t-1)/2) Left Smoothed
			ImageData ASYMestimateRight = new ImageDataFloat(rcest,ccest,scest,tasym);	// Create ImageData variable for Asymmetry Volume (Only Components for (t-1)/2) Right Smoothed
			ImageData ASYMestimateBoth = new ImageDataFloat(rcest,ccest,scest,tasym);	// Create ImageData variable for Asymmetry Volume (Only Components for (t-1)/2) Both Smoothed
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/
			this.setLabel("ASYM CALC");
			this.setTotalUnits(rcest);
			JistLogger.logOutput(JistLogger.INFO, "RUNNING CORE ALGORITHMS");
			double[][][][] left = new double[rcest][ccest][scest][tasym];	// Initialize negative offset volumes
			double[][][][] right = new double[rcest][ccest][scest][tasym];	// Initialize positive offset volumes
			idx = 0;
			boolean maskval = true;
			for(int i=0;i<rcest;i++){	// Calculate Asymmetry for every voxel
				this.setCompletedUnits(i);
				for(int j=0;j<ccest;j++){
					for(int k=0;k<scest;k++){
						if(mask != null){
							maskval = mask.getBoolean(i,j,k);
						}
						if(maskval){	// Calculate if mask volume is empty or voxel is inside mask
								idx = 0;
								double[] interpx = null;
								double[] interpy = null;
								double[] smoothval = null;
								double[] y = new double[tcest];
								Spline smoothfunc = null;
								for(int m=0;m<tcest;m++){
									y[idx] = cest.getDouble(i,j,k,m);
									idx++;
								}
								Spline func = new Spline(offsetArray,y);
								CESTSmooth smoothclass = new CESTSmooth();
								int interp = 0;
								if(interpolate.getValue())	// If interpolation is required, gets value from interpnum
									interp = interpnum.getInt();
								else{
									if(zero)
										interp = y.length/2+1;	// If not, original number of points is used
									else
										interp = y.length/2;
								}
//								switch(smooth.getIndex()){	// Cases based on smoothing option
//								case 0:	// No smoothing
									idx = 0;
									for(int m=0;m<tasym;m++){	// Calculate for the correct number of volumes
										left[i][j][k][idx] = cest.getDouble(i,j,k,m);	// Assign negative offset volumes to left (starting at 1)
										right[i][j][k][idx] = cest.getDouble(i,j,k,(tcest-1)-m);	// Assign positive offset values to right (starting at end)
										switch(ppm.getIndex()){
										case 0:
//											ASYMestimate.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateNone.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
										break;
										case 1:
//											ASYMestimate.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateNone.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											break;
										}
										idx++;
									}
//									break;
//								case 1:	// Right-hand smoothing
									interpx = new double[interp];
									interpy = new double[interp];
									smoothval = new double[interp];
									interpx = smoothclass.getxinterp_right(func,interp,offsetArray);	// Interpolate X-values
									interpy = smoothclass.getyinterp_right(func,interp,offsetArray);	// Interpolate Y-values
									smoothval = smoothclass.getsmooth(interpy,window);	//  Get Smoothed Values
									smoothfunc = new Spline(interpx,smoothval);	// Make Spline from smoothed values
									idx = 0;
									for(int m=0;m<tasym;m++){ // Assigns left and right to respective intensity values
										left[i][j][k][idx] = y[m];
										right[i][j][k][idx] = smoothfunc.spline_value(offsetArray[(tcest-1)-m]);
										switch(ppm.getIndex()){
										case 0:
//											ASYMestimate.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateRight.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
										break;
										case 1:
//											ASYMestimate.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateRight.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											break;
										}
										idx++;
									}
//									break;
//								case 2: // Left-hand smoothing
									interpx = new double[interp];
									interpy = new double[interp];
									smoothval = new double[interp];
									interpx = smoothclass.getxinterp_left(func,interp,offsetArray);	// Interpolate X-values
									interpy = smoothclass.getyinterp_left(func,interp,offsetArray);	// Interpolate Y-values
									smoothval = smoothclass.getsmooth(interpy,window);	//  Get Smoothed Values
									smoothfunc = new Spline(interpx,smoothval);	// Make Spline from smoothed values
									idx = 0;
									for(int m=0;m<tasym;m++){// Assigns left and right to respective intensity values
										right[i][j][k][idx] = y[(tcest-1)-m];
										left[i][j][k][idx] = smoothfunc.spline_value(offsetArray[m]);
										switch(ppm.getIndex()){
										case 0:
//											ASYMestimate.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateLeft.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
										break;
										case 1:
//											ASYMestimate.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateLeft.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											break;
										}
										idx++;
									}
//									break;
//								case 3:	// Smooth both
									double[] interpx1 = new double[interp];
									double[] interpy1 = new double[interp];
									double[] smoothval1 = new double[interp];
									double[] interpx2 = new double[interp];
									double[] interpy2 = new double[interp];
									double[] smoothval2 = new double[interp];
									interpx1 = smoothclass.getxinterp_left(func,interp,offsetArray);	// Interpolate X-values (Left)
									interpy1 = smoothclass.getyinterp_left(func,interp,offsetArray);	// Interpolate Y-values (Left)
									smoothval1 = smoothclass.getsmooth(interpy1,window);	//  Get Smoothed Values (Left)
									interpx2 = smoothclass.getxinterp_right(func,interp,offsetArray);	// Interpolate X-values (Right)
									interpy2 = smoothclass.getyinterp_right(func,interp,offsetArray);	// Interpolate Y-values (Right)
									smoothval2 = smoothclass.getsmooth(interpy2,window);	//  Get Smoothed Values (Right)
									Spline smoothfunc1 = new Spline(interpx1,smoothval1);	// Make Spline from smoothed values (Left)
									Spline smoothfunc2 = new Spline(interpx2,smoothval2);	// Make Spline from smoothed values (Right)
									idx = 0;
									for(int m=0;m<tasym;m++){// Assigns left and right to respective intensity values
										right[i][j][k][idx] = smoothfunc2.spline_value(offsetArray[(tcest-1)-m]);
										left[i][j][k][idx] = smoothfunc1.spline_value(offsetArray[m]);
										switch(ppm.getIndex()){
										case 0:
//											ASYMestimate.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateBoth.set(i,j,k,idx,right[i][j][k][idx]-left[i][j][k][idx]);  // Calculate ASYM
										break;
										case 1:
//											ASYMestimate.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											ASYMestimateBoth.set(i,j,k,idx,left[i][j][k][idx]-right[i][j][k][idx]);  // Calculate ASYM
											break;
										}
										idx++;
									}
//									break;
//								}
							}
						else{	// If outside of mask
							idx = 0;
							for(int m=0;m<tasym;m++){	// Set asymmetry to zero for all volumes
//								ASYMestimate.set(i,j,k,idx,0);
								ASYMestimateNone.set(i,j,k,idx,0); 
								ASYMestimateLeft.set(i,j,k,idx,0); 
								ASYMestimateRight.set(i,j,k,idx,0); 
								ASYMestimateBoth.set(i,j,k,idx,0); 
								idx++;
							}		
						}
						}
					}
				}
			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			ASYMestimateNone.setName(cest.getName()+"_asym_none");	// Set ASYMestimate Name
			ASYMestimateNone.setHeader(cest.getHeader());	// Set ASYMestimate Header
			ASYMVols.add(ASYMestimateNone);	// Assigns Local Variable to Output Parameter
			ASYMestimateLeft.setName(cest.getName()+"_asym_left");	// Set ASYMestimate Name
			ASYMestimateLeft.setHeader(cest.getHeader());	// Set ASYMestimate Header
			ASYMVols.add(ASYMestimateLeft);	// Assigns Local Variable to Output Parameter
			ASYMestimateRight.setName(cest.getName()+"_asym_right");	// Set ASYMestimate Name
			ASYMestimateRight.setHeader(cest.getHeader());	// Set ASYMestimate Header
			ASYMVols.add(ASYMestimateRight);	// Assigns Local Variable to Output Parameter
			ASYMestimateBoth.setName(cest.getName()+"_asym_both");	// Set ASYMestimate Name
			ASYMestimateBoth.setHeader(cest.getHeader());	// Set ASYMestimate Header
			ASYMVols.add(ASYMestimateBoth);	// Assigns Local Variable to Output Parameter
//			ASYMestimate.setName(cest.getName()+"_asym_"+smooth.getString());	// Set ASYMestimate Name
//			ASYMestimate.setHeader(cest.getHeader());	// Set ASYMestimate Header
//			ASYMVol.setValue(ASYMestimate);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
