package edu.vanderbilt.VUIIS.plugins;


import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.PipeAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parser.MipavPluginParser;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.jhu.ece.iacl.plugins.registration.MedicAlgorithmFLIRT;
import edu.vanderbilt.VUIIS.algorithms.GaussianFunction;
import edu.vanderbilt.VUIIS.algorithms.MSWASSRMasked;
import gov.nih.mipav.view.dialogs.ActionDiscovery;


public class FlattenImage extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	private ParamVolume CESTvol;	// 4-D Volume containing CEST data with a range of offsets
	
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
	private ParamVolume flatVol; // An Image Volume containing CEST Data
	private ParamVolume changeVol;

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "FLIRT Registration for CEST Volumes";
	private static final String longDescription = "\nInput 1: 4-D CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies] " +
	"\nInput 2: 3-D Anatomical Volume (x,y,z):" +
	" [rows,columns,slices]" +
	"\nInput 3: List of Offset Frequencies (.txt)" +
	"\n\tLabel the 'offset frequencies' associated with the S_0 images as 'NaN'." +
	"\nOutput 1: 4-D Registered CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies]" +
	"\nOutput 2: Execution Time [sec]; " +
	"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("Utilities");
		inputParams.setLabel("Flatten Image");
		inputParams.setName("Flatten Image");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt University");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.add(new Citation(""));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);

		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		inputParams.add(CESTvol = new ParamVolume("CEST Volume (4D)",null,-1,-1,-1,-1));
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
		outputParams.add(flatVol = new ParamVolume("Flattened CEST Volume (4D)",VoxelType.FLOAT,-1,-1,-1,-1));
		outputParams.add(changeVol = new ParamVolume("Flattening Changes Volume (4D)",VoxelType.FLOAT,-1,-1,-1,-1));
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute(this);
	}
	

	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute(ProcessingAlgorithm parent) {
			this.setLabel("PARSING INPUTS");
			
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUT DATA");
			this.setLabel("PARSING INPUTS");
			
			ImageData cest = CESTvol.getImageData();	// Extract CEST Data from Volume
			int rcest = cest.getRows(); int ccest = cest.getCols(); int scest = cest.getSlices(); int tcest = cest.getComponents();	// Assign Variables for Dimensions of Volume
			
			
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
			ImageData flatData = new ImageDataFloat(rcest,ccest,scest,tcest);
			ImageData changeData = new ImageDataFloat(rcest,ccest,scest,tcest);
			
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "FLATTENING IMAGE");
			this.setLabel("FLATTEN IMAGE");
			this.setTotalUnits(rcest);
			float[] slice_max = new float[scest];
			MSWASSRMasked core = new MSWASSRMasked();
			for(int k=0;k<scest;k++){
				slice_max[k] = (float) core.getSliceMax(cest,k);
			}
			for(int i=0;i<rcest;i++){
				this.setCompletedUnits(i);
				for(int j=0;j<ccest;j++){	
					for(int k=0;k<scest;k++){
						Vector<Double> list = new Vector<Double>();
						double[] y = new double[tcest];
						for(int m=0;m<tcest;m++){
							list.add(cest.getDouble(i,j,k,m));
							y[m] = cest.getDouble(i,j,k,m);
						}
						double median = 0;
						Collections.sort(list);
						if (list.size() % 2 == 1)
							median = list.get((list.size()+1)/2-1);
						else{
							double lower = list.get(list.size()/2-1);
							double upper = list.get(list.size()/2);
							median = (lower + upper) / 2.0;
						}
						for(int m=0;m<tcest;m++){
							double delta = 0;
							if(y[m]!=0){
								if(y[m]>1.25*median || y[m]<0.75*median){
									delta = median/y[m];
									flatData.set(i,j,k,m,delta*y[m]/slice_max[k]);
									changeData.set(i,j,k,m,delta/slice_max[k]);
								}
								else{
									flatData.set(i,j,k,m,y[m]/slice_max[k]);
									changeData.set(i,j,k,m,1/slice_max[k]);
								}
							}
							else{
								flatData.set(i,j,k,m,y[m]/slice_max[k]);
								changeData.set(i,j,k,m,1/slice_max[k]);
							}
						}
					}
				}
			}

			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			flatData.setName(cest.getName()+"_flat");	// Set CEST volume Name
			flatData.setHeader(cest.getHeader());	// Set CEST volume Header
			flatVol.setValue(flatData);	// Assigns Local Variable to Output Parameter
			changeData.setName(cest.getName()+"_change");	// Set CEST volume Name
			changeData.setHeader(cest.getHeader());	// Set CEST volume Header
			changeVol.setValue(changeData);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
