package edu.vanderbilt.VUIIS.plugins;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;


import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.VUIIS.algorithms.MSWASSRMasked;


public class MaxThresMask extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	private ParamVolume imgVolume;	// 3-D or 4-D Volume containing Image data
	private ParamFloat maskThres;	// Percent Threshold for Mask
		
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
	private ParamVolume maskVolume;	// A Volume of Masks for each Slice of Data

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Threshold Mask:";
	private static final String longDescription =
			"\n Creates a mask based on a threshold percentage of the maximum value." +
			"\nInput 1: 3-D or 4-D Image Volume (x,y,z(,t)): " +
			" [rows,columns,slices(,components)]" +
			"\nInput 2: Mask Threshold (% of Max)"+
			"\n\tImage values above threshold for a certain slice return true(1) in the mask volume" +
			"\n\tImage values below threshold for a certain slice return false(0) in the mask volume"+
			"\nOutput 1: Mask Volume [0 or 1] (x,y,z): " +
			" [rows,columns,slices]" +
			"\n\t image volume showing mask used for each slice (0 denotes false, 1 denotes true)" +
			"\nOutput 2: Execution Time [sec] " +
			"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("CEST");
		inputParams.setLabel("Threshold Mask");
		inputParams.setName("Threshold Mask");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt Univeristy");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);

		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		inputParams.add(imgVolume=new ParamVolume("Image Data (3D or 4D)",null,-1,-1,-1,-1));
		inputParams.add(maskThres = new ParamFloat("Mask Threshold (% of Max)",0,100,10f));
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
		outputParams.add(maskVolume = new ParamVolume("Boolean Mask Volume [0 or 1]",VoxelType.FLOAT,-1,-1,-1,1));
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}


	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute() {
			this.setLabel("PARSING INPUTS");
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUTS");
			ImageData scalarVol=new ImageDataFloat(imgVolume.getImageData());	// Extract Image Data from Volume
			int r=scalarVol.getRows(), c=scalarVol.getCols(), s=scalarVol.getSlices();	// Assign Variables for Dimensions of Volume
			this.setTotalUnits(s);
					
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
			ImageData maskData = new ImageDataFloat(r,c,s,1); // Declares Mask Data Variable
			
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/
			this.setLabel("MASKING");
			JistLogger.logOutput(JistLogger.INFO, "RUNNING CORE ALGORITHMS");
			MSWASSRMasked core = new MSWASSRMasked();	// Creates Algorithm Variable
			JistLogger.logOutput(JistLogger.INFO, "CALCULATING WASSR SHIFT");
			double sliceMax = 0;
			maskThres.setValue(maskThres.getFloat()/100);
			for(int k=0;k<s;k++) {
				sliceMax = core.getSliceMax(scalarVol,k); // Gets max value for slice
				this.setCompletedUnits(k);
				for(int i=0;i<r;i++){	// Calculates WASSR Shift for each Voxel
					for(int j=0;j<c;j++) {
						if(scalarVol.getDouble(i,j,k,1) > maskThres.getFloat()*sliceMax){ // Only checks against threshold if mask is selected or If value is above threshold
							maskData.set(i,j,k,1);
						}
						else{
							maskData.set(i,j,k,0);
						}
					}
				}
			}
			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			maskData.setName(scalarVol.getName()+"_mask");	// Sets filename for maskData
			maskData.setHeader(scalarVol.getHeader());	// Sets file header for maskData
			maskVolume.setValue(maskData);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
