package edu.vanderbilt.masi.algorithms.adaboost;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.utility.JistLogger;

public class SegAdapterTesting extends AbstractCalculation {
	
	private int [] tlabels;
	private int [][] feature_rads;
	private int [] dilations;
	private int [] numiters;
	private int num_channels;
	private int num_labels;
	private SegAdapterImageTesting im;
	private WeakLearner [][] learners;
		
	public SegAdapterTesting(ParamVolume est_vol,
					         ParamVolumeCollection feature_vols,
					         ParamFileCollection json_files,
					         boolean [][][][] initial_masks) {
		
		// read some initial information from the json files
		num_labels = json_files.size();
		dilations = new int [num_labels];
		tlabels = new int [num_labels];
		numiters = new int [num_labels];
		feature_rads = new int [num_labels][3];
		read_initial_json_files(json_files);
		num_channels = feature_vols.size();
		int fusion_type = AdaBoost.FUSION_TYPE_MEAN_CLASSIFIER; // hard-coded, doesn't matter for applying a single classifier
		
		// load the pertinent image information
		im = new SegAdapterImageTesting(est_vol, feature_vols, tlabels, dilations, feature_rads, initial_masks);
		im.print_info();
		
		// process the image information
		for (int l = 0; l < num_labels; l++) {
			
			JistLogger.logOutput(JistLogger.INFO, String.format("*** Processing Label %d ***", tlabels[l]));
			
			// set the values for this label
			int [] feature_rad = feature_rads[l];
			int dilation = dilations[l];
			int tlabel = tlabels[l];
			int numiter = numiters[l];
			int win_size = (2*feature_rad[0]+1) * (2*feature_rad[1]+1) * (2*feature_rad[2]+1);
			int num_features = ((1 + num_channels) * win_size + 3) * 4 - 3; 
			int num_samples = im.get_num_samples(l);
			
			JistLogger.logOutput(JistLogger.INFO, String.format("-> Found the following information:"));
			JistLogger.logOutput(JistLogger.INFO, String.format("Feature Radius: %dx%dx%d", feature_rad[0], feature_rad[1], feature_rad[2]));
			JistLogger.logOutput(JistLogger.INFO, String.format("Dilation: %d", dilation));
			JistLogger.logOutput(JistLogger.INFO, String.format("Number of Iterations (classifiers): %d", numiter));
			JistLogger.logOutput(JistLogger.INFO, String.format("Number of Feature Channels: %d", num_channels));
			JistLogger.logOutput(JistLogger.INFO, String.format("Number of Samples to Classify: %d", num_samples));

			// get the classifiers for this label
			JistLogger.logOutput(JistLogger.INFO, String.format("-> Reading AdaBoost Parameters: %s", json_files.getValue(l).getName()));
			learners = new WeakLearner [1][numiter];
			read_json_arrays(json_files.getValue(l), numiter);
			
			// process this image
			im.process(num_features, tlabel, l, dilation, feature_rad, numiter, learners, (initial_masks == null) ? null : initial_masks[l], fusion_type);
			
			JistLogger.logOutput(JistLogger.INFO, "");
			
		}
	}
	
	public ImageData get_estimate() { return(im.get_estimate()); }
	
	private void read_json_arrays(File json_file, int numiter) {
		try {
			// open up the JSON Object
			BufferedReader b_reader = new BufferedReader(new FileReader(json_file));
			String in = b_reader.readLine();
			JSONObject obj = new JSONObject(in);
			
			// get all of the learners for this object 
			JSONArray bwlList = obj.getJSONArray("Learners");
			
			// make sure we found the correct number of learners
			if (numiter != bwlList.length())
				throw new RuntimeException("Number of found learners " + bwlList.length() + " does not match the expected amount!" + numiter);
			
			// get the instance of each learner
			for (int i = 0; i < numiter; i++)
				learners[0][i] = WeakLearner.getInstance(bwlList.getJSONObject(i));
			
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (JSONException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	private void read_initial_json_files(ParamFileCollection files) {
		
		for (int i = 0; i < num_labels; i++) {
			try {
				BufferedReader b_reader = new BufferedReader(new FileReader(files.getValue(i)));
				String in = b_reader.readLine();
				JSONObject obj = new JSONObject(in);
				
				dilations[i] = obj.getInt("Dilation");
				tlabels[i] = obj.getInt("Label");
				numiters[i] = obj.getInt("Iters");
				feature_rads[i][0] = obj.getInt("RadX");
				feature_rads[i][1] = obj.getInt("RadY");
				feature_rads[i][2] = obj.getInt("RadZ");
			} catch (FileNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (JSONException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}
}
