package edu.vanderbilt.masi.plugins.classification;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.HashMap;
import java.util.zip.GZIPOutputStream;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.utility.FileUtil;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.algorithms.clasisfication.ErrorClassificationTrainingSetBuilder;

public class ErrorClassificationTrainingSetBuilderPlugin extends ProcessingAlgorithm {

	//Input Parameters
	ParamFileCollection feature_file;
	ParamVolumeCollection manual_segmentation;
	ParamVolumeCollection host_segmentation;
	ParamVolumeCollection mask;
	ParamInteger max_samples;
	ParamInteger dilation_distance;

	//Output Parameters
	ParamFileCollection label_features;

	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Break Parameters into training and testing";
	private static final String longDescription = "";

	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard","andrew.j.plassard@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);

		inputParams.setPackage("MASI");
		inputParams.setCategory("Classification");
		inputParams.setLabel("Training Set Builder");
		inputParams.setName("Training_Set_Builder");

		inputParams.add(feature_file = new ParamFileCollection("Feature File"));
		feature_file.setMandatory(true);

		inputParams.add(manual_segmentation = new ParamVolumeCollection("Manual Segmentation"));
		manual_segmentation.setMandatory(true);

		inputParams.add(host_segmentation = new ParamVolumeCollection("Host Segmentation"));
		host_segmentation.setMandatory(true);

		inputParams.add(mask = new ParamVolumeCollection("Mask"));
		mask.setMandatory(true);

		inputParams.add(max_samples = new ParamInteger("Maximum Number of Voxels per Label"));
		max_samples.setMandatory(false);
		max_samples.setValue(100000);
		
		inputParams.add(dilation_distance = new ParamInteger("Distance to dilate mask when searching for incorrect voxels"));
		dilation_distance.setMandatory(false);
		dilation_distance.setValue(3);

	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(label_features = new ParamFileCollection("Labeled Features"));

	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		File outdir = new File(
				this.getOutputDirectory() +
				File.separator +
				FileUtil.forceSafeFilename(this.getAlgorithmName()) +
				File.separator);
		outdir.mkdirs();
		ImageData im = host_segmentation.getParamVolume(0).getImageData();
		HashMap<Integer,BufferedWriter> files = new HashMap<Integer,BufferedWriter>();
		try{
			for(int i=0;i<im.getRows();i++){
				for(int j=0;j<im.getCols();j++){
					for(int k=0;k<im.getSlices();k++){
						int label = im.getInt(i, j, k);
						if(!files.containsKey(label)){
							File f = new File(outdir, String.format("Feature_Label_%d.json.gz",label));
							FileOutputStream fstream = new FileOutputStream(f);
							OutputStreamWriter ow = new OutputStreamWriter(new GZIPOutputStream(fstream),"utf-8");
							BufferedWriter bw = new BufferedWriter(ow);
							files.put(label, bw);
							this.label_features.add(f);
						}
					}
				}
			}

			for(int i=0;i<host_segmentation.size();i++){
				JistLogger.logOutput(JistLogger.INFO, "Starting number "+i);
				ErrorClassificationTrainingSetBuilder E = new ErrorClassificationTrainingSetBuilder(this.feature_file.getValue(i) ,this.manual_segmentation.getParamVolume(i) ,this.host_segmentation.getParamVolume(i),this.mask.getParamVolume(i),files,this.max_samples,this.dilation_distance);
			}
			for(BufferedWriter b:files.values()) b.close();
		}catch(IOException e){
			e.printStackTrace();
		}
	}
}
