package edu.vanderbilt.masi.plugins.classification;

import java.util.Collections;
import java.util.LinkedList;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;
import edu.jhu.ece.iacl.jist.utility.JistLogger;

public class IntensityNormalizer extends ProcessingAlgorithm {

	//Input Parameters
	ParamVolumeCollection intensity_input;
	ParamVolumeCollection image_mask;

	//Output Parameters
	ParamVolumeCollection intensity_output;

	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.6 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Normalizes Intensity to 0-1";
	private static final String longDescription = "";

	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard","andrew.j.plassard@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);

		inputParams.setPackage("MASI");
		inputParams.setCategory("Feature Calculation");
		inputParams.setLabel("Intensity Normalizer");
		inputParams.setName("Intensity_Normalizer");

		inputParams.add(intensity_input = new ParamVolumeCollection("Intensity Volume"));
		this.intensity_input.setMandatory(true);

		inputParams.add(image_mask = new ParamVolumeCollection("Mask"));
		this.image_mask.setMandatory(true);

	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(intensity_output = new ParamVolumeCollection("Normalized Intensity"));
	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		JistLogger.logOutput(JistLogger.INFO,"Starting Normalization");
		for(int l=0;l<this.intensity_input.size();l++){
			ImageData im = this.intensity_input.getParamVolume(l).getImageData();
			ImageData ma = this.image_mask.getParamVolume(l).getImageData();
			LinkedList<Float> vals = new LinkedList<Float>();
			for(int i=0;i<im.getRows();i++)
				for(int j=0;j<im.getCols();j++)
					for(int k=0;k<im.getSlices();k++)
						if(ma.getBoolean(i, j,k)) vals.add(im.getFloat(i, j, k));

			Collections.sort(vals);
			float fifth = vals.get((int) Math.round(0.05*vals.size()));
			float ninefifth = vals.get((int) Math.round(0.95*vals.size()));
			JistLogger.logOutput(JistLogger.INFO, "Normalizing distribution to "+fifth+" "+ninefifth);
			for(int i=0;i<im.getRows();i++)
				for(int j=0;j<im.getCols();j++)
					for(int k=0;k<im.getSlices();k++){
						float val = im.getFloat(i,j,k);
						if(val<=fifth) im.set(i,j,k,0);
						else
							im.set(i, j, k,1000*(im.getFloat(i, j,k)-fifth)/ninefifth);
					}



			this.intensity_output.add(im);
		}
	}


}
