package edu.vanderbilt.masi.plugins.labelfusion;

import java.util.List;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.vanderbilt.masi.algorithms.labelfusion.simple.NonLocalSIMPLE;

public class PluginNonLocalSIMPLE extends ProcessingAlgorithm {

	private static final String cvsversion = "$Revision: 1.4 $";
	private static final String revnum = cvsversion.replace("Revision: ", "")
			.replace("$", "").replace(" ", "");
	private static final String shortDescription = "Runs the Non-Local SIMPLE Label Fusion Algorithm.";
	private static final String longDescription = "";

	// inputs
	public ParamVolumeCollection obsLabs;
	public ParamVolume targetImage;
	public ParamVolumeCollection obsImgs;

	public ParamInteger patchRadius;
	public ParamInteger searchRadius;
	public ParamFloat searchWeight;
	public ParamInteger maxPatches;
	public ParamFloat minPatchWeight;

	public ParamInteger poolRegion;

	public ParamInteger maxIter;
	public ParamFloat epsilon;
	
	public ParamFloat numDevs;

	// outputs
	public ParamVolume segmentation;
	public ParamVolumeCollection localWeights;

	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard",
				"andrew.j.plassard@vanderbilt.edu",
				"https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(true);
		info.setStatus(DevelopmentStatus.ALPHA);

		inputParams.setPackage("MASI");
		inputParams.setCategory("Label Fusion");
		inputParams.setName("Non-Local SIMPLE");
		inputParams.setLabel("Non-Local SIMPLE");

		inputParams.add((obsLabs = new ParamVolumeCollection(
				"Registered Label Volumes")));
		obsLabs.setMandatory(true);
		inputParams.add((obsImgs = new ParamVolumeCollection(
				"Registered Image Volumes")));
		obsImgs.setMandatory(true);
		inputParams.add((targetImage = new ParamVolume("Target Image Volume")));
		targetImage.setMandatory(true);

		inputParams
				.add((patchRadius = new ParamInteger("Patch Radius (voxels)")));
		patchRadius.setMandatory(false);
		patchRadius.setValue(2);
		inputParams.add((searchRadius = new ParamInteger(
				"Search Radius (voxels)")));
		searchRadius.setMandatory(false);
		searchRadius.setValue(3);
		inputParams.add((searchWeight = new ParamFloat(
				"Search Standard Deviation")));
		searchWeight.setMandatory(true);
		searchWeight.setValue(1f);
		inputParams.add((maxPatches = new ParamInteger(
				"Maximum Number of Patches to Keep")));
		maxPatches.setMandatory(false);
		maxPatches.setValue(100);
		inputParams.add((minPatchWeight = new ParamFloat(
				"Minimum Weight to Keep a Patch")));
		minPatchWeight.setMandatory(false);
		minPatchWeight.setValue(0.05f);
		
		inputParams.add((numDevs = new ParamFloat("Number of standard deviations below mean to run patch selection")));
		numDevs.setMandatory(false);
		numDevs.setValue(2);

		inputParams.add((poolRegion = new ParamInteger(
				"Pool region Size (<0 is whole image)")));
		poolRegion.setMandatory(false);
		poolRegion.setValue(-1);
		inputParams.add((epsilon = new ParamFloat(
				"Mean change in weights for convergence")));
		epsilon.setMandatory(false);
		epsilon.setValue(1e-4f);
		inputParams.add((maxIter = new ParamInteger(
				"Maximum number of iterations")));
		maxIter.setMandatory(false);
		maxIter.setValue(10);
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add((segmentation = new ParamVolume("Segmentation")));

	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		List<ImageData> labs = obsLabs.getImageDataList();
		List<ImageData> imgs = obsImgs.getImageDataList();
		ImageData target = targetImage.getImageData();
		int pr = patchRadius.getInt();
		int sr = searchRadius.getInt();
		float sw = searchWeight.getFloat();
		float minWeight = minPatchWeight.getFloat();
		int maxNumPatches = maxPatches.getInt();
		int pool = poolRegion.getInt();
		float eps = epsilon.getFloat();
		int max = maxIter.getInt();
		float nD = numDevs.getFloat();
		NonLocalSIMPLE nls;
		try {
			nls = new NonLocalSIMPLE(labs, imgs, target, pr, sr, sw,
					minWeight, maxNumPatches, pool, eps, max, nD);
			segmentation.setValue(nls.getSegmentation());
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		

	}

}
