package bl.diffusion;

import java.util.Iterator;

/**
 * Created by IntelliJ IDEA.
 * User: bennett
 * Date: Nov 23, 2005
 * Time: 3:12:23 PM
 * To change this template use Options | File Templates.
 * ************************************
 * Magnetic Resonance in Medicine Final Project
 * Released: December 1, 2005
 *
 * class RandomWalkSimulator
 *      General programatic interface to the Brownian Motion Simulator.
 *      The order of operations must be:
 *      1) Initialize Java Simulator
 *          javaSimulator = RandomWalkSimulator;
 *          javaSimulator.setSimTime(timeStep,diffusionTime);
 *      2) set the lattice and global information
 *          javaSimulator.setLatticeInfo;
 *      3) enter the geometry
 *          javaSimulator.createCompartment(geo.diffusivity, geo.permeability, geo.shape.vertices, geo.shape.faces-1);
 *      4) select number of spins and create them
 *          javaSimulator.setNumberSpins(numSpinPackets);
 *          javaSimulator.initializeRandomSpins();
 *      5) simulate the random walks (non-interacting)
 *          javaSimulator.takeAWalk();
 *      6) read out results
 *          resultPositions = javaSimulator.reportResultPositions();
 *          origPositions = javaSimulator.reportOriginalPositions();
 *
 * Copyright (C) 2005 Bennett Landman, bennett@bme.jhu.edu
 */
public class RandomWalkSimulator {

    // Set this flag to true to turn on plentiful commandline output
    public static int intenseDebug = 0;
    // This stores the global environment
    SimRandomWalk env;
    // The total diffusion time
    double endTime;
    // The time increment
    double timeStep;
    // The number of spin particles to track
    int numSpins;
    // The number of finite duration pulses used
    private SimFiniteGradient []finiteGradients = null;
   
    
    long numSteps;
    long numCollisions; 

    // Create a new simulator
    public RandomWalkSimulator() {
    	System.out.println(getClass().getCanonicalName()+"\t"+"***********************************************");
        System.out.println(getClass().getCanonicalName()+"\t"+"Random Walk Simulator v2.0 (September 10, 2008)");
        System.out.println(getClass().getCanonicalName()+"\t"+"***New, improved, documented.***");
        System.out.println(getClass().getCanonicalName()+"\t"+"Major Revision by: ");
        System.out.println(getClass().getCanonicalName()+"\t"+"\tBennett Landman, landman@jhu.edu");
        System.out.println(getClass().getCanonicalName()+"\t"+"\tJonathan Farrell, jfarrell@jhu.edu");
        System.out.println(getClass().getCanonicalName()+"\t"+"\t(C) Copyright 2008, Bennett Landman and Jonathan Farrell");
        System.out.println(getClass().getCanonicalName()+"\t"+"Released under Lesser GNU Public License (http://www.gnu.org/copyleft/lesser.html)");
        System.out.println(getClass().getCanonicalName()+"\t"+"\tNot for clinical use.");
        System.out.println(getClass().getCanonicalName()+"\t"+"\tNo warranty expressed or implied.");
        System.out.println(getClass().getCanonicalName()+"\t"+"\tUse at your own risk.");
        System.out.println(getClass().getCanonicalName()+"\t"+"Download at: http://www.nitrc.org/projects/randomwalks/");
        System.out.println(getClass().getCanonicalName()+"\t"+"");
        System.out.println(getClass().getCanonicalName()+"\t"+"History:");
        System.out.println(getClass().getCanonicalName()+"\t"+"\tv1.0 Magnetic Resonance in Medicine 2005, Final Project");
        System.out.println(getClass().getCanonicalName()+"\t"+"\t(C) Copyright 2005, Bennett Landman, landman@jhu.edu");        
        System.out.println(getClass().getCanonicalName()+"\t"+"***********************************************");		
        env = new SimRandomWalk();
        numSpins=0;
finiteGradients=null;
    }

    // Set the total diffusion time and the time step (in microseconds)
    public void setSimTime(double timeStep, double totalTime) {
        this.timeStep = timeStep;
        endTime = totalTime;
        env.setTimeStep(timeStep);
    }

    // Set the number of spins
    public void setNumberSpins(double numspins) {
        numSpins =(int)numspins;
    }

    // Set the global diffusion constant and the lattice size
    public void setLatticeInfo(double diff, double latX, double latY, double latZ, double T2) {
        // [diff] = um2/us, [lat] = um
        double latSize[] = new double[3];
        latSize[0]=latX;
        latSize[1]=latY;
        latSize[2]=latZ;
        env.setExtraComparmentParameters(diff,latSize,T2);
    }

    // Create randomly generated points (spin packets)
    public void initializeRandomSpins() {
        for(int i=0;i<numSpins;i++)
            env.addUniformRandomPoint();
    }
    
    // Create randomly generated points (spin packets)
    public void initializeRandomSpinsInside() {
        for(int i=0;i<numSpins;i++)
            env.addUniformRandomPointInside();
    }
    
    // Create randomly generated points (spin packets)
    public void initializeRandomSpinsOutside() {
        for(int i=0;i<numSpins;i++)
            env.addUniformRandomPointOutside();
    }
    
    public void initializeFixedSpins(double []pos) {
        for(int i=0;i<numSpins;i++)
            env.addPoint(new SimPoint(pos[0],pos[1],pos[2]));
    }

    // Perform the appropriate number of timesteps and display progress bar
    public void takeAWalk() {
    	SimPoint.resetCollisions();
        // Initialize the parameters of all points based on their respective compartments
        env.initializeDiffusion();
        long totalSteps = (long)Math.round(endTime/timeStep);

        long nSteps = Math.min(250,totalSteps);
        ProgressBar progressbar = new ProgressBar("RandomWalkSimulator: takeAWalk progress...",(int)nSteps);
        long stepInc = totalSteps/nSteps;
        progressbar.updateProgress();
        progressbar.setVisible(true);

        for(int i=0;i<totalSteps;i++){
        	if(finiteGradients!=null)
        		for(int k =0;k<finiteGradients.length;k++)
        			finiteGradients[k].setSimTime(i*timeStep);
            env.performTimestep(finiteGradients);
            if(0==i%stepInc)
                progressbar.updateProgress();
            if(!progressbar.open) {
            	System.out.println(getClass().getCanonicalName()+"\t"+"talkAWalk: Abort! Results left in an incosistent state.");
            }
        }
        if(intenseDebug>0)
            env.debugPrintPoints();
        progressbar.hide();
        progressbar.dispose();;
        progressbar = null;
        
        numSteps = totalSteps*numSpins;
        numCollisions = SimPoint.getCollisions();
    }
    
    public long[] getCollisionStats() {
    	long[]result = new long[2];
    	result[0] = numCollisions; 
    	result[1] = numSteps; 
    	return result;
    }

    // create a compartment
    public int createCompartment(double diffusivity, double permiability, double [][]vertices, double [][]faces, double T2) {
        Polyhedron shape = new Polyhedron(vertices,faces);
        SimCompartment compartment = new SimCompartmentPoly(shape,permiability,diffusivity, T2);
        return env.addCompartment(compartment);
    }

    public int createCompartment(SimCompartment compartment) {
    	return env.addCompartment(compartment);
    }

    // set the inside/outside relationship between compartments
    public void setCompartmentOutsideInside(int outside, int inside) {
        env.setCompartmentOutsideInside(outside,inside);
    }

    // export the original positions
    public double[][] reportOriginalPositions() {
        double [][]positions;
        positions = new double[numSpins][3];
        Iterator i = env.getPts();
        int j=0;
        while(i.hasNext() && j<numSpins) {
            SimPoint pt = (SimPoint)i.next();
            positions[j][0] = pt.originalPosition.x;
            positions[j][1] = pt.originalPosition.y;
            positions[j][2] = pt.originalPosition.z;
            j++;
        }
        if(i.hasNext()) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: More Spins Than Expected Encountered1!!");
        }
        if(j<numSpins) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: Fewer Spins Than Expected Encountered1!!");
        }
        return positions;
    }

    // export the result positions
    public double[][] reportResultPositions() {
        double [][]positions;
        positions = new double[numSpins][3];
        Iterator i = env.getPts();
        int j=0;
        while(i.hasNext() && j<numSpins) {
            SimPoint pt = (SimPoint)i.next();
            positions[j][0] = pt.currentPosition.x;
            positions[j][1] = pt.currentPosition.y;
            positions[j][2] = pt.currentPosition.z;
            j++;
        }
        if(i.hasNext()) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: More Spins Than Expected Encountered1!!");
        }
        if(j<numSpins) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: Fewer Spins Than Expected Encountered1!!");
        }
        return positions;
    }
    // export the T2 signal Attenuation changes
    public double[] reportT2SignalAttenuation() {
        double []T2SignalAttenuation;
        T2SignalAttenuation = new double[numSpins];
        Iterator i = env.getPts();
        int j=0;
        while(i.hasNext() && j<numSpins) {
            SimPoint pt = (SimPoint)i.next();
            T2SignalAttenuation[j] = Math.exp(pt.logT2Attenuation);
            j++;
        }
        if(i.hasNext()) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: More Spins Than Expected Encountered1!!");
        }
        if(j<numSpins) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: Fewer Spins Than Expected Encountered1!!");
        }
        return T2SignalAttenuation;
    }

    public double[] reportLogT2SignalAttenuation() {
        double []T2SignalAttenuation;
        T2SignalAttenuation = new double[numSpins];
        Iterator i = env.getPts();
        int j=0;
        while(i.hasNext() && j<numSpins) {
            SimPoint pt = (SimPoint)i.next();
            T2SignalAttenuation[j] = (pt.logT2Attenuation);
            j++;
        }
        if(i.hasNext()) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: More Spins Than Expected Encountered1!!");
        }
        if(j<numSpins) {
            System.out.println(getClass().getCanonicalName()+"\t"+"Warning: Fewer Spins Than Expected Encountered1!!");
        }
        return T2SignalAttenuation;
    }
    
    public double[][] reportFinitePulsePhase() {
    	if(finiteGradients==null)
    		return null;
    	double [][]phases;
    	phases = new double[numSpins][finiteGradients.length];
    	Iterator i = env.getPts();
    	int j=0;
    	while(i.hasNext() && j<numSpins) {
    		SimPoint pt = (SimPoint)i.next();
    		for(int k=0;k<finiteGradients.length;k++)
    			phases[j][k] = pt.getPhase(k);    		
    		j++;
    	}
    	if(i.hasNext()) {
    		System.out.println(getClass().getCanonicalName()+"\t"+"Warning: More Spins Than Expected Encountered1!!");
    	}
    	if(j<numSpins) {
    		System.out.println(getClass().getCanonicalName()+"\t"+"Warning: Fewer Spins Than Expected Encountered1!!");
    	}
    	return phases;
    }
    
    public void setNumberFiniteGradients(int num) {
    	if(num<=0) {
    		finiteGradients=null;
    	} else {
    		finiteGradients = new SimFiniteGradient[num];
    	}    	
    }
    
    public void setFiniteGradient(int index, SimFiniteGradient fg) {
    	if(finiteGradients==null)
    		throw new RuntimeException("No finite gradients set.");
   		if(index>=0 && index<finiteGradients.length)
    		finiteGradients[index]=fg;
    	else 
    		throw new RuntimeException("Invalid finite gradient index.");
    }
}
