package bl.diffusion;

public class SimCompartmentCylinder implements SimCompartment {

	private PT myCenter = null;
	private double myRadius = 0;
	private double myLength = 0; 
	// the diffusion coefficient in the compartment
	private double diffusionCoefficient;

	// The probability of crossing the membrane - us/um
	private double membranePermeability; 

	private double T2; 
	public double getT2() {return T2;};

	public SimCompartmentCylinder(PT center, double radius, double length, double membranePermeability,
			double diffusionCoefficient, double T2) {
		this.T2 = T2;
		myCenter = new PT(center);
		if(radius<0)
			throw new RuntimeException("Invalid radius");
		myRadius = radius;
		if(length<0)
			throw new RuntimeException("Invalid length");
		myLength = length;		
		this.membranePermeability = membranePermeability;
		if(diffusionCoefficient<=0)
			throw new RuntimeException("Invalid diffusionCoefficient");
		this.diffusionCoefficient = diffusionCoefficient;
	}

	public static double square(double x) {return x*x;};
	public boolean contains(PT p) {		
		if(p.z>(myCenter.z+myLength/2) || p.z<(myCenter.z-myLength/2) ) 
			return false;
		if((square(p.x-myCenter.x)+square(p.y-myCenter.y))>square(myRadius))
			return false;			
		return true;
	}


	private IntersectResult capInterset(PT a, PT b) {
		// check caps
		PT unitStep = b.minus(a);
		PT normUnitStep = unitStep.normalize();

		double minZ = myCenter.z-myLength/2;
		double uMin = (minZ-a.z)/unitStep.z;
		double maxZ = myCenter.z+myLength/2;
		double uMax = (maxZ-a.z)/unitStep.z;
		//		System.out.println(getClass().getCanonicalName()+"\t"+"capInterset "+unitStep+" "+minZ+" "+maxZ+" "+uMin+" "+uMax+" "+unitStep.times(uMin)+" "+a.plus(unitStep.times(uMin)));
		if(uMin>0 && uMin <= 1) {
			//			System.out.println(getClass().getCanonicalName()+"\t"+"CAP-");
			PT intersect = a.plus(unitStep.times(uMin));
			double A = square(intersect.x-myCenter.x)+square(intersect.y-myCenter.y);
			if(A<=this.myRadius*this.myRadius) {
				PT normal = new PT(0, 0, contains(a)?1:-1);
				PT eps = normUnitStep.times(IntersectResult.EPSILON);
				PT fwd = intersect.plus(eps);
				PT bck = intersect.minus(eps);
				if(contains(fwd)==contains(bck))
					return null;	//feature too small
				else
					return new IntersectResult(intersect,uMin,normal,fwd,bck);
			} else 
				return null; //new IntersectResult(intersect,uMin,new PT(0, 0, 1)); 				
		}
		if(uMax>0 && uMax <= 1) {
			//			System.out.println(getClass().getCanonicalName()+"\t"+"CAP+");
			PT intersect = a.plus(unitStep.times(uMax));
			double A = square(intersect.x-myCenter.x)+square(intersect.y-myCenter.y);
			if(A<=this.myRadius*this.myRadius) {
				PT normal = new PT(0, 0, contains(a)?-1:1);
				PT eps = normUnitStep.times(IntersectResult.EPSILON);
				PT fwd = intersect.plus(eps);
				PT bck = intersect.minus(eps);
				if(contains(fwd)==contains(bck))
					return null;	//feature too small
				else
					return new IntersectResult(intersect,uMax,normal,fwd,bck);
			} else 
				return null;
		}
		//		System.out.println(getClass().getCanonicalName()+"\t"+"null");
		return null;
	}

	public IntersectResult findFirstIntersection(PT a, PT b) {
		// Derived from http://local.wasp.uwa.edu.au/~pbourke/geometry/sphereline/
		double A = square(b.x-a.x)+square(b.y-a.y);
		double B = 2.*((b.x-a.x)*(a.x-myCenter.x)+
				(b.y-a.y)*(a.y-myCenter.y));
		double C = square(myCenter.x)+square(myCenter.y)+
		square(a.x)+square(a.y)-
		2.*(myCenter.x*a.x+myCenter.y*a.y)-
		myRadius*myRadius;

		double det  = B*B-4*A*C;	

		PT unitStep = b.minus(a);
		PT normUnitStep = unitStep.normalize();

		IntersectResult capResult = capInterset(a,b);
		if(det<0) {
			// no intersection w/ cylinder
			//			System.out.println(getClass().getCanonicalName()+"\t"+"det<0");
			//			System.out.println(getClass().getCanonicalName()+"\t"+"AA6");
			return capResult;
		}
		if(det==0) {
			//			System.out.println(getClass().getCanonicalName()+"\t"+"det=0");
			// exactly 1 intersection
			double u = -B/2/A;
			//			System.out.println(getClass().getCanonicalName()+"\t"+u);
			if(u<0 || u>1 || Double.isNaN(u)) {
				//				System.out.println(getClass().getCanonicalName()+"\t"+"AA5");
				return capResult; // no intersection with sides
			}
			PT p = a.plus((b.minus(a).times((double)u)));

			PT surfaceNormal=null;
			if(contains(a))
				surfaceNormal = myCenter.minus(p);
			else 
				surfaceNormal = p.minus(myCenter);
			surfaceNormal.z=0;
			surfaceNormal=surfaceNormal.times((double)(1./surfaceNormal.norm()));
			if(capResult!=null)
				if(u>capResult.fractionalDistance) {
					//					System.out.println(getClass().getCanonicalName()+"\t"+"AA3");
					return capResult;
				}
			//			System.out.println(getClass().getCanonicalName()+"\t"+"AA4");
			if(p.z>(myCenter.z+myLength/2) || p.z<(myCenter.z-myLength/2) ) 
				return capResult; //intersect is beyond the caps

			PT intersect = p;
			PT normal = surfaceNormal;
			PT eps = normUnitStep.times(IntersectResult.EPSILON);
			PT fwd = intersect.plus(eps);
			PT bck = intersect.minus(eps);
			if(contains(fwd)==contains(bck))
				return capResult;	//feature too small
			else
				return new IntersectResult(intersect,u,normal,fwd,bck);


		}
		//		System.out.println(getClass().getCanonicalName()+"\t"+"det>0");
		//		exactly 2 intersections
		det = Math.sqrt(det);
		double u1 = (-B+det)/2/A;
		double u2 = (-B-det)/2/A;
		double u = 0;
		if(u2>=0 && u2<=1)
			u=u2; //ray hits u2 first
		else if (u1>=0 && u1<=1)
			u=u1; //ray hits u1 first
		else {
			//			System.out.println(getClass().getCanonicalName()+"\t"+"AA2");
			return capResult;  // no intersections are between a and b
		}

		if(capResult!=null)
			if(u>capResult.fractionalDistance)
				return capResult;

		PT p = a.plus((b.minus(a).times((double)u)));

		PT surfaceNormal=null;
		if(contains(a))
			surfaceNormal = myCenter.minus(p);
		else 
			surfaceNormal = p.minus(myCenter);
		surfaceNormal.z=0;
		surfaceNormal=surfaceNormal.times((double)(1./surfaceNormal.norm()));
		//		System.out.println(getClass().getCanonicalName()+"\t"+"AA1 "+u+" "+u1+" "+u2);
		if(p.z>(myCenter.z+myLength/2) || p.z<(myCenter.z-myLength/2) ) 
			return capResult; //intersect is beyond the caps

		PT intersect = p;
		PT normal = surfaceNormal;
		PT eps = normUnitStep.times(IntersectResult.EPSILON);
		PT fwd = intersect.plus(eps);
		PT bck = intersect.minus(eps);
		if(contains(fwd)==contains(bck))
			return capResult;	//feature too small
		else
			return new IntersectResult(intersect,u,normal,fwd,bck);
	}

	// return the membrane transmission probability
	public double getMembranePermeability() {
		return membranePermeability;
	}

	// return the compartment diffusion coefficient
	public double getDiffusionCoefficient() {
		return diffusionCoefficient;
	}

}
