function VORPOLY = voronoiSphere(x,y,z,TRI)
% function VORPOLY = voronoiSphere(x,y,z,TRI)
%
% Compute a voronoi tesselation of the surface of a sphere
% based on the delauny TRI triangulation (optional) or the points
% (x,y,z).
%
% (C)opyright 2005, Bennett Landman, bennett@bme.jhu.edu
% Revision History:
% Created: 2/11/2005

% Either used the supplied delaunay triangulation or compute it.
if(~exist('TRI'))
    TRI = delaunaySphere(x,y,z);
end

VORPOLY = [];
for jpt = 1:max(TRI(:))
    % Go through each point on the sphere
    % to find its neighbors
    [triangles,j] = find(TRI==jpt);
    neighbors = unique(TRI(triangles,:));
    neighbors = neighbors(find(neighbors~=jpt));

    if(length(neighbors)>0)
        % Project the neighbors onto a sphere with the current point at the
        % center so that we can determine the order that the voronoi edges must
        % intersect.
        [xV, yV] = projectSphere([x(neighbors)' y(neighbors)' z(neighbors)'], [x(jpt) y(jpt) z(jpt)]);
        ang = sortrows([atan2(yV,xV) (1:length(yV))']);
        order = [ang(:,2)' ang(1,2)];

        % Find the midpoint between the current point and each of its neighbors
        [th_0, phi_0, r0] = cart2sph(x(jpt), y(jpt), z(jpt));
        [th,phi,r] = cart2sph(x(neighbors),y(neighbors),z(neighbors));
        [midphi,midth] = midPtOnSphere(phi_0, th_0,phi,th,.5);
        [xx,yy,zz] = sph2cart(midth,midphi,r);

        % Numerically evaluate the forward derivate at each midpoint
        [midphieps,midtheps] = midPtOnSphere(phi_0, th_0,phi(1:end),th(1:end),.5+1e-6);
        [xxeps,yyeps,zzeps] = sph2cart(midtheps,midphieps,r);
        vect_par = [xxeps-xx; yyeps-yy; zzeps-zz]/1e-6;

        % Find a vector perpendicular to both the radius at the mid point and
        % the line between the point and each neighbor
        % This vector will be in the plane of the voronoi edge. Find another
        % point in this plane (in addition to the midpoint).
        pt_perp = cross([xx; yy; zz],vect_par);
        [pp_th,pp_phi,pp_r] = cart2sph(xx+pt_perp(1,:),yy+pt_perp(2,:),zz+pt_perp(3,:));
        [pp_x,pp_y,pp_z] = sph2cart(pp_th,pp_phi,r);

        % now we have pt_perp and midphi, two points on the planes that define
        % the voronoi edges.

        % Go through the planes in the order that we found above and find where
        % adjacent planes intersect each other and the sphere. String together
        % these sets of points as the bounding polygon for the voronoi region.
        XF=[];
        Xpt = [x(jpt) y(jpt) z(jpt)];
        for j=1:(length(order)-1)
            ABCD1 = fitPlane([0 pp_x(order(j)) xx(order(j))],[0 pp_y(order(j)) yy(order(j))],[0 pp_z(order(j)) zz(order(j))]);
            ABCD2 = fitPlane([0 pp_x(order(j+1)) xx(order(j+1))],[0 pp_y(order(j+1)) yy(order(j+1))],[0 pp_z(order(j+1)) zz(order(j+1))]);

            [X0, SLOPE] = planeIntersect(ABCD1,ABCD2);
            [XA,XB] = intersectLineSphere(1,X0,SLOPE);
            da = sum((XA-Xpt).^2);
            db = sum((XB-Xpt).^2);
            % There are two possible points of intersection for these planes.
            % We choose the one closer to the point of interest.
            if(da<db)
                XF(j,:) = XA;
            else
                XF(j,:)=XB;
            end

        end

        VORPOLY{jpt} = XF;
    else
        VORPOLY{jpt} = [];
    end
end