package bl.diffusion;

import java.util.Iterator;

/**
 * Created by IntelliJ IDEA.
 * User: bennett
 * Date: Nov 20, 2005
 * Time: 9:51:16 AM
 * To change this template use Options | File Templates.
 * ************************************
 * Magnetic Resonance in Medicine Final Project
 * Released: December 1, 2005
 *
 * class SimPoint
 *      Represent a single spin packet
 *
 * Copyright (C) 2005 Bennett Landman, bennett@bme.jhu.edu
 */
public class SimPoint {

	public static int collisions = 0;
	// provide an error tolerance contant to deal with round off error
	public static double EPS = 1e-3f; //was -5
	// store the "original position" adjusted for lattice wrap around
	public PT originalPosition;
	// store the current position
	public PT currentPosition;
	// keep track of the local compartment surrounding this point
	private SimCompartment localEnv;
	// keep track of the current diffusion
	private double currentDiffusionCoefficient;
	// a putative step that is incrementally updated
	private PT step;
	// the stepsize    
	private double stepSize;
	// time step
	private double timeStep;
	// time step remaining for iterative steps
	private double timeStepLeft;
	// logT2 attenuations
	public double logT2Attenuation;
	// current phase
	private double phases[];

	private int recursionDepth;
	// create a new simulation point
	public SimPoint(double x,double y, double z) {    	
		this(new PT(x,y,z));
	}

	// create a new simulation point
	public SimPoint(PT pt) {
		originalPosition = pt;
		currentPosition = pt;
		localEnv = null;
		currentDiffusionCoefficient=0;
		step = null;
		stepSize = 0;
		logT2Attenuation=0;
		timeStep=0;
		if(RandomWalkSimulator.intenseDebug>0) {
			System.out.println(getClass().getCanonicalName()+"\t"+"PT(): "+pt);
		}
	}

	private double getT2(SimEnvironment globalEnv) {
		if(localEnv==null)
			return globalEnv.getT2();
		else
			return localEnv.getT2();
	}

	// calculate the step size due to Brownian motion
	public static double calcStepSize(double diffusionCoefficient, double timeStep) {
		if(RandomWalkSimulator.intenseDebug>8) {
			System.out.println("\t"+"PT.calcVelocity: "+Math.sqrt(6*diffusionCoefficient/timeStep));
		}
		return Math.sqrt(6*diffusionCoefficient*timeStep);
	}
	
	// calculate the step size due to Brownian motion - um/us
	public static double calcVelocity(double diffusionCoefficient) {		
		return calcStepSize(diffusionCoefficient,1);
	}

	// update the point parameters based on the current local environment
	public void update(SimCompartment env, double diffusionCoefficient, double timeStep) {
		localEnv = env;
		currentDiffusionCoefficient=diffusionCoefficient;
		stepSize = calcStepSize(currentDiffusionCoefficient,timeStep);
		this.timeStep=timeStep;
		if(RandomWalkSimulator.intenseDebug>8) {
			System.out.println(getClass().getCanonicalName()+"\t"+"PT.update: env:"+(null==env)+" "+diffusionCoefficient+" "+timeStep);
		}
	}

	// perform a single time step
	public void performStep(SimEnvironment globalEnv, SimFiniteGradient []finiteGradients) {
		recursionDepth=0;
		setRandDirection();
		if(finiteGradients!=null) {			
			updatePhases(finiteGradients,timeStep);
		}
		//		double z = this.currentPosition.z;
		//		double dz = this.step.z;
		PT cur = currentPosition;
		PT stp = step;		
		timeStepLeft=timeStep;
		SimCompartment env = localEnv;
		try {
			performStepLocal(globalEnv);
		} catch (RuntimeException e) {
			System.out.println(getClass().getCanonicalName()+"\t"+"UNKNOWN CRASH: current\t"+cur+
					"\t performStep\t"+stp+"\t Size:"+stepSize+" \t Time:"+timeStep);
			e.printStackTrace();

		}
					
		if(RandomWalkSimulator.intenseDebug>1) {
			if(env!=null)
				if(!env.contains(currentPosition)) {
					System.out.println(getClass().getCanonicalName()+"\t"+"Whoops:");
					System.out.println(getClass().getCanonicalName()+"\t"+"\t"+cur+"\t"+stp+"\t"+currentPosition);        	
				}
			if(RandomWalkSimulator.intenseDebug>8) {
				System.out.println(getClass().getCanonicalName()+"\t"+"PT.performStep: pos="+currentPosition+" vel="+stepSize);
			}
		}

	}

	// perform the mechanics of checking collisions
	private void performStepLocal(SimEnvironment globalEnv)   {
		if(currentPosition==null)
			System.out.println(getClass().getCanonicalName()+"\t"+"!");
		if(RandomWalkSimulator.intenseDebug>7)
			System.out.println(getClass().getCanonicalName()+"\t"+"\t\t\t\tstepLocal: "+currentPosition+"\t"+step+"\t"+this.stepSize);
		recursionDepth++;		
		if(recursionDepth>1000) {
			System.out.println(getClass().getCanonicalName()+"\t"+"Recusion Depth Reached. Unwinding Step History");
			System.out.println(getClass().getCanonicalName()+"\t"+"\t"+currentPosition+"Current:\t"+step);
			throw new RuntimeException("Maximum Recusion Depth Reached! Lots of bounces here!");
		}
		PT localCurrentPosition = currentPosition;
		PT localStep = step;
		try {

			PT nextPosition = currentPosition.plus(step);

			// We are in extra-compartmental space
			Iterator envs = globalEnv.getPossibleCollisionIterator(currentPosition,nextPosition,localEnv);
			double dist = Double.MAX_VALUE;
			IntersectResult firstHit = null;
			SimCompartment collisionEnv = null;
			while(envs.hasNext()) {
				SimCompartment env = (SimCompartment)envs.next();
				IntersectResult res = env.findFirstIntersection(currentPosition,nextPosition);
				if(res!=null) {
					res = env.findFirstIntersection(currentPosition,nextPosition);										
					if(res.fractionalDistance<dist) {
						collisionEnv  = env;
						firstHit=res;
						dist = res.fractionalDistance;
					}
				}
			}

			IntersectResult latticeWrap =globalEnv.checkLatticeWrapAround(firstHit, currentPosition,nextPosition);
			boolean forceWrapping = false;
			if(firstHit!=null) {
				if(latticeWrap!=null) {
					if(latticeWrap.fractionalDistance<firstHit.fractionalDistance) {
						// we hit the lattice before a compartment
						forceWrapping = true;
					}
				}
			}
			if(firstHit!=null && !forceWrapping) {
				//we hit a compartment boundary
				performCollision(firstHit,collisionEnv,globalEnv);
				if(RandomWalkSimulator.intenseDebug>7)
					System.out.println(getClass().getCanonicalName()+"\t"+"\t\t\t\t Collision");
			} else {

				////use collision compartment information - check to see if we should wrap around\
				if(latticeWrap!=null){				
					performLatticeWrap(latticeWrap,globalEnv,nextPosition);
				} else {
					//we did not hit a lattice boundary
					if(globalEnv.checkLatticeWrapAround(nextPosition))
						currentPosition = nextPosition;
					else {
						System.out.println(getClass().getCanonicalName()+"\t"+"Out of bounds next position! "+nextPosition);
						System.out.println(getClass().getCanonicalName()+"\t"+"Current position:"+currentPosition);
						System.out.println(getClass().getCanonicalName()+"\t"+"Step:"+step);
						throw new RuntimeException("out of bounds");
					}
					logT2Attenuation+=-timeStepLeft/getT2(globalEnv);
					timeStepLeft=0;
					if(RandomWalkSimulator.intenseDebug>7)
						System.out.println(getClass().getCanonicalName()+"\t"+"\t\t\t\t N/C");
				}
			} 
		} catch (RuntimeException e) {
			System.out.println(getClass().getCanonicalName()+"\t"+"UNKNOWN CRASH: \t"+localCurrentPosition+"\t"+localStep);			
			throw e;
		}
	}

	// The point crossed the lattice boundary and must be wrapped
	private void performLatticeWrap(IntersectResult latticeWrap, SimEnvironment globalEnv, PT next) {
		if(RandomWalkSimulator.intenseDebug>8) {
			System.out.println(getClass().getCanonicalName()+"\t"+"PT.performLatticeWrap: old="+currentPosition+" new="+latticeWrap.intersectionPoint);
		}		
		if(globalEnv.checkLatticeWrapAround(latticeWrap.intersectionPoint))
			currentPosition = latticeWrap.intersectionPoint;
		else {
			System.out.println(getClass().getCanonicalName()+"\t"+"Bad lattice wrap "+latticeWrap.intersectionPoint);
			System.out.println(getClass().getCanonicalName()+"\t"+"Current position:"+currentPosition);
			System.out.println(getClass().getCanonicalName()+"\t"+"Step:"+step);
			System.out.println(getClass().getCanonicalName()+"\t"+"Next(before wrap):"+next);
			throw new RuntimeException("out of bounds");
		}
		step = step.times(1-latticeWrap.fractionalDistance);	
		logT2Attenuation+=-(timeStepLeft*((latticeWrap.fractionalDistance)))/getT2(globalEnv);
		timeStepLeft = timeStepLeft*((1-latticeWrap.fractionalDistance));		
		originalPosition = originalPosition.plus(latticeWrap.intersectionNormal);
		if(latticeWrap.fractionalDistance<1)
			performStepLocal(globalEnv);
	}

	private static synchronized void recordCollisions() {
		collisions++;
	}
	public static synchronized int getCollisions() {
		return collisions;
	}
	public static synchronized void resetCollisions() {
		collisions=0;
	}
	// The point collided with a boundary and must be either reflected or transmitted
	private void performCollision(IntersectResult inter,SimCompartment collsionCompartment, SimEnvironment globalEnv) {
		recordCollisions();


		if(inter.fractionalDistance<=0 || inter.fractionalDistance>1)
			System.out.println(getClass().getCanonicalName()+"\t"+"Bad distance:"+inter.fractionalDistance);
		SimCompartment lastLocalEnv = localEnv;
		if(RandomWalkSimulator.intenseDebug>8) {
			System.out.println(getClass().getCanonicalName()+"\t"+"PT.performCollision: old="+currentPosition+" new="+inter.intersectionPoint.plus(step.times(EPS)));
		}

		// We encountered a collision
		SimCompartment newEnv=globalEnv.getLocalEnvironment(inter.epsilonForward,localEnv); // will not work for nested environments 
		double transmissionProb = globalEnv.calcTransmissionProb(localEnv,newEnv);
//		System.out.println(getClass().getCanonicalName()+"\t"+"transmissionProb: "+transmissionProb);
		if(Math.random()<transmissionProb) {
			// selected transmission
			if(globalEnv.checkLatticeWrapAround( inter.epsilonForward))
				currentPosition = inter.epsilonForward;  
			else {
				performLatticeWrapNewPosition(globalEnv,inter.epsilonForward);				
			}
			step = step.times(1-inter.fractionalDistance);
			logT2Attenuation+=-(timeStepLeft*((inter.fractionalDistance)))/getT2(globalEnv);
			timeStepLeft = timeStepLeft*((1-inter.fractionalDistance));			
			globalEnv.updateSimPointEnv(this,newEnv);
			if(this.localEnv==lastLocalEnv){
				System.out.println(getClass().getCanonicalName()+"\t"+"Yipes! Env. did not change with transmission!");
			}

		} else {
			if(globalEnv.checkLatticeWrapAround( inter.epsilonBackward))
				currentPosition = inter.epsilonBackward;
			else {
				performLatticeWrapNewPosition(globalEnv,inter.epsilonBackward);			
			}


			step = step.times(1-inter.fractionalDistance);
			logT2Attenuation+=-(timeStepLeft*((inter.fractionalDistance)))/getT2(globalEnv);
			timeStepLeft = timeStepLeft*((1-inter.fractionalDistance));
			// specular reflection            
			reflect(inter);

			globalEnv.updateSimPointEnv(this,localEnv); //debug only
			if(this.localEnv!=lastLocalEnv){
				System.out.println(getClass().getCanonicalName()+"\t"+"Yipes! Env. changed with reflection!");
			}
		}
		if(inter.fractionalDistance<0.999)  // fuzz factor
			if(step.norm()>0.001) { // ignore fractional steps < 0.01 microns
				performStepLocal(globalEnv); 
			}
			else {
				//				System.out.println(getClass().getCanonicalName()+"\t"+"Terminating round off error.");
				logT2Attenuation+=-(timeStepLeft)/getT2(globalEnv);
				timeStepLeft = 0;
			}
		else {
			//				System.out.println(getClass().getCanonicalName()+"\t"+"Terminating round off error.");
			logT2Attenuation+=-(timeStepLeft)/getT2(globalEnv);
			timeStepLeft = 0;
		}
	}

	private void performLatticeWrapNewPosition(SimEnvironment globalEnv,
			PT nextPT) {
		IntersectResult ir = globalEnv.checkLatticeWrapAround(null, currentPosition, nextPT);
		if(ir==null) 
			currentPosition=nextPT;
		else {
			currentPosition=ir.intersectionPoint;
		}				
	}

	// The point hit a boundary and must be reflected
	private void reflect(IntersectResult hit) {
		if(RandomWalkSimulator.intenseDebug>8) {
			System.out.println(getClass().getCanonicalName()+"\t"+"PT.reflect: oldStep="+step+" newStep="+step.minus(hit.intersectionNormal.times(2*step.dot(hit.intersectionNormal))));
		}
		// V_exit = V_enter - 2 * normal * (normal dot V_enter)
		//   System.out.println(getClass().getCanonicalName()+"\t"+step+" "+hit.intersectionNormal);
		//		System.out.println(getClass().getCanonicalName()+"\t"+"MyNorm: "+hit.intersectionNormal.norm());
		step = step.minus(hit.intersectionNormal.times(2*step.dot(hit.intersectionNormal)));
		//		System.out.println(getClass().getCanonicalName()+"\t"+step);
	}

	// Select a random direction (uniformly over the surface of a sphere)
	private void setRandDirection() {
		//choose uniform random step directions
		double x=0,y=0,z=0;
		double dist = 0;
		while(dist==0) {  // this is necessary because there is a SMALL chance that the random direction could be all zeros
			x = (Math.random()-.5);
			y = (Math.random()-.5);
			z = (Math.random()-.5);
			dist = Math.sqrt(x*x+y*y+z*z);
		}
		//System.out.println(getClass().getCanonicalName()+"\t"+dist);
		step = new PT(stepSize*x/dist,stepSize*y/dist,stepSize*z/dist);
		if(RandomWalkSimulator.intenseDebug>8) {
			System.out.println(getClass().getCanonicalName()+"\t"+"PT.setRandDirection: vel="+step);
		}
	}

	// Return the current position of the point
	public PT getPoint() {
		return currentPosition;
	}
	
	public double getPhase(int j) {
		if(phases==null) 
			return 0;
		if(j>=phases.length)
			return 0;
		return phases[j];		
	}
	
	public void updatePhases(SimFiniteGradient []finiteGradients, double timeStep) {
		if(phases==null) {
			phases = new double[finiteGradients.length];
			for(int i=0;i<finiteGradients.length;i++) {
				phases[i]=0;
			}
		}
		for(int i=0;i<phases.length;i++) {
			phases[i]+=finiteGradients[i].getPhaseAccumulation(currentPosition.minus(originalPosition),timeStep);
		}
	}
	
}
