package bl.diffusion;

/**
 * Created by IntelliJ IDEA.
 * User: bennett
 * Date: Nov 20, 2005
 * Time: 9:28:30 AM
 * To change this template use Options | File Templates.
 * ************************************
 * Magnetic Resonance in Medicine Final Project
 * Released: December 1, 2005
 *
 * class Triangle
 *      Represent a triangle
 *
 * Copyright (C) 2005 Bennett Landman, bennett@bme.jhu.edu
 */
public class Triangle {
    // constant used for avoiding numerical accuracy
    static public double EPS=1e-4f;
    // set of points
    public PT pts[];
    // precompute the normal to triangle
    private PT planeNormal;
    // represent the distance from the origin to the plane of the triangle along the normal
    private double planeOffset;

    // create a new triangle
    public Triangle(PT a, PT b, PT c)  {
        pts = new PT[3];
        pts[0] = a;
        pts[1] = b;
        pts[2] = c;
        //Find plane normal
        planeNormal = (b.minus(a)).cross(c.minus(a));
        planeNormal = planeNormal.times(1f/(double)planeNormal.length());
        planeOffset = a.dot(planeNormal);
    }


    // detect an intersection between the triangle and the segment a->b
    public int intersect(PT a,PT b) throws DegenerateIntersectionException {
        int vol0, vol1, vol2;

        vol0 = volumeSign(a,pts[0],pts[1],b);
        vol1 = volumeSign(a,pts[1],pts[2],b);
        vol2 = volumeSign(a,pts[2],pts[0],b);

        /* same sign: segment intersects interior of triangle */
        if(((vol0>0)&&(vol1>0)&&(vol2>0))||((vol0<0)&&(vol1<0)&&(vol2<0)))
            return (vol0);

        /* oposite sign: segment does not intersect the interior */
        if(((vol0>0) || (vol1>0) || (vol2>0)) && ((vol0<0) || (vol1<0) || (vol2<0)))
            return 0;
        else if((vol0==0) && (vol1==0) && (vol2==0)) {
            // coplanar
            double dist0 = SegSegIntersect(a,b,pts[0],pts[1]);
            double dist1 = SegSegIntersect(a,b,pts[1],pts[2]);
            double dist2 = SegSegIntersect(a,b,pts[0],pts[2]);
            PT r = b.minus(a);
            if((dist0<=dist1)&&(dist0<=dist2) && (dist0<=1)) {
                PT pt = a.plus(r.times(dist0));
                if(pt.equals(pts[0]) || pt.equals(pts[1]))
                    throw new DegenerateIntersectionException("vertex intersection");
                if(dist0==0 || dist0==1)
                    throw new DegenerateIntersectionException("edge intersection");
                return 1;
            }
            if(dist1<=dist2 && dist1<=1) {
                PT pt = a.plus(r.times(dist1));
                if(pt.equals(pts[1]) || pt.equals(pts[1]))
                    throw new DegenerateIntersectionException("vertex intersection");
                if(dist0==0 || dist0==1)
                    throw new DegenerateIntersectionException("edge intersection");
                return 1;
            }
            if(dist2<=1) {
                PT pt = a.plus(r.times(dist2));
                if(pt.equals(pts[0]) || pt.equals(pts[2]))
                    throw new DegenerateIntersectionException("vertex intersection");
                if(dist0==0 || dist0==1)
                    throw new DegenerateIntersectionException("edge intersection");
                return 1;
            }
            throw new DegenerateIntersectionException("Error in Triangle.Intersect: Invalid result.");
        }

        /* two zeros: segment intersects vertex */
        if(  ((vol0==0) && (vol1==0)) || ((vol0==0) && (vol2==0)) || ((vol1==0) && (vol2==0)))
            throw new DegenerateIntersectionException("vertex intersection");

        /* one zero: segment intersects edge */
        if(  ((vol0==0) ) || ((vol1==0)) || ( (vol2==0)))
            throw new DegenerateIntersectionException("vertex intersection");

        throw new DegenerateIntersectionException("Error in Triangle.Intersect: Case failure.");
    }

    // find an intersection between the triangle and the segment a->b
    public IntersectResult findIntersect(PT a,PT b) throws DegenerateIntersectionException {
        PT r =b.minus(a);
        PT unitStep = b.minus(a);
		PT normUnitStep = unitStep.normalize();

        double denom = r.dot(planeNormal);
        if(denom==0) {
            //segment in plane of triangle
            if(this.contains(a))
                return new IntersectResult(a,0,planeNormal,a.plus(normUnitStep.times(IntersectResult.EPSILON)),
                		a.minus(normUnitStep.times(IntersectResult.EPSILON)));
            double dist0 = SegSegIntersect(a,b,pts[0],pts[1]);
            double dist1 = SegSegIntersect(a,b,pts[1],pts[2]);
            double dist2 = SegSegIntersect(a,b,pts[0],pts[2]);
            if((dist0<dist1)&&(dist0<dist2)) {
            	PT intersect = a.plus(r.times(dist0));
                return new IntersectResult(intersect,dist0,planeNormal,intersect.plus(normUnitStep.times(IntersectResult.EPSILON)),
                		intersect.minus(normUnitStep.times(IntersectResult.EPSILON)));
            }
            if(dist1<dist2) {
            	PT intersect = a.plus(r.times(dist1));
                return new IntersectResult(intersect,dist1,planeNormal,intersect.plus(normUnitStep.times(IntersectResult.EPSILON)),
                		intersect.minus(normUnitStep.times(IntersectResult.EPSILON)));
            }
            PT intersect = a.plus(r.times(dist2));
            return new IntersectResult(intersect,dist2,planeNormal,intersect.plus(normUnitStep.times(IntersectResult.EPSILON)),
            		intersect.minus(normUnitStep.times(IntersectResult.EPSILON)));
        }
        double t = (planeOffset-a.dot(planeNormal))/denom;
        if((t<0)||(t>1))
            throw new DegenerateIntersectionException("Error in findIntersect: No Intersection");

        PT planeIntersect =a.plus(r.times(t));
        if(planeNormal.z!=0) {
            planeIntersect.z = (planeOffset-planeIntersect.x*planeNormal.x
                    -planeNormal.y*planeIntersect.y)/planeNormal.z;
        } else {
            if(planeNormal.y!=0) {
                planeIntersect.y = (planeOffset-planeIntersect.x*planeNormal.x
                        -planeNormal.z*planeIntersect.z)/planeNormal.y;
            } else {
                planeIntersect.x = (planeOffset-planeIntersect.y*planeNormal.y
                        -planeNormal.z*planeIntersect.z)/planeNormal.x;
            }
        }

        if(contains(planeIntersect)) {
        	PT intersect = planeIntersect;
            return new IntersectResult(planeIntersect,t,planeNormal,intersect.plus(normUnitStep.times(IntersectResult.EPSILON)),
            		intersect.minus(normUnitStep.times(IntersectResult.EPSILON)));
        }
        else
            throw new DegenerateIntersectionException("No in plane intersection");
    }

    // determine the sign of the volume of the polyhedron a,b,c,d
    public int volumeSign(PT a, PT b, PT c, PT d) {
        double ax,ay,az,bx,by,bz,cx,cy,cz;
        ax = a.x-d.x;
        ay = a.y-d.y;
        az = a.z-d.z;
        bx = b.x-d.x;
        by = b.y-d.y;
        bz = b.z-d.z;
        cx = c.x-d.x;
        cy = c.y-d.y;
        cz = c.z-d.z;
        double vol = ax * (by*cz - bz*cy) +
                ay * (bz*cx - bx*cz) +
                az * (bx*cy - by*cx);
        if(vol>0)
            return 1;
        if(vol<0)
            return -1;
        return 0;
    }

/*
Calculate the line segment PaPb that is the shortest route between
two lines P1P2 and P3P4. Calculate also the values of mua and mub where
Pa = P1 + mua (P2 - P1)
Pb = P3 + mub (P4 - P3)
Returns fractional distance from p1 to p2 where it intersection p3 to p4
*/
    double SegSegIntersect(PT p1, PT p2, PT p3, PT p4)
    {
        PT p13,p43,p21;
        double d1343,d4321,d1321,d4343,d2121;
        double numer,denom;
        p13 = p1.minus(p3);
        p43 = p4.minus(p3);

        if (Math.abs(p43.x)<EPS && Math.abs(p43.y)<EPS &&  Math.abs(p43.z)<EPS)
            return Double.MAX_VALUE;
        p21 = p2.minus(p1);

        if (Math.abs(p21.x)<EPS && Math.abs(p21.y)<EPS &&  Math.abs(p21.z)<EPS)
            return Double.MAX_VALUE;

        d1343 = p13.x * p43.x + p13.y * p43.y + p13.z * p43.z;
        d4321 = p43.x * p21.x + p43.y * p21.y + p43.z * p21.z;
        d1321 = p13.x * p21.x + p13.y * p21.y + p13.z * p21.z;
        d4343 = p43.x * p43.x + p43.y * p43.y + p43.z * p43.z;
        d2121 = p21.x * p21.x + p21.y * p21.y + p21.z * p21.z;

        denom = d2121 * d4343 - d4321 * d4321;
        if (Math.abs(denom) < EPS)
            return Double.MAX_VALUE;
        numer = d1343 * d4321 - d1321 * d4343;

        double mua = numer / denom;
        double mub = (d1343 + d4321 * mua) / d4343;

        if(mub>=0 && mub<=1 && mua>=0 && mua<=1)
            return (double)mua;

        return Double.MAX_VALUE;

    }

    // determine if a and b are on the same side of p1->p2
    private boolean sameside(PT p1, PT p2, PT a, PT b) {
        PT ba = b.minus(a);
        PT cp1 = ba.cross(p1.minus(a));
        PT cp2 = ba.cross(p2.minus(a));
        return (cp1.dot(cp2)>=0);
    }

    // determine if this triangle contains a point
    public boolean contains(PT p) {
        return ( sameside(p,pts[0],pts[1],pts[2]) &&
                sameside(p,pts[1],pts[2],pts[0]) &&
                sameside(p,pts[2],pts[0],pts[1]));
    }
}
