function [MotionResults] = RWSsimulate(javaSimulator,timeStepUS,diffusionTimeUS)
% function RWSsimulate(javaSimulator,timeStepUS,diffusionTimeUS)
%
%   Run the simulator for a total of diffusionTimeUS in steps of
%   timeStepUS us. Note that diffusionTimeUS must be divisible
%   by timeStepUS. It is permissible to run the same simulator repeatedly.
%
%   Inputs:
%       javaSimulator - RWS simulator object
%       timeStepUS  - timestep in us
%       diffusionTimeUS - total diffusion time in us
%
%   Output:
%       MotionResults.parameters.diffusionTime = diffusion time in us
%       MotionResults.parameters.timeStep = time step in us
%       MotionResults.elapseTime = computation time in s
%       MotionResults.brownianMotion = result-orig (position in um as Nx3
%          matrix)
%       MotionResults.origPositions = original positions in um as Nx3. Note
%          these are not contrained by the lattice dimensions
%       MotionResults.resultPositions = result positions in um as Nx3.
%           Note these are contrained to lie within the lattice. 
%       MotionResults.parameters.numSpinPackets = N - number of particle
%          tracked 
%       MotionResults.T2SignalAttenuation = vector of T2 signal attenuations
%       MotionResults.fgPhaseReults = vector of phase results (M per
%       particle) where M gradient directions were specified with
%       RWSsetFiniteGradients.
%
% ***********************************************
% RandomWalkSimulator v2.1
% Major Revision by:
% 	Bennett Landman, landman@jhu.edu
% 	Jonathan Farrell, jfarrell@jhu.edu
%   (C) Copyright 2008, Bennett Landman and Jonathan Farrell
%   Released under Lessor GNU Public License v1.0
% 	Not for clinical use.
% 	No warranty expressed or implied.
%   Use at your own risk.
% Download at: http://www.nitrc.org/projects/randomwalks/
% ***********************************************
% 5/11/09 bl - added support for finite gradients
% 9/10/08 bl
import bl.diffusion.*;
if(abs(mod(diffusionTimeUS,timeStepUS))>1e6)
    error('Invalid timeStep');
end
javaSimulator.setSimTime(timeStepUS,diffusionTimeUS);

tic;
javaSimulator.takeAWalk();
stats= javaSimulator.getCollisionStats();
stats=single(stats);
disp(sprintf('Stats: %d collisions in %d steps (%.2f%%)',stats(1),stats(2),stats(1)/stats(2)*100))
time = toc;
disp(['Simulation Complete in ' num2str(time) sprintf(' seconds (%.1fk steps/second).',stats(2)/time/1000)]);
disp(['Step length: ' num2str(timeStepUS) ' us ~= ' num2str(sqrt(6*timeStepUS*3e-3)) ' um (free H20)']);

% Step 7. Save output
resultPositions = javaSimulator.reportResultPositions();
origPositions = javaSimulator.reportOriginalPositions();
MotionResults = [];
MotionResults.parameters.diffusionTime = diffusionTimeUS; % us
MotionResults.parameters.timeStep = timeStepUS; %us
MotionResults.elapseTime = time;
MotionResults.brownianMotion = resultPositions-origPositions;
MotionResults.origPositions = origPositions;
MotionResults.resultPositions = resultPositions;
MotionResults.parameters.numSpinPackets = size(resultPositions,1);
MotionResults.T2SignalAttenuation  = javaSimulator.reportT2SignalAttenuation();
MotionResults.LogT2SignalAttenuation  = javaSimulator.reportLogT2SignalAttenuation();
MotionResults.stats = stats;
MotionResults.fgPhaseReults = javaSimulator.reportFinitePulsePhase();