function TRI = delaunaySphere(x,y,z)
% function TRI = delaunaySphere(x,y,z)
% 
% Use projective geometry to calculate the 
% delaunay triangulation of a set of points
% lying on the surface of a unit sphere. 
%
% (C)opyright 2005, Bennett Landman, bennett@bme.jhu.edu
% Revision History:
% Created: 2/11/2005

R = epsunique(10,sqrt(x.^2+y.^2+z.^2));
if(length(epsunique(10,[1 R]))>1)
    warning('Points do not lie on the surface of a unit sphere within working precision.');
    [X] = forceToSphere([x(:) y(:) z(:)]);
    x = X(:,1)';
    y = X(:,2)';
    z = X(:,3)';
end

% First, we need to project the sphere onto two
% distint planes. Choose [1 1 1] and [-1 -1 -1] 
% as the projection points. If these are members
% of the point on the sphere, we'll shift them 
% around so that they are not on a sphere.
pt = [1 1 1]; pt = pt/sqrt(sum(pt.*pt));
while((sum(sum(repmat(pt,[length(x) 1])==[x; y; z]',2)==3)))
    pt = forceToSphere(pt+rand([1 1 1])/100,1);
end
    

% Use projective geometry to map the sphere onto
% a plane
[a,b] = projectSphere([x' y' z'],pt);

% Compute the delaunay triangulation on the plane - 
% This will be valid except for the collection of points on 
% the opposite side of the sphere where the mapping goes to infinity.
q = delaunay(a,b);

% We select another point on the opposite side of the sphere, subject
% to the condition that it is not one of the points that we
% are examining.
pt = [-1 -1 -1]; pt = pt/sqrt(sum(pt.*pt));
while((sum(sum(repmat(pt,[length(x) 1])==[x; y; z]',2)==3)))
    pt = forceToSphere(pt+rand([1 1 1])/100,1);
end

% Compute the delaunay triangulation on the plane - 
% This will allow us to fillin the region that was missing from the other 
% projection
[a,b] =projectSphere([x' y' z'],pt);

% Compute the delaunay triangulation on the plane - 
% This will be valid except for the collection of points on 
% the opposite side of the sphere where the mapping goes to infinity.
q2 = delaunay(a,b);

% % DEBUG: TO visualize the delaunay triangulations, use the following
% % code snipit. 
% % figure;
% % q2 = delaunay(a,b);
% % for j=1:length(q2)    
% %     pts = [q2(j,:) q2(j,1)];
% %     fillSphTri(10,x(pts),y(pts),z(pts),[1 1 1]);
% %     hold on;
% % end


% We now need to identify all unique triangles in this set. 
% To facilitate indexing, we sort the triangles and then map 
% each to a key number. We find unique key numbers and reverse
% the mapping. 
for j=1:length(q)
    q(j,:) = sort(q(j,:));
end
for j=1:length(q2)
    q2(j,:) = sort(q2(j,:));
end

F = length(x)+1;
tri = sum(repmat([F*F F 1],[length(q) 1]).*q,2);
tri2 = sum(repmat([F*F F 1],[length(q2) 1]).*q2,2);
allTri=[tri' tri2'];
allTri = unique(allTri);
TRI = [floor(allTri/F/F); floor(mod(allTri,F*F)/F); mod(allTri, F)]';
