package bl.diffusion;

import java.util.*;

/**
 * Created by IntelliJ IDEA.
 * User: bennett
 * Date: Nov 20, 2005
 * Time: 9:56:51 AM
 * To change this template use Options | File Templates.
 * ************************************
 * Magnetic Resonance in Medicine Final Project
 * Released: December 1, 2005
 *
 * class SimRandomWalk
 *      Control the mechanics of simulated a random walk.
 *
 * Copyright (C) 2005 Bennett Landman, bennett@bme.jhu.edu
 */
public class SimRandomWalk implements SimEnvironment {
    private Vector pts; // the list of points of the environment
    private Vector compartments; // the list of compartments
    private double latticeDimensions[]; // the size of the repeating element
    private double extraComparmentDiffusionCoefficient; //the diffusion external to all compartment
    private double timeStep; // the size of the time step [us]
    private double T2; // T2 of the extracelluar material in us
    private HashMap insideMap ; // a map of outside->inside mapping
    private HashMap outsideMap ; // a map of inside->outside mapping
    private static double EPS = 1e-4; // a constant used for dealing with numerical accuracy
    
    
    // Create a new walk environment
    public SimRandomWalk() {
        insideMap = new HashMap();
        outsideMap = new HashMap();
        pts = new Vector();
        compartments = new Vector();
        latticeDimensions = null;
    }

    // set the time step [us]
    public void setTimeStep(double timeStep) {
        this.timeStep = timeStep;
    }

    // set the global environment parameters
    public void setExtraComparmentParameters(double diffusionCoefficient, double lattice[], double T2) {
        // [diffusionCoefficient] = um2/us  [lattice]=um
        extraComparmentDiffusionCoefficient = diffusionCoefficient;
        latticeDimensions = lattice;
        this.T2=T2;
    }

    // initialize each point to its local environment
    public void initializeDiffusion() {
        Iterator i = pts.iterator();
        while(i.hasNext()) {
            SimPoint p = (SimPoint)i.next();
            SimCompartment env = getLocalEnvironment(p.getPoint(),null);
            updateSimPointEnv(p,env);
        }
    }

    // perform one time step
    public void performTimestep(SimFiniteGradient []finiteGradients) {
        Iterator i = pts.iterator();
        
        while(i.hasNext()) {
            SimPoint p = (SimPoint)i.next();
            PT first = p.getPoint();
            p.performStep(this, finiteGradients);
//            PT dist = p.getPoint().minus(first);
//            System.out.println(getClass().getCanonicalName()+"\t"+dist.length() + "\t"+ dist.toString());
           // System.out.println(getClass().getCanonicalName()+"\t"+"Step: "+p.originalPosition+" - "+p.currentPosition);
        }
    }

    // get the local environment (or null) for any point
    public SimCompartment getLocalEnvironment(PT p, SimCompartment lastEnv) {
        if(false && lastEnv !=null && outsideMap.containsKey(lastEnv)) { //DEBUG
            Vector map = (Vector)outsideMap.get(lastEnv);
            Iterator i= map.iterator();
            while(i.hasNext())  {
            	SimCompartment c = (SimCompartment)i.next();
                if(c.contains(p))
                    return c;
            }
            return null;
        } else {
            Iterator i=getPossibleCollisionIterator(p,p,null);
            while(i.hasNext()) {
            	SimCompartment c = (SimCompartment)i.next();
                if(c.contains(p))
                    return c;
            }
            return null;
        }
    }

    // get a list of all possible compartments that could hit the segment between currentPosition to nextPosition
    public Iterator getPossibleCollisionIterator(PT currentPosition, PT nextPosition, SimCompartment curEnv) {
        //return compartments.iterator();
        //add inside/outside to list.
        // have global Environment always return the list of possible intersections.
        // make this a function of local environment
        // if local is not null, then use it's inside information to limit search
        // otherwise, give all. also, always check cube BB for wrapping
        return new CompartmentIterator(compartments, curEnv,insideMap);
    }

    // calculate the probability of crossing between the two compartments
    public double calcTransmissionProb(SimCompartment oldEnv, SimCompartment newEnv) {
        // P = (1/4) vi * intersectionPoint(i->e) = (1/4) ve * intersectionPoint(e->i)
        double Dold,Dnew,vold,vnew;
        double probTrans = 0;
        if(newEnv == null) {
            if(oldEnv!=null) {
                Dold = oldEnv.getDiffusionCoefficient();
                vold = SimPoint.calcVelocity(Dold)/Math.sqrt(timeStep);
                probTrans =  4*oldEnv.getMembranePermeability()/vold;
                if(probTrans > 1)
                	throw new RuntimeException("Invalid membrane permeability leading to p_trans>1:\t"+oldEnv.getMembranePermeability()+" us/um\tv:"+vold+" um/us");
                else 
                	return probTrans;
            } else
                return 0;
                //throw new RuntimeException("Cannot have bordering external compartments.");
        } else {
            if(oldEnv==null) {
                Dnew = newEnv.getDiffusionCoefficient();
                vnew = SimPoint.calcVelocity(Dnew)/Math.sqrt(timeStep);
                probTrans= 4*newEnv.getMembranePermeability()/vnew;
                if(probTrans > 1)
                	throw new RuntimeException("Invalid membrane permeability leading to p_trans>1:\t"+newEnv.getMembranePermeability()+" us/um\tv:"+vnew+" um/us");
                else 
                	return probTrans;
            } else {
            	Dnew = newEnv.getDiffusionCoefficient();
            	vnew = SimPoint.calcVelocity(Dnew)/Math.sqrt(timeStep);
            	double Pnew = 4*newEnv.getMembranePermeability()/vnew;
            	Dold = oldEnv.getDiffusionCoefficient();
            	vold = SimPoint.calcVelocity(Dold)/Math.sqrt(timeStep);
            	double Pold = 4*oldEnv.getMembranePermeability()/vold;
            	if(Math.abs(Pnew-Pold)>EPS) {
            		System.err.println("Dnew:"+Dnew);
            		System.err.println("SimPoint.calcVelocity(Dnew)"+SimPoint.calcVelocity(Dnew));
            		System.err.println("vnew:"+vnew);
            		System.err.println("Pnew:"+Pnew);
            		System.err.println("newEnv.getMembranePermeability()"+newEnv.getMembranePermeability());
            		System.err.println("Dold:"+Dold);
            		System.err.println("SimPoint.calcVelocity(Dold)"+SimPoint.calcVelocity(Dold));
            		System.err.println("vold:"+vold);
            		System.err.println("Pold:"+Pold);
            		System.err.println("oldEnv.getMembranePermeability()"+oldEnv.getMembranePermeability());
            		System.err.flush();

            		throw new RuntimeException("Inconsistent Model definition. Tranmission probabilities not balanced. Difference="+Math.abs(Pnew-Pold)+" Tolerance set to: "+EPS);
            	} 
                if(Pnew > 1)
                	throw new RuntimeException("Invalid membrane permeability leading to p_trans>1:\t"+oldEnv.getMembranePermeability()+" us/um\tv:"+vold+" um/us");
                else 
                	return Pnew;
                
            }
        }
    }

    // force an update of a point's parameters
    public void updateSimPointEnv(SimPoint simPoint, SimCompartment newEnv) {
        if(newEnv!=null)
            simPoint.update(newEnv,newEnv.getDiffusionCoefficient(),timeStep);
        else
            simPoint.update(null,extraComparmentDiffusionCoefficient,timeStep);
    }

    // check to see if a step would cause a lattice wrap around
    public IntersectResult checkLatticeWrapAround(IntersectResult firstHit, PT curPT, PT nextPT) {
        if(nextPT.x<0 || nextPT.y<0 || nextPT.z<0 || nextPT.x>latticeDimensions[0] || nextPT.y>latticeDimensions[1]
                || nextPT.z>latticeDimensions[2]) {
            double dx, dy, dz;
            dx = Double.MAX_VALUE;
            dy=dx; dz=dx;
            
            if(nextPT.x<0) {
                dx = Math.abs(curPT.x/(curPT.x-nextPT.x));
            } else if(nextPT.x>latticeDimensions[0]) {
                dx = (double)Math.abs((latticeDimensions[0]-curPT.x)/(curPT.x-nextPT.x));
            }
            if(nextPT.y<0) {
                dy = Math.abs(curPT.y/(curPT.y-nextPT.y));
            } else if(nextPT.y>latticeDimensions[1]) {
                dy = (double)Math.abs((latticeDimensions[1]-curPT.y)/(curPT.y-nextPT.y));
            }
            if(nextPT.z<0) {
                dz = Math.abs(curPT.z/(curPT.z-nextPT.z));
            } else if(nextPT.z>latticeDimensions[2]) {
                dz = (double)Math.abs((latticeDimensions[2]-curPT.z)/(curPT.z-nextPT.z));
            }
            try {
            if(dx<dy && dx<dz) {
                if(nextPT.x<0) {
                    return new IntersectResult(
                            new PT(latticeDimensions[0]-SimPoint.EPS,curPT.y+dx*(nextPT.y-curPT.y),curPT.z+dx*(nextPT.z-curPT.z)),
                            dx,new PT(latticeDimensions[0],0,0),null,null); //are fwd/bck needed?
                } else {
                    return new IntersectResult(
                            new PT(0+SimPoint.EPS,curPT.y+dx*(nextPT.y-curPT.y),curPT.z+dx*(nextPT.z-curPT.z)),
                            dx,new PT(-latticeDimensions[0],0,0),null,null);//are fwd/bck needed?
                }
            } else {
                if(dy<dz) {
                    if(nextPT.y<0) {
                        return new IntersectResult(
                                new PT(curPT.x+dy*(nextPT.x-curPT.x),latticeDimensions[1]-SimPoint.EPS,curPT.z+dy*(nextPT.z-curPT.z)),
                                dy,new PT(0,latticeDimensions[1],0),null,null);//are fwd/bck needed?);
                    } else {
                        return new IntersectResult(
                                new PT(curPT.x+dy*(nextPT.x-curPT.x),0+SimPoint.EPS,curPT.z+dy*(nextPT.z-curPT.z)),
                                dy,new PT(0,-latticeDimensions[1],0),null,null);//are fwd/bck needed?
                    }
                } else {
                    if(nextPT.z<0) {
                        return new IntersectResult(
                                new PT(curPT.x+dz*(nextPT.x-curPT.x),curPT.y+dz*(nextPT.y-curPT.y),latticeDimensions[2]-SimPoint.EPS),
                                dz,new PT(0,0,latticeDimensions[2]),null,null);//are fwd/bck needed?
                    } else {
                        return new IntersectResult(
                                new PT(curPT.x+dz*(nextPT.x-curPT.x),curPT.y+dz*(nextPT.y-curPT.y),0+SimPoint.EPS),
                                dz,new PT(0,0,-latticeDimensions[2]),null,null);//are fwd/bck needed?
                    }
                }
            }
            } catch(RuntimeException e) {
            	System.out.println(getClass().getCanonicalName()+"\t"+"curPt:"+curPT);
            	System.out.println(getClass().getCanonicalName()+"\t"+"nextPT:"+nextPT);
            	System.out.println(getClass().getCanonicalName()+"\t"+dx);
            	System.out.println(getClass().getCanonicalName()+"\t"+dy);
            	System.out.println(getClass().getCanonicalName()+"\t"+dz);
            	throw e;
            }

        } else {
            return null;
        }

    }

    // add a new compartment to the environment
    public int addCompartment(SimCompartment env) {
        compartments.add(env);
        return compartments.lastIndexOf(env);
    }

    // add a point to the enviroment
    public void addPoint(SimPoint pt) {
        pts.add(pt);
    }

    // add a point to the environment, randomly
    public void addUniformRandomPoint() {
        pts.add(new SimPoint(Math.random()*latticeDimensions[0],
                Math.random()*latticeDimensions[1],
                Math.random()*latticeDimensions[2]));
    }
    
 // add a point to the environment, randomly inside
    public void addUniformRandomPointInside() {
    	for(int j=0;j<10000;j++) {
    		SimPoint pt = new SimPoint(Math.random()*latticeDimensions[0],
    				Math.random()*latticeDimensions[1],
    				Math.random()*latticeDimensions[2]);

    		Iterator i = compartments.iterator();
    		SimCompartment c  = null;    	
    		while(i.hasNext()) {
    			c=(SimCompartment)i.next();    		
    			if(c.contains(pt.getPoint())) {
    				pts.add(pt);    		
    				return;
    			}
    		}
    	}
    	throw new RuntimeException("Hey - I can't get a stupid point inside your geometry.");    	
    }    

    // add a point to the environment, randomly outside all compartments
    public void addUniformRandomPointOutside() {
    	for(int j=0;j<10000;j++) {
    		SimPoint pt = new SimPoint(Math.random()*latticeDimensions[0],
    				Math.random()*latticeDimensions[1],
    				Math.random()*latticeDimensions[2]);

    		Iterator i = compartments.iterator();
    		SimCompartment c  = null;
    		boolean inside = false;
    		while(i.hasNext()) {
    			c=(SimCompartment)i.next();    		
    			if(c.contains(pt.getPoint())) {
    				inside = true;    		    				
    			}
    		}
    		if(!inside) {
    			pts.add(pt);
    			return;
    		}
    	}
    	throw new RuntimeException("Hey - I can't get a stupid point outside your geometry.");    	
    }    
    // print information on all points
    public void debugPrintPoints() {
        Iterator i = pts.iterator();
        while(i.hasNext()) {
            SimPoint pt = (SimPoint)i.next();
            System.out.println(getClass().getCanonicalName()+"\t"+pt.toString());
        }
    }

    // get an iterator for all points
    public Iterator getPts() {
        return pts.iterator();
    }

    // set the inside/outside relationship between compartments
    public void setCompartmentOutsideInside(int outside, int inside) {
    	SimCompartment outC = (SimCompartment)compartments.get(outside);
    	SimCompartment inC = (SimCompartment)compartments.get(inside);

        Vector outC_in = null;
        if(insideMap.containsKey(outC)) {
            outC_in = (Vector)insideMap.get(outC);
        } else {
            outC_in = new Vector();
            insideMap.put(outC,outC_in);
        }
        outC_in.add(inC);

        Vector inC_out = null;
        if(outsideMap.containsKey(inC)) {
            inC_out = (Vector)outsideMap.get(inC);
        } else {
            inC_out = new Vector();
            outsideMap.put(inC,inC_out);
        }
        inC_out.add(outC);
    }

	public double getT2() { 
		return T2;
	}
	
	public boolean checkLatticeWrapAround(PT pt) {
		if(pt.x<0||pt.y<0||pt.z<0)
			return false;
		if(pt.x>latticeDimensions[0]||pt.y>latticeDimensions[1]||pt.z>latticeDimensions[2])
			return false;
		return true;
	}
}
