/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.neuro.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Date;

import org.mitre.neuro.containers.Session;
import org.mitre.neuro.containers.Study;

public class StoreSession {
    Study study;
    Connection connection;
    StoreSubject subjects;
    public StoreSession(Study study, Connection connection) {
        super();
        this.study = study;
        this.connection = connection;
        subjects = new StoreSubject(connection);
    }
    public StoreSession(Connection connection) {
        super();
        this.connection = connection;
    }


    public void store_session_event(Session session, Session.Image image)
    {
        String sql = "SELECT scan_id FROM SCAN_EVENT where subject_oid = ? and project_id =?";
        try {
            /* for case where scan_even DOES exist already  */
            PreparedStatement statement =  this.connection.prepareStatement(sql);
            statement.setInt(1,  image.subject_oid);
            // statement.setDate(2, new Date(System.currentTimeMillis()) );
            if( image.study_id == Integer.MIN_VALUE )
            {
                statement.setInt(2,  study.getStudy_id());  //for when doing spreadsheet upload
            }
            else
                statement.setInt(2, image.study_id);  //for processing from JHU files
            ResultSet rs = statement.executeQuery();
            if(rs.next()){
                //scan_event already exists
                //set the sesion_id as below and return
                image.session_id = rs.getString(1);
                session.setSession_id( image.session_id );
                System.out.println( "\t\tRetrieved Scan_id: " + image.session_id);
                rs.close();
                return;
            }
            rs.close();


                /*  for case where scan_event does not exist yet */
            sql = "INSERT INTO SCAN_EVENT (subject_oid, scan_date, project_id, scanner_id, location, software_version) VALUES (?,?,?,?,?,?)";
            statement =  this.connection.prepareStatement(sql);
            // statement.setInt(1,  session.getSession_id());
            statement.setInt(1,  image.subject_oid);
            statement.setDate(2, new Date(System.currentTimeMillis()) );
            if( image.study_id == Integer.MIN_VALUE )
            {
                statement.setInt(3,  study.getStudy_id());  //for when doing spreadsheet upload
            }
            else
                statement.setInt(3, image.study_id);  //for processing from JHU files
            statement.setString(4, session.getScanner());
            statement.setString(5, session.getLocation());
            statement.setString(6, session.getSoftware_version());

            statement.executeUpdate();

            //sql = "SELECT currval(pg_get_serial_sequence('session_event', 'session_id'));";
            sql = "select last_value from scan_event_scan_id_seq";
            // System.out.println( sql );
            statement = this.connection.prepareStatement(sql);
            ResultSet generated_keys = statement.executeQuery();
            if(generated_keys.next()){
                image.session_id = generated_keys.getString(1);
                session.setSession_id( image.session_id );
                System.out.println( "\t\tStored Scan_id: " + image.session_id);
            }
            generated_keys.close();
        } catch (SQLException e) {
            e.printStackTrace();//we want to continue here
            return;
        }



    }

    /**
     * Stores the pointer to the file in the images table - depends on the image id
     * @param image
     * @param file
     */
    public void store_scanfile(Session.Image image){
        String sql = "INSERT INTO IMAGES (volume_id, filename) VALUES (?,?)";
        try {
            PreparedStatement statement =  this.connection.prepareStatement(sql);

            statement.setInt(1, image.volume_id);
            statement.setString(2, image.path); // + '/' + file.getFilename());
            statement.executeUpdate();

        } catch (SQLException e) {
            e.printStackTrace();//we want to continue here
            return;
        }

    }

    /**
     * This is for storing processed data
     * @param session_id
     * @param image
     */
    public void store_image(Session.Image image) {
        /*
         *  Going to do this in two steps.  If the Volume does not exist, insert a bare bones one.  Then do an update
         *  that includes only what is defined.
         */
        String sql;
        if( image.volume_id == Integer.MIN_VALUE )
        {
            sql = "INSERT INTO VOLUME (scan_id) values (?)";
            try
            {
                PreparedStatement statement =  this.connection.prepareStatement(sql);
                statement.setInt(1,  Integer.parseInt(image.session_id));
                statement.executeUpdate();

                sql = "select last_value from volume_volume_id_seq";
                statement = this.connection.prepareStatement(sql);
                ResultSet generated_keys = statement.executeQuery();
                if(generated_keys.next()){
                    image.volume_id = generated_keys.getInt(1);
                    System.out.println( "\t\tStored Image: " + image.volume_id);
                }
                generated_keys.close();
            } catch (SQLException e) {
                e.printStackTrace();//we want to continue here
                return;
            }
        }
        updateVolume("modality", image.modality, image.volume_id);
        updateVolume("numerical_format", image.numerical_format, image.volume_id);
        updateVolume("xorientation",image.xorientation, image.volume_id);
        updateVolume("yorientation",image.yorientation, image.volume_id);
        updateVolume("zorientation",image.zorientation, image.volume_id);

        updateVolume("scan_id", image.session_id==null? null : Integer.valueOf(image.session_id), image.volume_id);

        updateVolume("xfov", image.xfov == Double.MIN_VALUE ? null : new Double(image.xfov), image.volume_id);
        updateVolume("yfov", image.yfov == Double.MIN_VALUE ? null : new Double(image.yfov), image.volume_id);
        updateVolume("zfov", image.zfov == Double.MIN_VALUE ? null : new Double(image.zfov), image.volume_id);

        updateVolume("xresolution", image.xresolution == Double.MIN_VALUE ? null : new Double(image.xresolution), image.volume_id);
        updateVolume("yresolution", image.yresolution == Double.MIN_VALUE ? null : new Double(image.yresolution), image.volume_id);
        updateVolume("zresolution", image.zresolution == Double.MIN_VALUE ? null : new Double(image.zresolution), image.volume_id);


        updateVolume("xdimension", image.xdimension == Double.MIN_VALUE ? null : new Double(image.xdimension), image.volume_id);
        updateVolume("ydimension", image.ydimension == Double.MIN_VALUE ? null : new Double(image.ydimension), image.volume_id);
        updateVolume("zdimension", image.zdimension == Double.MIN_VALUE ? null : new Double(image.zdimension), image.volume_id);

        updateVolume("protocol_name", image.protocol_name, image.volume_id);
        updateVolume("t1", image.t1 == Double.MIN_VALUE ? null : new Double(image.t1), image.volume_id);
        updateVolume("t2", image.t2 == Double.MIN_VALUE ? null : new Double(image.t2), image.volume_id);
        updateVolume("tr", image.tr == Double.MIN_VALUE ? null : new Double(image.tr), image.volume_id);
        updateVolume("ti", image.ti == Double.MIN_VALUE ? null : new Double(image.ti), image.volume_id);
        updateVolume("te", image.te == Double.MIN_VALUE ? null : new Double(image.te), image.volume_id);

        updateVolume("excitations", image.excitations == Double.MIN_VALUE ? null : new Double(image.excitations), image.volume_id);
        updateVolume("flip_angle", image.flip_angle == Double.MIN_VALUE ? null : new Double(image.flip_angle), image.volume_id);

        //TODO: where do the previews go?



    }

    protected void updateVolume( String field, Object value, int vol_id )
    {
        if( value == null )
        {
            return;
        }
        try
        {
            String sql = "UPDATE volume set " + field + "=? where volume_id=?";
            PreparedStatement statement =  this.connection.prepareStatement(sql);
            if( value instanceof String )
            {
                statement.setString( 1, (String)value );
                System.out.println( field + " is  String" );
            }
            else if( value instanceof Double)
            {
                System.out.println( field + " is  Double" );
                statement.setDouble( 1, ((Double)value).doubleValue() );
            }
            else if( value instanceof Integer )
            {
                statement.setInt(1, ((Integer)value).intValue());
                System.out.println( field + " is  Integer" );
            }
            statement.setInt(2, vol_id);
            statement.execute();
            System.out.println( "\t" + field  + " updated." );
            statement.close();
        } catch (SQLException e) {
            e.printStackTrace();//we want to continue here
            return;
        }

    }

    /**
     * This is for storing meta-data
     * @param scan
     * @throws Exception
     */
    public void store_scan(Session scan) throws Exception {
        subjects.store_subject(scan.getSubject(), study);
        for(Session.Image i : scan.getImages())
        {
            //the store_subject call above grabbed the subject_oid from the database...this wasn't available to the image processor.
            i.subject_oid = scan.getSubject().getObject_id();
            store_session_event( scan, i );
            store_image( i );
            store_scanfile( i );
        }
    }
}
