/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.neuro.excel;

import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * Classes which implement this interface will generate an xls
 * compatible document when the getSpreadSheet method is invoked.
 * This document is populated by the rows created by this class
 * You may manipulate the contents of the rows directly in order
 * to affect changes to the spreadsheet.
 * @author esax
 */
public abstract class ExcelGenerator {
	public Map<Integer, Worksheet> worksheets;

	public ExcelGenerator(){
		this.worksheets = new HashMap<Integer, Worksheet>();
	}

	/**
	 * The .xls is returned via reading the output stream.
	 *
	 * The onus is on the _CALLER_ to close() this stream
	 *
	 * Use:
	 * Content-type=application/vnd.ms-excel
	 *
	 * In general, this is the only abstract method
	 * which must be implemented by an extending class
	 */
	public abstract void getSpreadSheet(OutputStream stream) throws Exception;

	/**
	 * Use the title as an identifier to refer to this worksheet when adding
	 * rows
	 */
	public Worksheet addWorkSheet(String title, int cardinality){
		Worksheet worksheet = new Worksheet(title, cardinality);
		this.worksheets.put(cardinality, worksheet);
		return worksheet;
	}

	/**
	 * updating the reference to this row will update the memory representation for the row
	 * changes will be reflected in the output
	 * creates the row if need be, otherwise retrieves it from the specified sheet.
	 */
	public Row getRow(int worksheet_number, int number) {
		Worksheet sheet = this.worksheets.get(worksheet_number);
		Row row = sheet.getOrAddRow(number);
		return row;
	}

	/**
	 * Releases all references
	 */
	public void clear() {
		for(Worksheet sheet : this.worksheets.values()){
			sheet.clear();
		}
		this.worksheets.clear();
	}
}
