/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.neuro.excel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * A row is a container for spreadsheet data
 * @author esax
 */
public class Row {
	private HashMap<Integer, Datum> columns;
	private int index;
	/**
	 * Some book keeping
	 * the largest column index in this row
	 */
	private int max;
	/**
	 * This object is only generated by the excel generator implimentation
	 */
	public Row(int row_number) {
		this.columns = new HashMap<Integer, Datum>();
		this.index = row_number;
		max = -1;
	}
	/**
	 * add a datum of information at the specified column index
	 */
	public void addColumn(int c, Datum d){
		this.columns.put(new Integer(c), d);
		if(c > max){
			max = c;
		}
	}
	/**
	 * This function is similar to add_column except that
	 * it locks the field if the string is empty
	 * and indicates it needs to be filled otherwise
	 */
	public void addColumnEntry(int column, String entry){
		if(entry == null || entry.length() == 0){
			this.addColumn(column, Datum.empty());
		}
		else{
			this.addColumn(column, Datum.string(entry).lock());
		}
	}

	/**
	 * This function is similar to add_column except that
	 * it locks the field if the string is empty
	 * and indicates it needs to be filled otherwise
	 */
	public void addColumnEntry(int column, double entry){
		if(entry == -1 ){
			this.addColumn(column, Datum.empty());
		}
		else{
			this.addColumn(column, Datum.number(entry).lock());
		}
	}

	public int getMax() {
		return max;
	}
	/**
	 * get a list of the column index's used in this row
	 * @return -list of x's for this y
	 */
	public List<Integer> getColumns(){
		return new ArrayList<Integer>(this.columns.keySet());
	}
	/**
	 * Get a datum of information for this row at the specified column index.
	 * @param number
	 * @return Datum - the information for a cell
	 */
	public Datum getColumn(int number){
		return this.columns.get(new Integer(number));
	}
	public int getIndex() {
		return index;
	}
	public void setIndex(int index) {
		this.index = index;
	}
	/**
	 * releases all references to cells
	 * in this row
	 */
	public void clear(){
		this.columns.clear();
	}
}