/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.neuro.excel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

public class Worksheet {

	/**
	 * Title of this sheet
	 */
	String title;

	/**
	 * Mapping of rows organized by index
	 */
	HashMap<Integer, Row> rows;

	/**
	 * Which sheet is this?
	 */
	int cardinality;

	int max_row;

	int max_width;

	public Worksheet(String title, int cardinality){
		this.rows = new HashMap<Integer, Row>();
		this.title = title;
		max_row = 0;
		max_width = 0;
	}
	public void addRow(Row row){
		this.rows.put(row.getIndex(), row);
		if(row.getIndex() > this.max_row){
			this.max_row = row.getIndex();
		}
		if(row.getMax() > this.max_width){
			this.max_width = row.getMax();
		}
	}

	/**
	 * this is if the meta is on the leftmost column
	 * and the data is organized in vertical rows.<br>
	 * tuples numbered left to right
	 * @param specify the column number which contains the meta-names for the
	 * tuples.
	 * @return a list of tuples
	 */
	public List<HashMap<String, Datum>> extractVerticalTuples(int meta_column, int start_row){
		List<HashMap<String, Datum>> tuples = new ArrayList<HashMap<String, Datum>>();

		//obtain taxonomy
		String [] meta = new String[this.max_row+1];
		for( int i = start_row; i <= this.max_row; i++ ){
			Datum d = this.getCell(i, meta_column);
			if(d != null){
				meta[i] = d.getString_dat();
			}
		}

		//populate the tuples
		for (int column_index = meta_column+1; column_index <= this.max_width; column_index++){
			HashMap<String, Datum> tuple = new HashMap<String, Datum>();
			for(int row_index = start_row; row_index <= this.max_row; row_index++){
				Datum d = this.getCell(row_index, column_index);
				if(d != null){
					if(d.getT()==Datum.Type.EMPTY){
						continue;
					}
					if(meta[row_index] == null){
						System.err.println("horizontal field out of bounds: " + column_index);
						continue;
					}
					//System.out.println("dat: " + d.getString_dat());
					tuple.put(meta[row_index], d);
				}
			}
			if(tuple.size() > 0){
				tuples.add(tuple);
			}
		}

		return tuples;
	}
	/**
	 * Use this to get a list of tuples
	 * from a worksheet organized with the meta horizontally<br>
	 * tuples numbered top to bottom.
	 * @return
	 */
	public List<HashMap<String, Datum>> extractHorizontalTuples(int meta_row){
		List<HashMap<String, Datum>> tuples = new ArrayList<HashMap<String, Datum>>();

		//use a sorted list of rows
		Integer [] rows = new Integer[this.rows.size()];
        System.out.println( "Rows on this sheet: " + this.rows.size() );
		this.rows.keySet().toArray(rows);
		Arrays.sort(rows);

		//obtain the taxonomy
		Row naming = this.rows.get(meta_row);
		String [] meta = new String[naming.getMax()+1];
		for(int index : naming.getColumns()){
			Datum d = naming.getColumn(index);
			if(d != null){
				meta[index] = d.getString_dat();
	            //System.out.println("meta: " + meta[index]);
            }
		}

		// populate each tuple
		for(int row_index : rows ){
			if(row_index <= meta_row){
				continue;
			}
			Row row = this.rows.get(row_index);
			HashMap<String, Datum> tuple = new HashMap<String, Datum>();
			for(int column_index : row.getColumns()){
				if(meta[column_index] == null){
					System.err.println("horizontal field out of bounds: " + column_index);
					continue;
				}
				//System.out.println("checking: " + row_index + " by " + column_index);
                Datum d = this.getCell(row_index, column_index);
	            //	Datum d = row.getColumn(column_index);
				if(d != null){
					if(d.getT() == Datum.Type.EMPTY){
						continue;
					}
					//System.out.println(d.getString_dat());
                    tuple.put(meta[column_index], d);
				}
			}
			if(tuple.size() > 0){
				tuples.add(tuple);
			}
		}

		return tuples;
	}

	public Collection<Row> getRows(){
		return this.rows.values();
	}

	public Datum getCell(int row_index, int column_index){
		Row row = this.getRow(row_index);
		if(row == null){
			return null;
		}
		return row.getColumn(column_index);
	}

	public void setCell(Datum d, int row_index, int column_index){
		Row row = this.getOrAddRow(row_index);
		row.addColumn(column_index, d);
	}

	public Row getRow(int row_index){
		if( ! this.rows.containsKey(row_index)){
			return null;
		}
		return this.rows.get(row_index);
	}

	public Row getOrAddRow(int row_index){
		Row row = this.getRow(row_index);
		if(row == null){
			row = new Row(row_index);
			this.addRow(row);
		}
		return row;
	}

	public int getSize(){
		return this.rows.size();
	}
	public void clear(){
		for(Row row : this.rows.values()){
			row.clear();
		}
		this.rows.clear();
	}
	public String getName(){
		return this.title;
	}
	public int getCardinality() {
		return cardinality;
	}

}
