/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package test.org.mitre.lattice.lattice;

import java.util.ArrayList;
import java.util.List;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.mitre.lattice.lattice.InvalidLatticeStructureException;
import org.mitre.lattice.lattice.LatticeNode;
import org.mitre.mrald.util.Config;
/**
 *  Description of the Class
 *
 *@author     JCHOYT
 *@created    June 5, 2003
 */
public class LatticeNodeTest extends TestCase
{
    LatticeNode childNode1;
    LatticeNode childNode2;
    LatticeNode childNode3;
    //LatticeNode[] children;
    LatticeNode mainNode;
    LatticeNode parentNode1;
    LatticeNode parentNode2;
    //LatticeNode[] parents;

    ArrayList<LatticeNode> children = new ArrayList<LatticeNode>();
    ArrayList<LatticeNode> parents = new ArrayList<LatticeNode>();


    /**
     *  The main program for the LatticeNodeTest class
     *
     *@param  args  The command line arguments
     */
    public static void main( String[] args )
    {
        junit.textui.TestRunner.run( suite() );
    }


    /**
     *  A unit test suite for JUnit
     *
     *@return    The test suite
     */
    public static Test suite()
    {
        /*
         *  the dynamic way
         */
        return new TestSuite( LatticeNodeTest.class );
    }



    /**
     *  Description of the Method
     */
    public void resetNodes()
    {
        setUp();
    }


    /**
     *  test that adding children works
     */
    public void testAddChildren()
    {
       try
       {
        /*
         *  adding null children
         */
        LatticeNode loneNode = Config.getLatticeFactory().createNode( "loner" );
        assertEquals( "loneNode should have no children", 0, loneNode.getChildren().size());
        /*
         *  adding one child
         */
        java.util.ArrayList<LatticeNode> nodeArray = new java.util.ArrayList<LatticeNode>();
	nodeArray.add(childNode1);

        loneNode.addChildren( nodeArray );
        assertEquals( "loneNode should have one child", 1, loneNode.getChildren().size() );
        assertTrue( "the child node should be childNode1", loneNode.getChildren().get(0) == childNode1 );
        /*
         *  adding three children
         */

	assertEquals( "loneNode should have already added this child ", false, loneNode.addChild(childNode1) );
	assertEquals( "loneNode should have added a child ", true, loneNode.addChild(childNode2) );

        loneNode.addChildren( children );
        assertEquals( "loneNode should have two children", 2, loneNode.getChildren().size() );
        List childNodes =  loneNode.getChildren() ;
        assertTrue( "childNode1 should be one of the child nodes", childNodes.contains( childNode1 ) );
        assertTrue( "childNode2 should be one of the child nodes", childNodes.contains( childNode2 ) );
        assertEquals( "childNode3 should not be one of the child nodes", false, childNodes.contains( childNode3 ) );
       }
       catch(Exception e)
       {
	 fail(e.getMessage());
       }
    }


    /**
     *  test that adding parentren works
     */
    public void testAddParents()
    {
      try
      {
        /*
         *  adding null parentren
         */
        LatticeNode loneNode = Config.getLatticeFactory().createNode( "loner" );
        /*loneNode.addParents( null );*/
        assertEquals( "loneNode should have no parents", 0, loneNode.getParents().size() );
        /*
         *  adding one parent
         */
        loneNode = Config.getLatticeFactory().createNode( "loner" );
        ArrayList<LatticeNode> nodeArray = new ArrayList<LatticeNode>();
	nodeArray.add(parentNode1);
        loneNode.addParents( nodeArray );
        assertEquals( "loneNode should have one parent", 1, loneNode.countParentNodes() );
        assertTrue( "the parent node should be parentNode1", loneNode.getParents().get(0) == parentNode1 );
        /*
         *  adding three parentren
         */
        loneNode = Config.getLatticeFactory().createNode( "loner" );
        loneNode.addParents( parents );
        assertEquals( "loneNode should have three parents", 2, loneNode.getParents().size() );
        List parentNodes = loneNode.getParents();
        assertTrue( "parentNode1 should be one of the parent nodes", parentNodes.contains( parentNode1 ) );
        assertTrue( "parentNode2 should be one of the parent nodes", parentNodes.contains( parentNode2 ) );
      }
      catch(InvalidLatticeStructureException e)
      {
	      fail(e.getMessage());
      }
    }


    /*
     *  a node that constructed from another node should be a clone if it, not the exact same node
     *  (or should it?  Todd - your call)
     */
    /**
     *  A unit test for JUnit
     */
    public void testInitialization()
    {
        LatticeNode cloneNode = Config.getLatticeFactory().createNode( mainNode );
        assertTrue( "These two should NOT be equal", mainNode != cloneNode );
        /*assertEquals( "These two should NOT be the same object, but clones of each other", mainNode, cloneNode );*/
    }


    /**
     *  A unit test for JUnit
     */
    public void testRemoveChildren()
    {
        /*
         *  remove all children
         */
        mainNode.removeChildren( children );
        List childNodes =  mainNode.getChildren() ;
        assertFalse( "childNode1 should no longer be one of the child nodes", childNodes.contains( childNode1 ) );
        assertFalse( "childNode2 should no longer be one of the child nodes", childNodes.contains( childNode2 ) );
        assertFalse( "childNode3 should no longer be one of the child nodes", childNodes.contains( childNode3 ) );
        /*
         *  remove one child
         */
        resetNodes();

        ArrayList<LatticeNode> oneKid = new ArrayList<LatticeNode>();
    	oneKid.add(childNode1 );
        mainNode.removeChildren( oneKid );
        childNodes =  mainNode.getChildren() ;
        assertFalse( "childNode1 should no longer be one of the child nodes", childNodes.contains( childNode1 ) );
        assertTrue( "childNode2 should still be one of the child nodes", childNodes.contains( childNode2 ) );
        assertTrue( "childNode3 should still be one of the child nodes", childNodes.contains( childNode3 ) );
        /*
         *  if any children do not exist then RemoveCHildren should fail
         */

	assertTrue("The nodes should still be consistent", mainNode.checkConsistent( false, children ));

	mainNode.removeChildren( children );

        childNodes = mainNode.getChildren() ;
        assertFalse( "childNode1 should no longer be one of the child nodes", childNodes.contains( childNode1 ) );
        assertTrue( "childNode2 should still be one of the child nodes", childNodes.contains( childNode2 ) );
        assertTrue( "childNode3 should still be one of the child nodes", childNodes.contains( childNode3 ) );
    }


    /**
     *  A unit test for JUnit
     */
    public void testRemoveParents()
    {
        /*
         *  remove all parents
         */
        mainNode.removeParents( parents );
        List parentNodes = mainNode.getParents();
        assertFalse( "parentNode1 should no longer be one of the parent nodes", parentNodes.contains( parentNode1 ) );
        assertFalse( "parentNode2 should no longer be one of the parent nodes", parentNodes.contains( parentNode2 ) );
        /*
         *  remove one parent
         */
        resetNodes();
        ArrayList<LatticeNode> oneDult = new ArrayList<LatticeNode>();
	oneDult.add(parentNode1);
        mainNode.removeParents( oneDult );
        parentNodes =  mainNode.getParents() ;
        assertFalse( "parentNode1 should no longer be one of the parent nodes", parentNodes.contains( parentNode1 ) );
        assertTrue( "parentNode2 should still be one of the parent nodes", parentNodes.contains( parentNode2 ) );
        /*
         *  removing parents that aren't there shoudl be graceful
         */
        mainNode.removeParents( parents );
        parentNodes =  mainNode.getParents() ;

	  /*
         *  if any parent do not exist then RemoveCHildren should fail
         */

	assertTrue("The nodes should no longer be consistent", mainNode.checkConsistent( false, children ));

	parentNodes =  mainNode.getParents() ;

        assertFalse( "parentNode1 should no longer be one of the parent nodes", parentNodes.contains( parentNode1 ) );
        assertTrue( "parentNode2 should still be one of the parent nodes", parentNodes.contains( parentNode2 ) );

    }

    /**
     *  The JUnit setup method
     */
    protected void setUp()
    {
       try
       {
        Config.setProperty("LOGPATH", "/devel/gail/webapps/digilib/webapps/ROOT/mrald.log");
        Config.setProperty("mraldLogLevel", "2");
        Config.setLatticeFactory( new org.mitre.mrald.util.LatticeFactory() );
        mainNode = Config.getLatticeFactory().createNode( "mainNode" );
        parentNode1 = Config.getLatticeFactory().createNode( "parentNode1" );
        parentNode2 = Config.getLatticeFactory().createNode( "parentNode2" );
        childNode1 = Config.getLatticeFactory().createNode( "childNode1" );
        childNode2 = Config.getLatticeFactory().createNode( "childNode2" );
        childNode3 = Config.getLatticeFactory().createNode( "childNode3" );

        children.add(childNode1);
        children.add(childNode2);
        children.add( childNode3);
        parents.add( parentNode1);
        parents.add(parentNode2);
        mainNode.addChildren( children );
        mainNode.addParents( parents );
       }
       catch(Exception e)
       {
         fail(e.getMessage());
       }
    }


    /**
     *  The JUnit teardown method
     */
    protected void tearDown()
    {
    }
}


