/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.neuro;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

/**
 * scponly is a way of creating a chroot jail which allows users to interact
 * with sftp/scp without giving them the ability to get a terminal into the
 * machine. As an example, scponly user 'tom' may %sftp tom@hostname:22; and an
 * %ls -l; of the root '/' will yield the listing relative to their home
 * directory on the system proper, '/home/tom/'. Their home directory will
 * contain everything needed to run sftp/scp, a simple set of binaries, but
 * nothing more. The compromise of a single account, then, does not jeopardize
 * the entire system. <br>
 * Each user will have their own root with which to interact and thus will not
 * be allowed to see the rest of the system or the upload accounts of any other
 * user. <br>
 * The JRE does not accomplish this task, instead it forks processes defined by
 * bash scripts as the root system user via sudo. This makes it a unix sysV
 * dependent implementation, but it is at least as portable as bash, sudo, ssh,
 * and the scponlyc implementation. <br>
 * The neuroweb user on the prototype virtual box is set up to allow sudo without a
 * password on the setup_chroot and delete_chroot executables in it's home
 * directory. this is accomplished by adding the following line to the visudo
 * file: neuroweb ALL=(ALL) NOPASSWD: /home/neuroweb/bin/delete_chroot,
 * /home/neuroweb/bin/create_chroot <br>
 * These executables are symlinks into the project source directory, and are
 * bash scripts hacked together by MITRE(esax).
 *
 * @author esax
 *
 */
public class Scponly {
	public static final String delete_path = "/usr/local/bin/delete_chroot.sh";
	public static final String create_path = "/usr/local/bin/setup_chroot.sh";
	public static final String passwd = "/home/neuroweb/etc/passwd";
	public static final String backups = "/home/neuroweb/etc/backup_uploads";

	/**
	 * Specify the configuration file
	 *
	 * @param config
	 */
	public Scponly(File config) {
	}
	public Scponly(){

	}
	public static boolean userExists(String username) throws Exception{
		Process process = Runtime.getRuntime().exec("id " + username);
		if(process.waitFor()==0){
			return true;
		}
		else{
			return false;
		}
	}
	public static List<String> getPasswd(String username) throws Exception{
		List<String> list = new ArrayList<String>();
		BufferedReader reader = new BufferedReader(new FileReader(passwd));
		String line = "";
		while((line = reader.readLine()) != null){
			if(line.contains(username)){
				list.add(line);
			}
		}
		reader.close();
		return list;
	}

	public static String makeUserPassword(String username){
		try {
			String pass = generatePassword();
			PrintWriter out = new PrintWriter(new FileOutputStream(new File(passwd), true));
			out.println(username + ":" + pass);
			out.close();
			return pass;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	public static String generatePassword() throws Exception{
		MessageDigest m = null;
		try {
			m = MessageDigest.getInstance("SHA-256");
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
		FileInputStream fis = new FileInputStream(new File("/dev/random"));
		byte [] input = new byte[1024];
		int len = 0;
		if((len = fis.read(input) ) > 0){
				m.update(input, 0, len);
		}
		fis.close();
		//format hash bytes as hexidecimal
	    return asHex(m.digest()).substring(0, 8);
	}
	private static final char[] HEX_CHARS = "0123456789abcdef".toCharArray();

    public static String asHex(byte[] buf)
    {
        char[] chars = new char[2 * buf.length];
        for (int i = 0; i < buf.length; ++i)
        {
            chars[2 * i] = HEX_CHARS[(buf[i] & 0xF0) >>> 4];
            chars[2 * i + 1] = HEX_CHARS[buf[i] & 0x0F];
        }
        return new String(chars);
    }
	protected static Process start_process(String command) throws Exception {
		System.out.println("Running: " + command);
		return Runtime.getRuntime().exec(command);
	}

	protected static List<String> run_process(Process process) throws Exception {
		List<String> results = new ArrayList<String>();
		// process.waitFor();
		BufferedReader reader = new BufferedReader(new InputStreamReader(
				process.getInputStream()));
		String line = null;
		while ((line = reader.readLine()) != null) {
			results.add(line);
		}
		int error_code = process.waitFor();
		if (error_code == 0) {
			results.add("The command completed successfully");
		} else {
			results.add("There was an error. Please inform the sysadmin or webmaster of this problem. ");
		}
		return results;
	}

	/**
	 * Adds the user to the system as an scponly user
	 *
	 * Accomplishes: useradd chsh scponly mkdir $HOME
	 *
	 * @param userid
	 * @param password
	 */
	public static List<String> adduser(String userid, String password)
			throws Exception {
		String command = "sudo " + create_path + " " + userid + " " + password
				+ " NO 2>&1";
		Process process = start_process(command);
		return run_process(process);
	}

	public static List<String> adduser(String userid, String password,
			boolean restore) throws Exception {
		if(!restore){
			return adduser(userid, password);
		}

		String command = "sudo " + create_path + " " + userid + " " + password
				+ " restore 2>&1";
		Process process = start_process(command);
		return run_process(process);
	}

	/**
	 * removes the system account for this user id
	 *
	 * @param userid
	 */
	public static List<String> deluser(String userid) throws Exception {
		// System.out.println("Removing user: " + userid);
		String command = "sudo " + delete_path + " " + userid + " backup 2>&1";
		Process process = start_process(command);
		return run_process(process);
	}

	/**
	 * This is really just for testing. The class itself exposes the "API", use
	 * the scripts directly instead of invoking this jre.
	 *
	 * @param args
	 */
	public static void main(String[] args) {
		System.out.println("starting...");
		try {
			// for(String line : adduser("tester", "test")){
			for (String line : deluser("tester")) {
				System.out.println("STDOUT: " + line);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		System.out.println("done.");
	}
}
