/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.mitre.neuro;

import java.io.File;

import java.io.FileInputStream;

import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.mitre.neuro.containers.Interview;
import org.mitre.neuro.containers.Session;
import org.mitre.neuro.containers.Study;
import org.mitre.neuro.containers.Subject;
import org.mitre.neuro.excel.Datum;
import org.mitre.neuro.excel.ExcelReader;
import org.mitre.neuro.excel.Worksheet;
import org.mitre.neuro.excel.biff.BIFFReader;

public class SpreadSheetReader {
    ExcelReader reader;
    File baseDir = null;

    public enum InputType {
        BIFF
        //,HTML
    };

    public SpreadSheetReader(InputType type){
        this.reader = new BIFFReader();
    }


    public Study read(InputStream stream, String baseDir) throws Exception{
        this.baseDir = new File(baseDir);
    //    System.out.println("Reading...");
        HashMap<String, Worksheet> sheets = this.reader.readSheets(stream);
    //    System.out.println("Worksheets extracted...");

        System.out.println( "Extracting from Study" );
        Study study = extractStudy(sheets.get("Study"));
        System.out.println( "Extracting from Subjects" );
        Map<String, Subject> subjects = this.extractSubjects( sheets.get("Subjects") );
        System.out.println( "Extracting from Sessions" );
        this.extractSessions(study, sheets.get("Sessions"), subjects);
        System.out.println( "Extracting from Scans" );
        this.extractImages(study, sheets.get("Scans"), subjects);
        System.out.println( "Extracting from Interviews" );
        this.extractInterviews(sheets.get("Interviews"), subjects);
    //    System.out.println("Done reading xls...");
        System.out.println( study.toString() );
        return study;
    }

    protected void extractInterviews(Worksheet sheet, Map<String, Subject> subjects) throws Exception {
        if(sheet == null){
            System.err.println("Interview sheet not found");
            return;
        }
        List<HashMap<String, Datum>> tuples = sheet.extractHorizontalTuples(2);

        for (Map<String, Datum> tuple : tuples){
            String subject_id = this.extractString(tuple, "Subject ID");
            String interview_date = this.extractString(tuple, "Interview Date");
            // System.out.println( interview_date );
            tuple.remove("Subject ID");
            tuple.remove("Interview Date");
            tuple.remove("Session ID");

            Interview interview = new Interview();
            Subject subject = subjects.get(subject_id);
            interview.setInterview_date( interview_date );
            if(subject == null){
                throw new Exception("Subject " + subject_id + " not defined.");
            }

            for (String key : tuple.keySet()){
                String value = this.extractString(tuple, key);
                if(value == null){
                    continue;
                }
                interview.addValue(key, value);
            }
            subject.addInterview(interview);
        }
    }

    protected void extractSessions(Study study, Worksheet sheet, Map<String, Subject> subjects){
        if(sheet == null){
            System.err.println("Sessions sheet not found");
            return;
        }
        List<Session> sessions = new ArrayList<Session>();
        List<HashMap<String, Datum>> tuples = sheet.extractHorizontalTuples(2);
        Session session;

        for(Map<String, Datum> tuple : tuples){
            String subject = extractString( tuple,"Subject ID");
            String session_id =  extractString(tuple,"Session ID");
            // System.out.println( "subject: " + subject + " session_id:" + session_id);
            String dte =  extractString(tuple,"Date");
            String scanner =  extractString(tuple,"Scanner");
            String location =  extractString(tuple,"Location");

            String software_v = extractString(tuple,"Software Version");

            session = new Session();
            /* these three are used to define a unique session */
            session.setSubject(subjects.get(subject));
            session.setSession_id( session_id );
            session.setDate_taken( dte );

            int existingSession = sessions.indexOf( session );
            if( existingSession == -1 )
            {
                //this is a new session
                sessions.add(session);
                // System.out.println( "Added session for " + session.getSubject().getSubject_id() + session.getSession_id() + session.getDate_taken() );
            }
            else
            {
                //grab the existing one
                session = sessions.get( existingSession );
            }

            session.setScanner( scanner );
            session.setLocation( location );
            session.setSoftware_version( software_v );
        }
        study.setSessions(sessions);
    }

    protected void extractImages(Study study, Worksheet sheet, Map<String, Subject> subjects)
    {
        if(sheet == null){
            System.err.println("Scans sheet not found");
            return;
        }
        List<Session> sessions = study.getSessions();
        List<HashMap<String, Datum>> tuples = sheet.extractHorizontalTuples(2);
        Session session;

        for(Map<String, Datum> tuple : tuples){
            String subject =  extractString(tuple,"Subject ID");
            String session_id =  extractString(tuple,"Session ID");
            String dte =  extractString(tuple,"Date");
            session = new Session();
            // these three are used to define a unique session - want to add the image to the proper session
            session.setSubject(subjects.get(subject));
            session.setSession_id( session_id );
            session.setDate_taken( dte );
            // System.out.println( sessions.size() );
            int existingSession = sessions.indexOf( session );
            if( existingSession == -1 )
            {
                throw new RuntimeException( "A Session to match the scan for " + session.getSubject().getSubject_id() + " on date " +
                    session.getDate_taken()  + " with a session id of " + session.getSession_id()  +
                    " could not be found.  This is an error in the spreadsheet.  Please fix it and try again." );
            }
            else
            {
                //grab the existing one
                session = sessions.get( existingSession );
            }

            //now we have the session that was previously created...populate a new image for it.
            String filename= tuple.get("Filename").getString_dat();
            File uploadedFile = null;
            if( filename != null )
            {
                uploadedFile = new File(baseDir, filename);
            }
            else
                throw new RuntimeException("You must specify a file name for subject " + subject + " on the Scans sheet." );
            String protocol_name= extractString(tuple,"Protocol Name");
            String modality= extractString(tuple,"Modality");
            double t1=extractInteger(tuple,"T1");
            double t2=extractInteger(tuple,"T2");
            double tr=extractInteger(tuple,"TR");
            double ti=extractInteger(tuple,"TI");
            double te=extractInteger(tuple,"TE");
            String num_format=extractString(tuple,"Numercial Format");
            double excitations=extractInteger(tuple,"Excitations");
            double flip_angle=extractInteger(tuple,"Flip Angle");

            Session.Image image = session.addImage();
            image.filename=filename;
            image.subject_oid = session.getSubject().getObject_id();
            image.missing = !uploadedFile.exists();
            image.t1=t1;
            image.t2=t2;
            image.tr=tr;
            image.ti=ti;
            image.te=te;
            image.num_format=num_format;
            image.flip_angle=flip_angle;
            image.excitations=excitations;
            image.modality=modality;
            image.protocol_name=protocol_name;
            }

    }

    protected Map<String, Subject> extractSubjects(Worksheet sheet){
        if(sheet == null){
            System.err.println("Subject worksheet not found!");
            return null;
        }
        Map<String,Subject> subjects = new HashMap<String, Subject>();
        List<HashMap<String, Datum>> tuples = sheet.extractHorizontalTuples(2);
        for(Map<String, Datum> tuple : tuples){
            Subject subject = new Subject();
            subject.setSubject_id(  extractString(tuple, "Subject ID"));
            subject.setBirth( extractDate(  tuple, "Birthdate"));
            subject.setGender( extractString(tuple, "Gender"));
            subject.setRace( extractString(tuple, "Race"));
            subject.setEducation( extractInteger(tuple, "Education"));
            subject.setHandedness( extractString(tuple, "Handedness"));
            subject.setHeight( extractInteger(tuple, "Height"));
            subject.setWeight( extractInteger(tuple, "Weight"));

            subjects.put(subject.getSubject_id(), subject);
        }
        return subjects;
    }

    protected Study extractStudy(Worksheet sheet){
        if(sheet == null){
            System.err.println("Study worksheet not found!");
            return null;
        }
        Study study = new Study();
        List<HashMap<String, Datum>> tuples = sheet.extractVerticalTuples(0, 2);
        //should only be one row
        if(tuples.size() != 1){
            System.err.println("Study information not found!");
            return null;
        }
        Map<String, Datum> tuple = tuples.get(0);
        study.setInvestigator(tuple.get("PI Name").getString_dat());
        study.setPi_email(tuple.get("PI Email Address").getString_dat());
        study.setFunding_info(tuple.get("Funding Information").getString_dat());
        study.setGrant_name(tuple.get("Grant name").getString_dat());
        return study;
    }

    /*
     * Helper functions
     */


    protected int extractInteger(Map<String, Datum> tuple,String field){
        String str = tuple.containsKey(field)?tuple.get(field).getString_dat():null;
        int num = str==null?Integer.MIN_VALUE:Integer.parseInt(str);
        return num;
    }
    protected String extractString(Map<String, Datum> tuple, String field){
        String str = tuple.containsKey(field)?tuple.get(field).getString_dat():null;
        return str;
    }

    protected Date extractDate(Map<String, Datum> tuple, String field){
        String str = extractString(tuple, field);
        try {
              return str==null?null:new Date(str);
        } catch (Exception e ) {
            return extractYear(str);
       }
    }
    protected Date extractYear(String text){
         SimpleDateFormat format =
           new SimpleDateFormat("yyyy");
         try{
              return format.parse(text);
         } catch (Exception e ){
            System.out.println("Failed extracting year: " + text);
            return null;
         }
    }

    public static void main(String[] args)
    {
        try
        {
            InputStream in = new FileInputStream(args[0]);
            SpreadSheetReader reader = new SpreadSheetReader( InputType.BIFF );
            reader.read( in, args[0] );
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }
}
