/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.neuro;

import java.io.File;
import java.io.FileInputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.mitre.Unpack;
import org.mitre.neuro.SpreadSheetReader.InputType;
import org.mitre.neuro.containers.Session;
import org.mitre.neuro.containers.Study;
import org.mitre.neuro.database.StoreStudy;

public class Upload{

	Connection connection;
    String study_name;
	public static final String TMP_DIRECTORY = "/home/neuroweb/tmp/staging";

	public Upload(Connection connection, String study_name){
		this.connection = connection;
	    this.study_name = study_name;
    }

	public  void beginTransaction() throws Exception{
		connection.setAutoCommit(false);
	}
	public  void endTransaction() throws Exception{
		connection.commit();
	}
	public  void rollbackTransaction() throws Exception{
		connection.rollback();
	}
	/**
	 * Verifies the data in the specified upload directory
	 * @param directory
	 * @throws Exception
	 */
	public List<Session> validate(File target) throws Exception {
		this.beginTransaction();
		List<Session> scans = this.merge(target);
		this.rollbackTransaction();
		return scans;
	}

	/**
	 * Loads and commits data in the specified upload directory
	 * @param directory
	 * @throws Exception
	 */
	public List<Session> merge(File target) throws Exception {
		List<Session> scans = null;
		if(target.isFile()){
			scans = this.load_file(target);
		}
		else{
			scans = this.load_directory(target.listFiles());
		}
		return scans;
	}

	public List<Session> load_archive(File archive) throws Exception{
		if ( archive.getName().contains(".tar.") ){
			File tarball = null;
			if(archive.getName().endsWith(".gz")){
				tarball = Unpack.gunzip(archive, TMP_DIRECTORY + "/" +
						archive.getParent());
			} else if(archive.getName().endsWith("bz2")) {
				tarball = Unpack.bunzip2(archive, TMP_DIRECTORY + "/" +
						archive.getParent());
			}
			else {
				throw new Exception ("Unsupported archive type: " + archive.getName());
			}
			File directory = Unpack.unTar(tarball);
			return this.load_directory(directory.listFiles());
		}

		if ( archive.getName().endsWith(".zip") ) {
			File directory = Unpack.unZip(archive, TMP_DIRECTORY + "/" +
							archive.getParent());
			return this.load_directory(directory.listFiles());
		}

		throw new Exception ("Unsupported archive type: " + archive.getName());
	}

	public List<Session> load_xls(File spreadsheet) throws Exception{
		StoreStudy database = new StoreStudy(connection);
		ImageLoading image_loader = new ImageLoading(spreadsheet.getParent(), this.connection);

	//	System.out.println("Reading: " + spreadsheet.getName());

		FileInputStream input = new FileInputStream( spreadsheet );
		SpreadSheetReader reader = new SpreadSheetReader(InputType.BIFF);
		Study study = reader.read(input, spreadsheet.getParent());
		input.close();
        if ( ! study.getGrant_name().equals(this.study_name)) {
            return new ArrayList<Session>();
        }
		if( ! image_loader.check_scans(study)){
			System.out.println("Missing files...");
			return study.getSessions();
		}

	//	System.out.println( study.getSessions().size() + " scans found...");

		database.store(study);
		image_loader.queue_scans(study);
		return study.getSessions();
	}
	/**
	 * Unpacks and recurses into archives, reads xls files for meta-data
	 */
	public List<Session> load_file(File file) throws Exception{
		if( file.getName().endsWith(".tar.gz")
				|| file.getName().endsWith(".tar.bz2")
				|| file.getName().endsWith(".zip")){
			return this.load_archive(file);
		}
		if( ! file.getName().endsWith(".xls")){
			return null;
		}
		try{
			return this.load_xls(file);
		} catch (Exception e){
			System.out.println("Exception, rolling back..");
			this.rollbackTransaction();
			throw e;
		}
	}

    /** This is the one used by uploadStatus.jsp */
	public List<Session> load_directory(File [] list) throws Exception{
		if(list == null){
            System.out.println( "directory listing isn't working..null being passed to Upload.load_directory" );
			return null;
		}
		List<Session> scan_images = new ArrayList<Session>();
		for(File file : list){
			if ( file.getName().startsWith(".")) {
				continue;
			}
			if(file.isDirectory()){//DFS
				scan_images.addAll(this.load_directory(file.listFiles()));
				continue;
			}
			List<Session> scans_from_file = this.load_file(file);
			if (scans_from_file != null) {
				scan_images.addAll(scans_from_file);
			}
		}
		return scan_images;
	}

	/*
	 * Static methods
	 * 	for main execution (testing)
	 */

	public static void printTime(){
		System.out.println(new Date(System.currentTimeMillis()));
	}

	//TODO: pull from config
	public static Connection connect(){
		try{
			Class.forName("org.postgresql.Driver").newInstance();
			return DriverManager.getConnection(
					"jdbc:postgresql://localhost:5432/neuroweb",
					"jchoyt",
			"ou812ic");
		} catch (Exception e){
			e.printStackTrace();
		}

		return null;
	}

	public static void main(String [] args){
		if(args.length != 1){
			System.err.println("Please specify a directory to probe");
		}
		File incoming_dir = new File(args[0]);
		if( ! incoming_dir.exists() ){
			System.err.println("Directory " + incoming_dir + " does not exist");
			System.exit(-1);
		}
		Connection connection = connect();
		if(connection == null){
			System.err.println("Failed to connect to database...");
			System.exit(-1);
		}

		System.out.println("checking: " + incoming_dir);
		try{
			Upload upload = new Upload(connection, ".");
			List<Session> scans = upload.validate(incoming_dir);
			if(scans == null){
				System.err.println("No scans found!");
				System.exit(-1);
			}
			else{
				System.out.println("\n\n\nExtracted Sessions:");
				for (Session scan : scans) {
					for ( Session.Image file : scan.getImages())
						System.out.println(
							file.filename + ": "
						    + (file.missing?"missing":"found"));
				}
			}
		}catch(Exception e){
			e.printStackTrace();
		}finally{
			try{//clean up the connection
				connection.close();
			}catch(Exception e){}
		}
	}
}
