/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.util.*;
import java.io.*;
import java.sql.*;


/**
 *  This file enters a new Interview session into the database.  If the entity/attributes aren't already in there, they
 *  will be created, therefore care is advises when using this.
 *
 *  An example of a successful program run is below.
 */
public class InsertInterview
{
    // order of ? is date, value, attribute_id, subject_oid
    private String insertQuery = "insert into subject_values values (?,?,?,?)";
    private String getSubjectOidQuery = "select subject_oid from subject where subject_id=?";
    private String getEntityIdQuery = "select entity_id from entity where name=?";
    private String getAttributeIdQuery = "select attribute_id from attribute a where a.name=? and a.entity_id=?";
    private Connection conn = null;

    private int subjectOid = 0;
    private java.sql.Date date = null;
    private Properties dbConfig = new Properties();
    private Properties interview = new Properties();

    public static void main (String[] args)
        throws Exception
    {
        InsertInterview doer = new InsertInterview();
        if( args.length != 2 )
        {
            System.out.println( "usage: java InsertInterview <db config file> <interview file>" );
            System.exit(10);
        }
        doer.doInsert(args[0], args[1]);
    }

    /**
     *  Programmic entry and initialization
     */
    public void doInsert(String config, String interviewFile)
        throws Exception
    {
        //intialization
        dbConfig.load(new FileReader(config));
        interview.load(new FileReader(interviewFile));
        Class.forName( dbConfig.getProperty("driver"));
        conn = DriverManager.getConnection(dbConfig.getProperty("server"), dbConfig);
        String dateString = interview.getProperty("date");
        try
        {
            date = java.sql.Date.valueOf( dateString );
        }
        catch (IllegalArgumentException e)
        {
            throw new IllegalArgumentException("The date in the interview file must be in the form YYYY-MM-DD", e);
        }
        subjectOid = getSubjectOid( );
        processInserts();
        conn.close();
    }


    /**
     *  Real work goes here.  For each property that isn't "date" or "subject id", insert it into the database.
     */
    private void processInserts()
    throws Exception
    {

         for (Enumeration<String> e = (Enumeration <String>)interview.propertyNames(); e.hasMoreElements();)
         {
             String name = e.nextElement();
             if( name.equals("date") || name.equals("subjectid") )
             {
                 continue;
             }
             // System.out.println( name );
             String[] s = name.split("\\.");
             String entityName = s[0];
             String attributeName = s[1];
            int entityId = getEntityId( entityName );
            int attributeId = getAttributeId( entityId, attributeName );
            //TODO: insert the new subject values
            try
            {
                insertValue( attributeId, interview.getProperty(name));
                System.out.println( name + " inserted for subject " + subjectOid );
            }
            catch (SQLException se)
            {
                System.out.println( name + " NOT inserted.  An error occured: " + se.getMessage() );
            }
         }

    }


    private void insertValue( int attributeId, String value )
    throws Exception
    {
        PreparedStatement s = conn.prepareStatement( insertQuery );
        s.setDate( 1, date );
        s.setString( 2, value );
        s.setInt( 3, attributeId );
        s.setInt( 4, subjectOid );
        s.executeUpdate();
    }


    private int getAttributeId( int entityId, String name )
    throws Exception
    {
        PreparedStatement s = conn.prepareStatement( getAttributeIdQuery );
        s.setString( 1, name );
        s.setInt( 2, entityId );
        ResultSet rs = s.executeQuery();
        if(rs.next())
        {
            return rs.getInt( 1 );
        }
        else
        {
            //doesn't exist...create one
            //get the new attribute_id first
            Statement st = conn.createStatement();
            rs=st.executeQuery("select max(attribute_id) from attribute");
            int attribute_id = 0;
            rs.next();
            attribute_id=rs.getInt(1) + 1;
            rs.close();
            st.close();
            //use that to do the insert
            s = conn.prepareStatement( "insert into attribute (entity_id, attribute_id, name) values (?, ?, ? )" );
            s.setInt( 1, entityId );
            s.setInt( 2, attribute_id );
            s.setString( 3, name );
            s.executeUpdate();
            s.close();
            System.out.println( name + " was not found as an existing attribute for that entity, but it has been created.  Contact the administrator to provide a good, meaningful description for the new attribute."  );
            return attribute_id;
        }
    }

    private int getEntityId( String name )
    throws Exception
    {
        PreparedStatement s = conn.prepareStatement( getEntityIdQuery );
        s.setString( 1, name );
        ResultSet rs = s.executeQuery();
        if(rs.next())
        {
            return rs.getInt( 1 );
        }
        else
        {
            rs.close();
            //doesn't exist...create one
            //get the new entity_id first
            Statement st = conn.createStatement();
            rs=st.executeQuery("select max(entity_id) from entity");
            int entity_id = 0;
            rs.next();
            entity_id=rs.getInt(1) + 1;
            rs.close();
            st.close();
            //use that to do the insert
            s = conn.prepareStatement( "insert into entity (entity_id, name, type) values (?, ?, 'Untyped' )" );
            s.setInt( 1, entity_id );
            s.setString( 2, name );
            s.executeUpdate();
            s.close();
            System.out.println( name + " was not found as an existing entity, but it has been created.  Contact the administrator to provide a good, meaningful description for the new entity."  );
            return entity_id;
        }
    }


    private int getSubjectOid( )
    throws Exception
    {
        PreparedStatement s = conn.prepareStatement( getSubjectOidQuery );
        s.setString( 1, interview.getProperty("subjectid") );
        ResultSet rs = s.executeQuery();
        if(rs.next())
        {
            return rs.getInt( 1 );
        }
        else
        {
            throw new RuntimeException( "Subject " + interview.getProperty("subjectid") + " is not in the database."  );
        }
    }
}

        // //get a reference to the statement
        // PreparedStatement s = conn.prepareStatement( query );
        // s.setString( 1, filename );
        // ResultSet rs = s.executeQuery();
        // Properties output = new Properties();
        // String volumeId = null;
        // ResultSetMetaData md = rs.getMetaData();
        // int count = md.getColumnCount();

        // if(rs.next())
        // {
        //     for ( int i = 0; i < count; i++ )
        //     {
        //         String name = md.getColumnName( i + 1 );
        //         String value = rs.getString( i + 1 );
        //         //System.out.println( i + name + value );
        //         if( name.equalsIgnoreCase("volume_id") )
        //         {
        //             volumeId = value;
        //         }
        //         output.setProperty( name, String.valueOf(value));
        //     }
        // }
        // else
        // {
        //     output.setProperty( filename, "not found" );
        // }
        // //FileWriter writer = new FileWriter(volumeId + ".props" );
        // String comments = "This is the dbConfigrmation for file " + filename;
        // output.store( System.out, comments);


/*
Example interview dbConfig file.  Note the escaped spaces (\ ) - these are necessary so Java reads the entire string before the = as the name:
subjectid=1000102
date=2008-12-29
Stroop\ Test.Hits=20
Stroop\ Test.Misses=5
Stroop\ Test.Time=33
FindB.Time100=7
FindB.Time1000=27

Example db config file:
driver=org.postgresql.Driver
user=jchoyt
password=atlas1ng
server=jdbc:postgresql://localhost:5432/atlas

Example:
[jchoyt@mm141783-pc aux]$ java -classpath .:../tomcat/webapps/neuroWeb/WEB-INF/lib/postgresql-8.1-407.jdbc3.jar InsertInterview dbconfig.props 1000102Tests2.properties
Stroop Test.Hits inserted for subject 1
Stroop Test.Misses inserted for subject 1
FindB was not found as an existing entity, but it has been created.  Contact the administrator to provide a good, meaningful description for the new entity.
Time1000 was not found as an existing attribute for that entity, but it has been created.  Contact the administrator to provide a good, meaningful description for the new attribute.
FindB.Time1000 inserted for subject 1
Time100 was not found as an existing attribute for that entity, but it has been created.  Contact the administrator to provide a good, meaningful description for the new attribute.
FindB.Time100 inserted for subject 1
Stroop Test.Time inserted for subject 1

*/
