/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.mrald.analysis;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Enumeration;
import java.util.Properties;
import java.util.Vector;

import org.mitre.mrald.control.AbstractStep;
import org.mitre.mrald.control.MsgObject;
import org.mitre.mrald.control.WorkflowStepException;
import org.mitre.mrald.output.OutputControllerException;
import org.mitre.mrald.util.Config;
import org.mitre.mrald.util.MiscUtils;
import org.mitre.mrald.util.MraldOutFile;


public class AnalysisController extends AbstractStep
{
    protected static Vector keys = new Vector();
    protected static Properties currentClasses;
    protected static PropertyChangeListener pcl;
	protected static String propertyName = "";
    protected MsgObject messageObject = new MsgObject();

    public AnalysisController()
	{
		propertyName = "analysisProps";
	}

    public AnalysisController( MsgObject thisMessage )
    {
		propertyName = "analysisProps";
        messageObject = thisMessage;
    }

    public void execute( MsgObject msgObject )
        throws WorkflowStepException
    {
		if ( currentClasses == null )
		{
			AnalysisController.init();
			Config.addPropertyChangeListener( AnalysisController.pcl );
		}

		try
		{
			messageObject = msgObject;
			processAnalysis();
		}
		catch ( AnalysisControllerException e )
		{
			msgObject.closeOut();
			MiscUtils.handleException( msgObject, e, true );
		}
    }

    protected static void init()
    {
        loadProperties();
        pcl = (
            new PropertyChangeListener()
            {
                public void propertyChange( PropertyChangeEvent evt )
                {
                    loadProperties();
                }
            } );
    }

    /**
     *  Loads the properties from permanent storage. Uses a call to the MRALD
     *  global configuration to get the location of this file. After the
     *  properties are loaded, the keys are stored in a Vector to make retrieval
     *  easier and faster.
     *
     *@since
     */
    @SuppressWarnings("unchecked")
    protected static void loadProperties()
    {
        currentClasses = MiscUtils.loadProperties( Config.getProperty( propertyName ) );
        Enumeration eKeys = currentClasses.propertyNames();
        while ( eKeys.hasMoreElements() )
        {
        	// PM: Warning suppressed: the type is uncertain.
            keys.add( eKeys.nextElement() );
        }
    }

    public void processAnalysis()
        throws WorkflowStepException, AnalysisControllerException
    {
		String returnFormat = ( messageObject.getValue( "Format" ) )[0];
        AnalysisManager analysisType = castElement( returnFormat );
        analysisType.execute( messageObject );
    }

    protected AnalysisManager castElement( String currentName )
        throws AnalysisControllerException
    {
        try
        {
			String className = getElementType( currentName );
            if ( className == null )
            {
                return null;
            }
            Class classDefinition = Class.forName( className );
            AnalysisManager analysisType = ( AnalysisManager ) classDefinition.newInstance();
            return analysisType;
        }
        catch ( InstantiationException wfe )
        {
            throw new AnalysisControllerException( "InstantiationException" + wfe.getMessage() );
        }
        catch ( ClassNotFoundException cne )
        {
            MraldOutFile.logToFile( Config.getProperty( "LOGFILE" ), "AnalysisControllerException in castElements: Class " + currentName + " not found." );
        }
        catch ( IllegalAccessException iae )
        {
            throw new AnalysisControllerException( "Illegal access exception: " + iae.getMessage() );
        }
        return null;
    }

    /**
     *  Returns the fully qualified class name for a given name.
     *
     *@param  	outputType			Name obtained from the MsgObject workingObjects HashTable. This should start with some value given in the buildables Properties object.
     *@return                                The fullly qualified class name
     *      associated with the submitted name.
     *@exception  OutputControllerException  Description of Exception
     */

    protected String getElementType( String classType )
		throws AnalysisControllerException
    {
        for ( int i = 0; i < keys.size(); i++ )
        {
            if ( classType.startsWith( ( String ) keys.get( i ) ) )
            {
                return ( String ) currentClasses.get( keys.get( i ) );
            }
        }
        throw new NullPointerException("An class type of " + classType + " is not recognized.  Please check the class type in the form and the appropriate properties file.");
    }
}

