/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.mrald.analysis;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import org.mitre.mrald.control.MsgObject;
import org.mitre.mrald.control.MsgObjectException;

public abstract class WekaAnalysis extends AnalysisManager
{
    public WekaAnalysis() {
		super();
	}

    protected void prepareHeaders( MsgObject msg ) throws AnalysisManagerException
    {
        try
        {
            msg.SetOutPrintWriter();
			msg.setContentType( "text/plain" );
			msg.setHeader( "Content-Disposition", "inline;" );
        }
        catch ( MsgObjectException mge )
        {
            AnalysisManagerException aee = new AnalysisManagerException( mge.getMessage() );
            throw aee;
        }
    }

    protected String makeTemplateFile() throws AnalysisManagerException
    {
		try
		{
			String fileLocation = msg.getUserId() + System.currentTimeMillis() + ".template";
			FileOutputStream fout = new FileOutputStream( new File( System.getProperty( "java.io.tmpdir" ) + "/" + fileLocation ) );
			AnalysisElement currentElement = null;
			String formattedString;

			formattedString = "@RELATION " + msg.getUserId() + "\n\n";
			fout.write( formattedString.getBytes() );

			for ( int k = 0; k < sortedElements.size(); k++ )
			{
				formattedString = "@ATTRIBUTE ";
				currentElement = sortedElements.get( k );

				if ( currentElement.getType().equals( "Timestamp" ) )
					formattedString += currentElement.getField() + " " + " DATE " + currentElement.getFormat() + "\n";

				else if ( currentElement.getType().equals( "Nominal" ) )
				{
					formattedString += currentElement.getField() + " {";
					String[] tempStrings = currentElement.getValues();
					for ( int i = 0; i < tempStrings.length; i++ )
					{
						formattedString += "\"" + tempStrings[ i ] + "\"";
						if ( i < ( tempStrings.length - 1 ) )
							formattedString += ", ";
					}
					formattedString += "}\n";
				}

				else if ( currentElement.getType().equals( "Numeric" ) )
					formattedString += currentElement.getField() + " NUMERIC\n";

				else if ( currentElement.getType().equals( "String" ) )
					formattedString += currentElement.getField() + " STRING\n";

				fout.write( formattedString.getBytes() );
			}

			formattedString = "\n@DATA\n";
			fout.write( formattedString.getBytes() );

			fout.close();
			return System.getProperty( "java.io.tmpdir" ) + "/" + fileLocation;
		}
		catch ( FileNotFoundException fnfe )
		{
			throw new AnalysisManagerException( fnfe );
		}
		catch ( IOException ioe )
		{
			throw new AnalysisManagerException( ioe );
		}
    }

    protected abstract void performAnalysis( String template_file, String data_file ) throws AnalysisManagerException;

    protected abstract void displayResults() throws AnalysisManagerException;
}

